import React, { useState } from 'react';
import AppLayout from '@/layouts/app-layout';
import { Head, router } from '@inertiajs/react';
import PlayerFormSteps from '@/components/players/player-form-steps';
import BasicInfoStep from '@/components/players/form-steps/basic-info-step';
import ContactSocialStep from '@/components/players/form-steps/contact-social-step';
import GamingProfileStep from '@/components/players/form-steps/gaming-profile-step';
import SettingsPrivacyStep from '@/components/players/form-steps/settings-privacy-step';
import { Users, Phone, Trophy, Shield } from 'lucide-react';

export default function AdminPlayerEdit({ player, tournaments, users }: any) {
  const [currentStep, setCurrentStep] = useState(0);
  const [isSubmitting, setIsSubmitting] = useState(false);
  
  const [form, setForm] = useState({
    display_name: player.display_name || '',
    bio: player.bio || '',
    country: player.country || '',
    city: player.city || '',
    date_of_birth: player.date_of_birth || '',
    gender: player.gender || '',
    phone: player.phone || '',
    discord_username: player.discord_username || '',
    steam_id: player.steam_id || '',
    twitch_username: player.twitch_username || '',
    youtube_channel: player.youtube_channel || '',
    twitter_handle: player.twitter_handle || '',
    instagram_handle: player.instagram_handle || '',
    website: player.website || '',
    player_type: player.player_type || '',
    experience_level: player.experience_level || '',
    primary_game: player.primary_game || '',
    status: player.status || '',
    verification_status: player.verification_status || '',
    tournament_id: player.tournament_id || '',
    user_id: player.user_id || '',
    global_rating: player.global_rating || '',
    tournament_rating: player.tournament_rating || '',
    seeding_position: player.seeding_position || '',
    is_public_profile: player.is_public_profile ?? true,
    allow_messages: player.allow_messages ?? true,
    show_contact_info: player.show_contact_info ?? false,
  });

  const [errors, setErrors] = useState<any>({});

  const steps = [
    {
      id: 'basic',
      title: 'Basic Information',
      description: 'Personal details and user association',
      icon: <Users className="w-5 h-5" />,
    },
    {
      id: 'contact',
      title: 'Contact & Social',
      description: 'Contact information and social media links',
      icon: <Phone className="w-5 h-5" />,
    },
    {
      id: 'gaming',
      title: 'Gaming Profile',
      description: 'Tournament and gaming-related information',
      icon: <Trophy className="w-5 h-5" />,
    },
    {
      id: 'settings',
      title: 'Settings & Privacy',
      description: 'Privacy settings and final review',
      icon: <Shield className="w-5 h-5" />,
    },
  ];

  const handleInputChange = (field: string, value: any) => {
    setForm((prev: any) => ({ ...prev, [field]: value }));
    if (errors[field]) {
      setErrors((prev: any) => ({ ...prev, [field]: null }));
    }
  };

  const validateCurrentStep = () => {
    const stepErrors: any = {};
    
    switch (currentStep) {
      case 0: // Basic Information
        if (!form.display_name) stepErrors.display_name = 'Display name is required';
        if (!form.user_id) stepErrors.user_id = 'User selection is required';
        break;
      case 1: // Contact & Social - No required fields
        break;
      case 2: // Gaming Profile
        if (!form.tournament_id) stepErrors.tournament_id = 'Tournament selection is required';
        if (!form.player_type) stepErrors.player_type = 'Player type is required';
        if (!form.experience_level) stepErrors.experience_level = 'Experience level is required';
        if (!form.status) stepErrors.status = 'Status is required';
        if (!form.verification_status) stepErrors.verification_status = 'Verification status is required';
        break;
      case 3: // Settings & Privacy - No required fields
        break;
    }

    setErrors(stepErrors);
    return Object.keys(stepErrors).length === 0;
  };

  const handleNext = () => {
    if (validateCurrentStep()) {
      setCurrentStep(prev => Math.min(prev + 1, steps.length - 1));
    }
  };

  const handlePrevious = () => {
    setCurrentStep(prev => Math.max(prev - 1, 0));
  };

  const handleStepChange = (step: number) => {
    // Allow going to any step in edit mode
    setCurrentStep(step);
  };

  const handleSubmit = () => {
    if (!validateCurrentStep()) return;

    setIsSubmitting(true);
    router.put(route('admin.players.update', { player: player.id }), form, {
      onError: (errors) => {
        setErrors(errors);
        setIsSubmitting(false);
        // Go back to the step with errors
        if (errors.display_name || errors.user_id) setCurrentStep(0);
        else if (errors.tournament_id || errors.player_type || errors.experience_level || errors.status || errors.verification_status) setCurrentStep(2);
      },
      onSuccess: () => {
        router.get(route('admin.players.index'));
      },
      onFinish: () => {
        setIsSubmitting(false);
      }
    });
  };

  const renderCurrentStep = () => {
    switch (currentStep) {
      case 0:
        return (
          <BasicInfoStep
            form={form}
            errors={errors}
            onInputChange={handleInputChange}
            users={users}
          />
        );
      case 1:
        return (
          <ContactSocialStep
            form={form}
            errors={errors}
            onInputChange={handleInputChange}
          />
        );
      case 2:
        return (
          <GamingProfileStep
            form={form}
            errors={errors}
            onInputChange={handleInputChange}
            tournaments={tournaments}
          />
        );
      case 3:
        return (
          <SettingsPrivacyStep
            form={form}
            errors={errors}
            onInputChange={handleInputChange}
          />
        );
      default:
        return null;
    }
  };

  return (
    <AppLayout breadcrumbs={[
      { title: 'Players', href: '/admin/players' },
      { title: player.display_name, href: `/admin/players/${player.id}` },
      { title: 'Edit', href: `/admin/players/${player.id}/edit` }
    ]}>
      <Head title={`Edit Player: ${player.display_name}`} />
      
      <div className="flex flex-col gap-6 p-4 mx-auto">
        <div className="flex justify-between items-center">
            
          <div className="flex gap-2">
            <button
              onClick={() => router.get(route('admin.players.index'))}
              className="px-4 py-2 text-sm border rounded-md hover:bg-muted transition-colors"
            >
              Cancel
            </button>
          </div>
        </div>

        <PlayerFormSteps
          steps={steps}
          currentStep={currentStep}
          onStepChange={handleStepChange}
          onNext={handleNext}
          onPrevious={handlePrevious}
          onSubmit={handleSubmit}
          isSubmitting={isSubmitting}
          canProceed={true}
          isLastStep={currentStep === steps.length - 1}
        >
          {renderCurrentStep()}
        </PlayerFormSteps>
      </div>
    </AppLayout>
  );
} 