import React, { useState } from 'react';
import { Head, router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { 
  ArrowLeft, 
  Download, 
  Printer, 
  QrCode, 
  Shield, 
  User, 
  Trophy, 
  Calendar,
  MapPin,
  Phone,
  Mail,
  Globe,
  Star,
  Award,
  Eye,
  EyeOff
} from 'lucide-react';

// Add print styles
const printStyles = `
  @media print {
    body { background: white !important; }
    .no-print { display: none !important; }
    .page { 
      width: 297mm !important;
      height: 210mm !important;
      margin: 0 !important;
      padding: 0 !important;
      box-shadow: none !important;
      page-break-after: always !important;
    }
    .page:last-child { page-break-after: avoid !important; }
    .digital-id-card {
      width: 100% !important;
      height: 100% !important;
      margin: 0 !important;
      padding: 20px !important;
      border: 2px solid #000 !important;
      background: white !important;
    }
    .front-side, .back-side {
      width: 100% !important;
      height: 100% !important;
      margin: 0 !important;
      padding: 20px !important;
    }
  }
`;

interface Player {
  id: number;
  digital_id: string;
  display_name: string;
  bio?: string;
  country?: string;
  city?: string;
  date_of_birth?: string;
  gender?: string;
  phone?: string;
  avatar?: string;
  player_type: string;
  experience_level: string;
  global_rating?: number;
  tournament_rating?: number;
  status: string;
  verification_status: string;
  registered_at: string;
  user: {
    id: number;
    name: string;
    email: string;
  };
  tournament: {
    id: number;
    name: string;
  };
}

interface Props {
  player: Player;
  digital_id: string;
  generated_at: string;
  valid_until: string;
  qr_code: string;
  // Additional data for ID card template
  player_name: string;
  player_email: string;
  player_phone: string;
  player_type: string;
  tournament_edition: string;
  verification_status: string;
  registration_status: string;
  player_id: number;
  tournament_year: number;
  barcode_text: string;
  generated_date: string;
  valid_until_date: string;
  tournament: string;
  registration: string;
  id: number;
  validity_period: string;
}

const PlayerDigitalId: React.FC<Props> = ({ 
  player, 
  digital_id, 
  generated_at, 
  valid_until, 
  qr_code,
  // Additional props for ID card
  player_name,
  player_email,
  player_phone,
  player_type,
  tournament_edition,
  verification_status,
  registration_status,
  player_id,
  tournament_year,
  barcode_text,
  generated_date,
  valid_until_date,
  tournament,
  registration,
  id,
  validity_period
}) => {
  const [activeTab, setActiveTab] = useState('front');
  const [showQR, setShowQR] = useState(true);

  const handlePrint = () => {
    window.print();
  };

  const handleDownloadPDF = () => {
    window.open(route('admin.players.digital-id.print', player.id), '_blank');
  };

  const getStatusColor = (status: string) => {
    const colors = {
      'active': 'bg-green-100 text-green-800',
      'registered': 'bg-blue-100 text-blue-800',
      'eliminated': 'bg-red-100 text-red-800',
      'withdrawn': 'bg-gray-100 text-gray-800',
      'suspended': 'bg-yellow-100 text-yellow-800',
      'banned': 'bg-red-100 text-red-800',
    };
    return colors[status as keyof typeof colors] || 'bg-gray-100 text-gray-800';
  };

  const getVerificationColor = (status: string) => {
    const colors = {
      'verified': 'bg-green-100 text-green-800',
      'pending': 'bg-yellow-100 text-yellow-800',
      'unverified': 'bg-gray-100 text-gray-800',
      'rejected': 'bg-red-100 text-red-800',
    };
    return colors[status as keyof typeof colors] || 'bg-gray-100 text-gray-800';
  };

  const getPlayerTypeColor = (type: string) => {
    const colors = {
      'amateur': 'bg-blue-100 text-blue-800',
      'semi_pro': 'bg-purple-100 text-purple-800',
      'professional': 'bg-orange-100 text-orange-800',
      'legend': 'bg-red-100 text-red-800',
    };
    return colors[type as keyof typeof colors] || 'bg-gray-100 text-gray-800';
  };

  return (
    <AppLayout>
      <Head title={`Digital ID - ${player.display_name}`}>
        <style>{printStyles}</style>
      </Head>
      
      <div className="p-6">
        {/* Header */}
        <div className="flex items-center justify-between mb-6 no-print">
          <div className="flex items-center space-x-4">
            <Button variant="outline" size="sm" asChild>
              <a href={route('admin.players.show', player.id)}>
                <ArrowLeft className="h-4 w-4 mr-2" />
                Back to Player
              </a>
            </Button>
         
          </div>
          <div className="flex space-x-2">
            <Button variant="outline" size="sm" onClick={handlePrint}>
              <Printer className="h-4 w-4 mr-2" />
              Print
            </Button>
            <Button variant="outline" size="sm" onClick={handleDownloadPDF}>
              <Download className="h-4 w-4 mr-2" />
              Download PDF
            </Button>
          </div>
        </div>

        {/* Digital ID Card */}
        <div className="max-w-4xl mx-auto">
          <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-6">
            <TabsList className="grid w-full grid-cols-2 no-print">
              <TabsTrigger value="front" className="flex items-center gap-2">
                <Eye className="w-4 h-4" />
                Front Side - {digital_id}
              </TabsTrigger>
              <TabsTrigger value="back" className="flex items-center gap-2">
                <EyeOff className="w-4 h-4" />
                Back Side - {digital_id}
              </TabsTrigger>
            </TabsList>

            <TabsContent value="front">
              <Card className="overflow-hidden digital-id-card">
                <div className="bg-gradient-to-br from-primary to-primary/80 text-primary-foreground p-6">
                  <div className="flex justify-between items-start">
                    <div>
                      <h2 className="text-2xl font-bold mb-2">PLAYER DIGITAL ID</h2>
                      <p className="text-primary-foreground/80">Tournament Management System</p>
                    </div>
                    <div className="text-right">
                      <div className="text-sm text-primary-foreground/70">Digital ID</div>
                      <div className="font-mono text-lg font-bold">{digital_id}</div>
                    </div>
                  </div>
                </div>

                <CardContent className="p-6">
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                    {/* Player Photo */}
                    <div className="md:col-span-1">
                      <div className="flex flex-col items-center space-y-4">
                        <div className="w-32 h-32 rounded-full overflow-hidden bg-muted border-4 border-primary">
                          {player.avatar ? (
                            <img 
                              src={player.avatar} 
                              alt={player.display_name}
                              className="w-full h-full object-cover"
                            />
                          ) : (
                            <div className="w-full h-full bg-gradient-to-br from-primary to-primary/80 flex items-center justify-center">
                              <User className="w-16 h-16 text-primary-foreground" />
                            </div>
                          )}
                        </div>
                        <div className="text-center">
                          <h3 className="text-xl font-bold">{player.display_name}</h3>
                          <p className="text-muted-foreground">{player.user.name}</p>
                        </div>
                      </div>
                    </div>

                    {/* Player Details */}
                    <div className="md:col-span-2 space-y-4">
                      <div className="grid grid-cols-2 gap-4">
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Email</label>
                          <p className="text-sm">{player.user.email}</p>
                        </div>
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Phone</label>
                          <p className="text-sm">{player.phone || 'Not provided'}</p>
                        </div>
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Country</label>
                          <p className="text-sm">{player.country || 'Not specified'}</p>
                        </div>
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">City</label>
                          <p className="text-sm">{player.city || 'Not specified'}</p>
                        </div>
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Date of Birth</label>
                          <p className="text-sm">{player.date_of_birth ? new Date(player.date_of_birth).toLocaleDateString() : 'Not provided'}</p>
                        </div>
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Gender</label>
                          <p className="text-sm">{player.gender ? player.gender.charAt(0).toUpperCase() + player.gender.slice(1) : 'Not specified'}</p>
                        </div>
                      </div>

                      <div className="grid grid-cols-2 gap-4">
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Player Type</label>
                          <Badge className={getPlayerTypeColor(player.player_type)}>
                            {player.player_type.replace('_', ' ').toUpperCase()}
                          </Badge>
                        </div>
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Experience Level</label>
                          <Badge variant="outline">
                            {player.experience_level.charAt(0).toUpperCase() + player.experience_level.slice(1)}
                          </Badge>
                        </div>
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Status</label>
                          <Badge className={getStatusColor(player.status)}>
                            {player.status.charAt(0).toUpperCase() + player.status.slice(1)}
                          </Badge>
                        </div>
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Verification</label>
                          <Badge className={getVerificationColor(player.verification_status)}>
                            {player.verification_status.charAt(0).toUpperCase() + player.verification_status.slice(1)}
                          </Badge>
                        </div>
                      </div>

                      <div className="grid grid-cols-2 gap-4">
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Global Rating</label>
                          <p className="text-sm font-bold">{player.global_rating || 'N/A'}</p>
                        </div>
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Tournament Rating</label>
                          <p className="text-sm font-bold">{player.tournament_rating || 'N/A'}</p>
                        </div>
                      </div>
                    </div>
                  </div>

                  {/* QR Code */}
                  {showQR && (
                    <div className="mt-6 flex justify-center">
                      <div className="text-center">
                        <img src={qr_code} alt="QR Code" className="mx-auto" />
                        <p className="text-xs text-muted-foreground mt-2">Scan to verify Digital ID</p>
                      </div>
                    </div>
                  )}

                  {/* Footer */}
                  <div className="mt-6 pt-4 border-t border-border">
                    <div className="flex justify-between items-center text-sm text-muted-foreground">
                      <div>
                        <span className="font-medium">Generated:</span> {generated_at}
                      </div>
                      <div>
                        <span className="font-medium">Valid Until:</span> {valid_until}
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="back">
              <Card className="overflow-hidden digital-id-card">
                <div className="bg-gradient-to-br from-secondary to-secondary/80 text-secondary-foreground p-6">
                  <div className="flex justify-between items-start">
                    <div>
                      <h2 className="text-2xl font-bold mb-2">SECURITY FEATURES</h2>
                      <p className="text-secondary-foreground/80">Official Tournament Management System</p>
                    </div>
                    <div className="text-right">
                      <div className="text-sm text-secondary-foreground/70">Digital ID</div>
                      <div className="font-mono text-lg font-bold">{digital_id}</div>
                    </div>
                  </div>
                </div>

                <CardContent className="p-6">
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    {/* Security Features */}
                    <div className="space-y-4">
                      <h3 className="text-lg font-semibold flex items-center gap-2">
                        <Shield className="w-5 h-5 text-primary" />
                        Security Features
                      </h3>
                      <div className="space-y-3">
                        <div className="flex items-center gap-3">
                          <div className="w-2 h-2 bg-green-500 rounded-full"></div>
                          <span className="text-sm">Unique Digital ID</span>
                        </div>
                        <div className="flex items-center gap-3">
                          <div className="w-2 h-2 bg-green-500 rounded-full"></div>
                          <span className="text-sm">QR Code Verification</span>
                        </div>
                        <div className="flex items-center gap-3">
                          <div className="w-2 h-2 bg-green-500 rounded-full"></div>
                          <span className="text-sm">Encrypted Data</span>
                        </div>
                        <div className="flex items-center gap-3">
                          <div className="w-2 h-2 bg-green-500 rounded-full"></div>
                          <span className="text-sm">Real-time Validation</span>
                        </div>
                      </div>
                    </div>

                    {/* Tournament Information */}
                    <div className="space-y-4">
                      <h3 className="text-lg font-semibold flex items-center gap-2">
                        <Trophy className="w-5 h-5 text-yellow-600" />
                        Tournament Info
                      </h3>
                      <div className="space-y-3">
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Tournament</label>
                          <p className="text-sm font-medium">{player.tournament.name}</p>
                        </div>
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Registration Date</label>
                          <p className="text-sm">{new Date(player.registered_at).toLocaleDateString()}</p>
                        </div>
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Player ID</label>
                          <p className="text-sm font-mono">{player.id}</p>
                        </div>
                      </div>
                    </div>
                  </div>

                  {/* Additional Information */}
                  <div className="mt-6 pt-4 border-t border-border">
                    <h3 className="text-lg font-semibold mb-3">Additional Information</h3>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                      <div>
                        <label className="text-sm font-medium text-muted-foreground">Bio</label>
                        <p className="text-sm">{player.bio || 'No bio provided'}</p>
                      </div>
                      <div>
                        <label className="text-sm font-medium text-muted-foreground">Digital ID Status</label>
                        <Badge className="bg-green-100 text-green-800">
                          ACTIVE
                        </Badge>
                      </div>
                    </div>
                  </div>

                  {/* Footer */}
                  <div className="mt-6 pt-4 border-t border-border">
                    <div className="flex justify-between items-center text-sm text-muted-foreground">
                      <div>
                        <span className="font-medium">Generated:</span> {generated_at}
                      </div>
                      <div>
                        <span className="font-medium">Valid Until:</span> {valid_until}
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </TabsContent>
          </Tabs>
        </div>
      </div>
    </AppLayout>
  );
};

export default PlayerDigitalId; 