import React from 'react';
import { Head, router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Card, CardContent } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { 
  ArrowLeft, 
  Download, 
  Printer, 
  QrCode
} from 'lucide-react';

interface Player {
  id: number;
  display_name: string;
  bio?: string;
  country?: string;
  city?: string;
  avatar?: string;
  player_type: string;
  experience_level: string;
  global_rating?: number;
  tournament_rating?: number;
  status: string;
  verification_status: string;
  registered_at: string;
  user: {
    id: number;
    name: string;
    email: string;
  };
  tournament: {
    id: number;
    name: string;
  };
}

interface Props {
  player: Player;
  certificate_id: string;
  generated_at: string;
  valid_until: string;
  qr_code: string;
}

const PlayerCertificate: React.FC<Props> = ({ 
  player, 
  certificate_id, 
  generated_at, 
  valid_until, 
  qr_code 
}) => {
  const handlePrint = () => {
    window.print();
  };

  const handleDownloadPDF = () => {
    window.open(route('admin.players.certificate.print', player.id), '_blank');
  };

  return (
    <AppLayout>
      <Head title={`Certificate - ${player.display_name}`}>
        <style>{`
          @media print {
            body { background: white !important; }
            .no-print { display: none !important; }
            .certificate-container {
              width: 100% !important;
              height: 100% !important;
              margin: 0 !important;
              padding: 20px !important;
              border: 2px solid #000 !important;
              background: white !important;
            }
          }
        `}</style>
      </Head>
      
      <div className="p-6">
        {/* Header */}
        <div className="flex items-center justify-between mb-6 no-print">
          <div className="flex items-center space-x-4">
            <Button variant="outline" size="sm" asChild>
              <a href={route('admin.players.show', player.id)}>
                <ArrowLeft className="h-4 w-4 mr-2" />
                Back to Player
              </a>
            </Button>
           
          </div>
          <div className="flex space-x-2">
            <Button variant="outline" size="sm" onClick={handlePrint}>
              <Printer className="h-4 w-4 mr-2" />
              Print
            </Button>
            <Button variant="outline" size="sm" onClick={handleDownloadPDF}>
              <Download className="h-4 w-4 mr-2" />
              Download PDF
            </Button>
          </div>
        </div>

        {/* Certificate */}
        <div className="max-w-4xl mx-auto">
          <Card className="overflow-hidden certificate-container">
            {/* Certificate Header */}
            <div className="text-center p-8 bg-gradient-to-br from-amber-50 to-yellow-50 border-b-2 border-amber-200">
              <div className="w-16 h-16 mx-auto mb-4 bg-gradient-to-br from-amber-400 to-yellow-500 rounded-full flex items-center justify-center text-white font-bold text-lg shadow-lg">
                MTC
              </div>
              <div className="text-amber-800 font-bold text-xl mb-2">
                {player.tournament.name}
              </div>
            </div>

            <CardContent className="p-8 text-center">
              {/* Main Content */}
              <div className="mb-8">
                <h1 className="text-4xl font-bold text-amber-600 mb-2">CERTIFICATE</h1>
                <p className="text-lg text-gray-600 mb-8">of Participation</p>
                
                <p className="text-sm text-gray-500 uppercase tracking-wider mb-4">
                  This is to certify that
                </p>
                
                <h2 className="text-3xl font-bold text-gray-800 mb-4 pb-4 border-b-2 border-amber-400 inline-block min-w-[300px]">
                  {player.display_name}
                </h2>
                
                <p className="text-lg text-gray-600 italic mb-8">
                  ({player.user.name})
                </p>
                
                <div className="max-w-2xl mx-auto">
                  <p className="text-base text-gray-700 leading-relaxed">
                    has successfully participated in the {player.tournament.name} tournament
                    and has demonstrated exceptional sportsmanship, skill, and dedication throughout the competition.
                  </p>
                </div>
              </div>

              {/* QR Code */}
              <div className="flex justify-center mb-8">
                <div className="text-center">
                  <div className="bg-white p-3 rounded-lg shadow-md inline-block">
                    <img src={qr_code} alt="QR Code" className="w-16 h-16" />
                  </div>
                  <p className="text-xs text-gray-500 mt-2">Scan to verify certificate authenticity</p>
                </div>
              </div>
            </CardContent>

            {/* Footer */}
            <div className="border-t border-gray-200 p-6">
              <div className="text-right text-sm text-gray-500">
                <div className="font-semibold text-gray-700 mb-1">Certificate ID: {certificate_id}</div>
                <div>Generated on {generated_at}</div>
              </div>
            </div>
          </Card>
        </div>
      </div>
    </AppLayout>
  );
};

export default PlayerCertificate; 