import React from 'react';
import { Head, Link, router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { 
  ArrowLeft,
  Edit,
  Trash2,
  Key,
  Shield,
  Users,
  Calendar,
  Hash,
  Tag,
  FileText,
  Settings,
  AlertTriangle,
  CheckCircle,
  XCircle
} from 'lucide-react';
import { Alert, AlertDescription } from '@/components/ui/alert';

interface Permission {
  id: number;
  name: string;
  display_name: string;
  description: string;
  category: string;
  is_system: boolean;
  is_active: boolean;
  sort_order: number;
  created_at: string;
  updated_at: string;
  category_icon: string;
  category_color: string;
  roles?: Array<{
    id: number;
    name: string;
    display_name: string;
    color: string;
  }>;
}

interface Props {
  permission: Permission;
  stats: {
    roles_count: number;
    users_count: number;
  };
}

const PermissionShow: React.FC<Props> = ({ permission, stats }) => {
  const getCategoryIcon = (category: string) => {
    const iconMap: Record<string, any> = {
      dashboard: Shield,
      tournaments: Key,
      players: Users,
      teams: Users,
      matches: Key,
      brackets: Key,
      users: Users,
      roles: Shield,
      permissions: Key,
      payments: Key,
      settings: Settings,
      media: Key,
      analytics: Key,
      system: Settings,
    };
    const IconComponent = iconMap[category] || Key;
    return <IconComponent className="h-4 w-4" />;
  };

  const handleDelete = () => {
    if (permission.is_system) {
      alert('Cannot delete system permissions');
      return;
    }

    if (confirm('Are you sure you want to delete this permission? This action cannot be undone.')) {
      router.delete(route('admin.permissions.destroy', permission.id), {
        onSuccess: () => {
          router.visit(route('admin.permissions.index'));
        }
      });
    }
  };

  const toggleStatus = () => {
    router.put(route('admin.permissions.update', permission.id), {
      ...permission,
      is_active: !permission.is_active,
    });
  };

  return (
    <AppLayout breadcrumbs={[
      { title: 'Permissions', href: '/admin/permissions' },
      { title: permission.display_name, href: `/admin/permissions/${permission.id}` }
    ]}>
      <Head title={permission.display_name} />
      
      <div className="space-y-6 p-4 flex flex-col gap-4">
        {/* Header */}
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-4">
            <Button variant="outline" size="sm" asChild>
              <Link href={route('admin.permissions.index')}>
                <ArrowLeft className="h-4 w-4 mr-2" />
                Back to Permissions
              </Link>
            </Button>
            <div>
              <h1 className="text-3xl font-bold">{permission.display_name}</h1>
              <p className="text-muted-foreground font-mono">{permission.name}</p>
            </div>
          </div>
          <div className="flex items-center gap-2">
            <Badge variant={permission.is_active ? 'default' : 'secondary'}>
              {permission.is_active ? (
                <>
                  <CheckCircle className="h-3 w-3 mr-1" />
                  Active
                </>
              ) : (
                <>
                  <XCircle className="h-3 w-3 mr-1" />
                  Inactive
                </>
              )}
            </Badge>
            <Badge variant={permission.is_system ? 'destructive' : 'outline'}>
              {permission.is_system ? (
                <>
                  <Shield className="h-3 w-3 mr-1" />
                  System
                </>
              ) : (
                <>
                  <Tag className="h-3 w-3 mr-1" />
                  Custom
                </>
              )}
            </Badge>
          </div>
        </div>

        {/* System Permission Warning */}
        {permission.is_system && (
          <Alert>
            <AlertTriangle className="h-4 w-4" />
            <AlertDescription>
              This is a core system permission. Modifying or deactivating it may affect application functionality.
            </AlertDescription>
          </Alert>
        )}

        <div className="grid gap-6 md:grid-cols-3">
          {/* Permission Details */}
          <Card className="md:col-span-2">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Key className="h-5 w-5" />
                Permission Details
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="grid gap-4 md:grid-cols-2">
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Display Name</label>
                  <div className="flex items-center gap-2 mt-1">
                    <FileText className="h-4 w-4 text-muted-foreground" />
                    <span className="font-medium">{permission.display_name}</span>
                  </div>
                </div>
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Permission Name</label>
                  <div className="flex items-center gap-2 mt-1">
                    <Hash className="h-4 w-4 text-muted-foreground" />
                    <span className="font-mono text-sm bg-muted px-2 py-1 rounded">
                      {permission.name}
                    </span>
                  </div>
                </div>
              </div>

              <div>
                <label className="text-sm font-medium text-muted-foreground">Category</label>
                <div className="flex items-center gap-2 mt-1">
                  {getCategoryIcon(permission.category)}
                  <span className="capitalize font-medium">{permission.category}</span>
                </div>
              </div>

              {permission.description && (
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Description</label>
                  <div className="mt-1 p-3 bg-muted rounded-lg">
                    <p className="text-sm">{permission.description}</p>
                  </div>
                </div>
              )}

              <div className="grid gap-4 md:grid-cols-2">
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Sort Order</label>
                  <div className="mt-1">
                    <span className="font-medium">{permission.sort_order}</span>
                  </div>
                </div>
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Status</label>
                  <div className="mt-1">
                    <Badge variant={permission.is_active ? 'default' : 'secondary'}>
                      {permission.is_active ? 'Active' : 'Inactive'}
                    </Badge>
                  </div>
                </div>
              </div>

              <div className="grid gap-4 md:grid-cols-2">
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Created At</label>
                  <div className="flex items-center gap-2 mt-1">
                    <Calendar className="h-4 w-4 text-muted-foreground" />
                    <span className="text-sm">
                      {new Date(permission.created_at).toLocaleDateString('en-US', {
                        year: 'numeric',
                        month: 'long',
                        day: 'numeric',
                        hour: '2-digit',
                        minute: '2-digit'
                      })}
                    </span>
                  </div>
                </div>
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Last Updated</label>
                  <div className="flex items-center gap-2 mt-1">
                    <Calendar className="h-4 w-4 text-muted-foreground" />
                    <span className="text-sm">
                      {new Date(permission.updated_at).toLocaleDateString('en-US', {
                        year: 'numeric',
                        month: 'long',
                        day: 'numeric',
                        hour: '2-digit',
                        minute: '2-digit'
                      })}
                    </span>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Actions & Stats Sidebar */}
          <div className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle>Actions</CardTitle>
                <CardDescription>
                  Manage this permission
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-3">
                <Button className="w-full" asChild>
                  <Link href={route('admin.permissions.edit', permission.id)}>
                    <Edit className="h-4 w-4 mr-2" />
                    Edit Permission
                  </Link>
                </Button>

                <Button 
                  variant="outline" 
                  className="w-full"
                  onClick={toggleStatus}
                >
                  {permission.is_active ? (
                    <>
                      <XCircle className="h-4 w-4 mr-2" />
                      Deactivate
                    </>
                  ) : (
                    <>
                      <CheckCircle className="h-4 w-4 mr-2" />
                      Activate
                    </>
                  )}
                </Button>

                {!permission.is_system && (
                  <Button 
                    variant="destructive" 
                    className="w-full"
                    onClick={handleDelete}
                  >
                    <Trash2 className="h-4 w-4 mr-2" />
                    Delete Permission
                  </Button>
                )}
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Usage Statistics</CardTitle>
                <CardDescription>
                  How this permission is being used
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-center justify-between">
                  <div className="flex items-center gap-2">
                    <Shield className="h-4 w-4 text-blue-600" />
                    <span className="text-sm">Assigned to Roles</span>
                  </div>
                  <Badge variant="secondary">{stats.roles_count}</Badge>
                </div>

                <div className="flex items-center justify-between">
                  <div className="flex items-center gap-2">
                    <Users className="h-4 w-4 text-green-600" />
                    <span className="text-sm">Total Users</span>
                  </div>
                  <Badge variant="secondary">{stats.users_count}</Badge>
                </div>
              </CardContent>
            </Card>

            {/* Assigned Roles */}
            {permission.roles && permission.roles.length > 0 && (
              <Card>
                <CardHeader>
                  <CardTitle>Assigned Roles</CardTitle>
                  <CardDescription>
                    Roles that have this permission
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="space-y-2">
                    {permission.roles.map((role) => (
                      <div key={role.id} className="flex items-center justify-between p-2 border rounded-lg">
                        <div className="flex items-center gap-2">
                          <Shield className="h-4 w-4 text-muted-foreground" />
                          <Link 
                            href={route('admin.roles.show', role.id)}
                            className="text-sm font-medium hover:underline"
                          >
                            {role.display_name}
                          </Link>
                        </div>
                        <Badge 
                          variant="outline" 
                          className={role.color || 'bg-gray-100'}
                        >
                          {role.name}
                        </Badge>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            )}

            <Card>
              <CardHeader>
                <CardTitle>Permission Info</CardTitle>
              </CardHeader>
              <CardContent className="text-sm space-y-2">
                <div>
                  <strong>ID:</strong>
                  <p className="text-muted-foreground font-mono">#{permission.id}</p>
                </div>
                <div>
                  <strong>Resource:</strong>
                  <p className="text-muted-foreground capitalize">
                    {permission.name.split('.')[0]}
                  </p>
                </div>
                <div>
                  <strong>Action:</strong>
                  <p className="text-muted-foreground capitalize">
                    {permission.name.split('.')[1] || 'N/A'}
                  </p>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </AppLayout>
  );
};

export default PermissionShow;