import React, { useState } from 'react';
import { Head, Link, router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Checkbox } from '@/components/ui/checkbox';
import { 
  Key, 
  Shield, 
  Users, 
  Settings,
  Search,
  Filter,
  Eye,
  Edit,
  Trash2,
  Plus,
  MoreHorizontal,
  CheckCircle,
  XCircle,
  Archive
} from 'lucide-react';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';

interface Permission {
  id: number;
  name: string;
  display_name: string;
  description: string;
  category: string;
  is_system: boolean;
  is_active: boolean;
  sort_order: number;
  created_at: string;
  category_icon: string;
  category_color: string;
}

interface Props {
  permissions: {
    data: Permission[];
    current_page: number;
    last_page: number;
    per_page: number;
    total: number;
  };
  stats: {
    total: number;
    active: number;
    inactive: number;
    system: number;
    custom: number;
    by_category: Array<{
      category: string;
      count: number;
    }>;
  };
  categories: Record<string, string>;
  filters: {
    search: string;
    category: string;
    status: string;
    sort_by: string;
    sort_order: string;
  };
}

const PermissionsIndex: React.FC<Props> = ({ permissions, stats, categories, filters }) => {
  const [searchTerm, setSearchTerm] = useState(filters.search || '');
  const [selectedPermissions, setSelectedPermissions] = useState<number[]>([]);

  const getCategoryIcon = (category: string) => {
    const iconMap: Record<string, any> = {
      dashboard: Shield,
      tournaments: Key,
      players: Users,
      teams: Users,
      matches: Key,
      brackets: Key,
      users: Users,
      roles: Shield,
      permissions: Key,
      payments: Key,
      settings: Settings,
      media: Key,
      analytics: Key,
      system: Settings,
    };
    const IconComponent = iconMap[category] || Key;
    return <IconComponent className="h-4 w-4" />;
  };

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    router.get(route('admin.permissions.index'), {
      ...filters,
      search: searchTerm,
    });
  };

  const handleFilter = (key: string, value: string) => {
    router.get(route('admin.permissions.index'), {
      ...filters,
      [key]: value,
    });
  };

  const handleSort = (column: string) => {
    const newOrder = filters.sort_by === column && filters.sort_order === 'asc' ? 'desc' : 'asc';
    router.get(route('admin.permissions.index'), {
      ...filters,
      sort_by: column,
      sort_order: newOrder,
    });
  };

  const handleSelectAll = (checked: boolean) => {
    if (checked) {
      setSelectedPermissions(permissions.data.map(p => p.id));
    } else {
      setSelectedPermissions([]);
    }
  };

  const handleSelectPermission = (permissionId: number, checked: boolean) => {
    if (checked) {
      setSelectedPermissions([...selectedPermissions, permissionId]);
    } else {
      setSelectedPermissions(selectedPermissions.filter(id => id !== permissionId));
    }
  };

  const handleBulkAction = (action: string, value?: string) => {
    if (selectedPermissions.length === 0) return;

    router.post(route('admin.permissions.bulk-update'), {
      ids: selectedPermissions,
      action,
      value,
    }, {
      onSuccess: () => {
        setSelectedPermissions([]);
      }
    });
  };

  const handleDelete = (permissionId: number) => {
    if (confirm('Are you sure you want to delete this permission?')) {
      router.delete(route('admin.permissions.destroy', permissionId));
    }
  };

  const statCards = [
    {
      title: 'Total Permissions',
      value: stats.total,
      icon: Key,
      description: 'All permissions',
      color: 'text-blue-600',
    },
    {
      title: 'Active Permissions',
      value: stats.active,
      icon: CheckCircle,
      description: 'Currently active',
      color: 'text-green-600',
    },
    {
      title: 'System Permissions',
      value: stats.system,
      icon: Shield,
      description: 'Core system permissions',
      color: 'text-red-600',
    },
    {
      title: 'Custom Permissions',
      value: stats.custom,
      icon: Settings,
      description: 'User-defined permissions',
      color: 'text-purple-600',
    },
  ];

  return (
    <AppLayout breadcrumbs={[{ title: 'Permissions', href: '/admin/permissions' }]}>
      <Head title="Permission Management" />
      
      <div className="space-y-6 p-4 flex flex-col gap-4">
        {/* Header */}
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-3xl font-bold">Permissions</h1>
          </div>
         
        </div>

        {/* Statistics Grid */}
        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
          {statCards.map((stat) => {
            const IconComponent = stat.icon;
            return (
              <Card key={stat.title}>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">
                    {stat.title}
                  </CardTitle>
                  <IconComponent className={`h-4 w-4 ${stat.color}`} />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{stat.value}</div>
                  <p className="text-xs text-muted-foreground">
                    {stat.description}
                  </p>
                </CardContent>
              </Card>
            );
          })}
        </div>

        {/* Category Breakdown */}
        <Card>
          <CardHeader>
            <CardTitle>Permissions by Category</CardTitle>
            <CardDescription>Distribution of permissions across different categories</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
              {stats.by_category.map((category) => (
                <div key={category.category} className="flex items-center justify-between p-4 border rounded-lg">
                  <div className="flex items-center gap-3">
                    {getCategoryIcon(category.category)}
                    <div>
                      <p className="font-medium capitalize">{categories[category.category] || category.category}</p>
                      <p className="text-sm text-muted-foreground">{category.count} permissions</p>
                    </div>
                  </div>
                  <Badge variant="secondary">{category.count}</Badge>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>

        {/* Filters and Search */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Filter className="h-5 w-5" />
              Filters & Search
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="flex flex-col sm:flex-row gap-4">
              <form onSubmit={handleSearch} className="flex gap-2 flex-1">
                <div className="relative flex-1">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
                  <Input
                    placeholder="Search permissions..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="pl-10"
                  />
                </div>
                <Button type="submit">Search</Button>
              </form>

              <div className="flex gap-2">
                <Select
                  value={filters.category || 'all'}
                  onValueChange={(value) => handleFilter('category', value)}
                >
                  <SelectTrigger className="w-40">
                    <SelectValue placeholder="Category" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Categories</SelectItem>
                    {Object.entries(categories).map(([key, label]) => (
                      <SelectItem key={key} value={key}>{label}</SelectItem>
                    ))}
                  </SelectContent>
                </Select>

                <Select
                  value={filters.status || 'all'}
                  onValueChange={(value) => handleFilter('status', value)}
                >
                  <SelectTrigger className="w-32">
                    <SelectValue placeholder="Status" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Status</SelectItem>
                    <SelectItem value="active">Active</SelectItem>
                    <SelectItem value="inactive">Inactive</SelectItem>
                    <SelectItem value="system">System</SelectItem>
                    <SelectItem value="custom">Custom</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Bulk Actions */}
        {selectedPermissions.length > 0 && (
          <Card>
            <CardContent className="py-4">
              <div className="flex items-center justify-between">
                <span className="text-sm text-muted-foreground">
                  {selectedPermissions.length} permission(s) selected
                </span>
                <div className="flex gap-2">
                  <Button 
                    variant="outline" 
                    size="sm"
                    onClick={() => handleBulkAction('activate')}
                  >
                    <CheckCircle className="h-4 w-4 mr-2" />
                    Activate
                  </Button>
                  <Button 
                    variant="outline" 
                    size="sm"
                    onClick={() => handleBulkAction('deactivate')}
                  >
                    <XCircle className="h-4 w-4 mr-2" />
                    Deactivate
                  </Button>
                  <Button 
                    variant="destructive" 
                    size="sm"
                    onClick={() => {
                      if (confirm('Are you sure you want to delete selected permissions?')) {
                        router.post(route('admin.permissions.bulk-delete'), {
                          ids: selectedPermissions
                        });
                      }
                    }}
                  >
                    <Trash2 className="h-4 w-4 mr-2" />
                    Delete
                  </Button>
                </div>
              </div>
            </CardContent>
          </Card>
        )}

        {/* Permissions Table */}
        <Card>
          <CardHeader>
            <CardTitle>Permissions</CardTitle>
            <CardDescription>
              Showing {permissions.data.length} of {permissions.total} permissions
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="overflow-x-auto">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="w-12">
                      <Checkbox
                        checked={selectedPermissions.length === permissions.data.length}
                        onCheckedChange={handleSelectAll}
                      />
                    </TableHead>
                    <TableHead 
                      className="cursor-pointer hover:bg-muted/50"
                      onClick={() => handleSort('name')}
                    >
                      Permission
                    </TableHead>
                    <TableHead 
                      className="cursor-pointer hover:bg-muted/50"
                      onClick={() => handleSort('category')}
                    >
                      Category
                    </TableHead>
                    <TableHead>Description</TableHead>
                    <TableHead>Status</TableHead>
                    <TableHead>Type</TableHead>
                    <TableHead className="text-right">Actions</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {permissions.data.map((permission) => (
                    <TableRow key={permission.id}>
                      <TableCell>
                        <Checkbox
                          checked={selectedPermissions.includes(permission.id)}
                          onCheckedChange={(checked) => handleSelectPermission(permission.id, checked)}
                        />
                      </TableCell>
                      <TableCell>
                        <div>
                          <div className="font-medium">{permission.display_name}</div>
                          <div className="text-sm text-muted-foreground font-mono">
                            {permission.name}
                          </div>
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-2">
                          {getCategoryIcon(permission.category)}
                          <span className="capitalize">{categories[permission.category] || permission.category}</span>
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="max-w-xs truncate" title={permission.description}>
                          {permission.description}
                        </div>
                      </TableCell>
                      <TableCell>
                        <Badge variant={permission.is_active ? 'default' : 'secondary'}>
                          {permission.is_active ? 'Active' : 'Inactive'}
                        </Badge>
                      </TableCell>
                      <TableCell>
                        <Badge variant={permission.is_system ? 'destructive' : 'outline'}>
                          {permission.is_system ? 'System' : 'Custom'}
                        </Badge>
                      </TableCell>
                      <TableCell className="text-right">
                        <DropdownMenu>
                          <DropdownMenuTrigger asChild>
                            <Button variant="ghost" className="h-8 w-8 p-0">
                              <MoreHorizontal className="h-4 w-4" />
                            </Button>
                          </DropdownMenuTrigger>
                          <DropdownMenuContent align="end">
                            <DropdownMenuLabel>Actions</DropdownMenuLabel>
                            <DropdownMenuItem asChild>
                              <Link href={route('admin.permissions.show', permission.id)}>
                                <Eye className="h-4 w-4 mr-2" />
                                View
                              </Link>
                            </DropdownMenuItem>
                            <DropdownMenuItem asChild>
                              <Link href={route('admin.permissions.edit', permission.id)}>
                                <Edit className="h-4 w-4 mr-2" />
                                Edit
                              </Link>
                            </DropdownMenuItem>
                            <DropdownMenuSeparator />
                            {!permission.is_system && (
                              <DropdownMenuItem 
                                onClick={() => handleDelete(permission.id)}
                                className="text-red-600"
                              >
                                <Trash2 className="h-4 w-4 mr-2" />
                                Delete
                              </DropdownMenuItem>
                            )}
                          </DropdownMenuContent>
                        </DropdownMenu>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>

            {/* Pagination */}
            {permissions.last_page > 1 && (
              <div className="flex items-center justify-between mt-4">
                <div className="text-sm text-muted-foreground">
                  Showing {((permissions.current_page - 1) * permissions.per_page) + 1} to{' '}
                  {Math.min(permissions.current_page * permissions.per_page, permissions.total)} of{' '}
                  {permissions.total} permissions
                </div>
                <div className="flex gap-2">
                  {permissions.current_page > 1 && (
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => router.get(route('admin.permissions.index'), {
                        ...filters,
                        page: permissions.current_page - 1,
                      })}
                    >
                      Previous
                    </Button>
                  )}
                  {permissions.current_page < permissions.last_page && (
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => router.get(route('admin.permissions.index'), {
                        ...filters,
                        page: permissions.current_page + 1,
                      })}
                    >
                      Next
                    </Button>
                  )}
                </div>
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </AppLayout>
  );
};

export default PermissionsIndex;