import React from 'react';
import { Head, Link, useForm } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Switch } from '@/components/ui/switch';
import { ArrowLeft, Save, Key, AlertTriangle } from 'lucide-react';
import { Alert, AlertDescription } from '@/components/ui/alert';

interface Permission {
  id: number;
  name: string;
  display_name: string;
  description: string;
  category: string;
  is_system: boolean;
  is_active: boolean;
  sort_order: number;
  created_at: string;
  updated_at: string;
}

interface Props {
  permission: Permission;
  categories: Record<string, string>;
  actions: Record<string, string>;
}

const EditPermission: React.FC<Props> = ({ permission, categories, actions }) => {
  const { data, setData, put, processing, errors } = useForm({
    name: permission.name,
    display_name: permission.display_name,
    description: permission.description || '',
    category: permission.category,
    is_system: permission.is_system,
    is_active: permission.is_active,
    sort_order: permission.sort_order,
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    put(route('admin.permissions.update', permission.id), {
      onSuccess: () => {
        // Redirect will be handled by the controller
      }
    });
  };

  const generateName = () => {
    const category = data.category;
    const displayName = data.display_name.toLowerCase();
    
    // Try to extract action from display name
    let action = '';
    Object.keys(actions).forEach(key => {
      if (displayName.includes(key)) {
        action = key;
      }
    });
    
    if (category && action) {
      setData('name', `${category}.${action}`);
    } else if (category && displayName) {
      // Fallback: use simplified display name
      const simplifiedName = displayName
        .replace(/[^a-z0-9\s]/gi, '')
        .replace(/\s+/g, '_');
      setData('name', `${category}.${simplifiedName}`);
    }
  };

  const handleCategoryChange = (category: string) => {
    setData('category', category);
    // Auto-generate name if display name exists
    if (data.display_name) {
      setTimeout(generateName, 100);
    }
  };

  const handleDisplayNameChange = (displayName: string) => {
    setData('display_name', displayName);
    // Auto-generate name if category exists
    if (data.category) {
      setTimeout(generateName, 100);
    }
  };

  return (
    <AppLayout breadcrumbs={[
      { title: 'Permissions', href: '/admin/permissions' },
      { title: permission.display_name, href: `/admin/permissions/${permission.id}` },
      { title: 'Edit', href: `/admin/permissions/${permission.id}/edit` }
    ]}>
      <Head title={`Edit ${permission.display_name}`} />
      
      <div className="space-y-6 p-4 flex flex-col gap-4">
        {/* Header */}
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-4">
            <Button variant="outline" size="sm" asChild>
              <Link href={route('admin.permissions.show', permission.id)}>
                <ArrowLeft className="h-4 w-4 mr-2" />
                Back to Permission
              </Link>
            </Button>
            <div>
              <h1 className="text-3xl font-bold">Edit Permission</h1>
              <p className="text-muted-foreground">Update permission details and settings</p>
            </div>
          </div>
        </div>

        {/* System Permission Warning */}
        {permission.is_system && (
          <Alert>
            <AlertTriangle className="h-4 w-4" />
            <AlertDescription>
              This is a system permission. Modifying it may affect core application functionality.
              Please proceed with caution.
            </AlertDescription>
          </Alert>
        )}

        <div className="grid gap-6 md:grid-cols-3">
          {/* Main Form */}
          <Card className="md:col-span-2">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Key className="h-5 w-5" />
                Permission Details
              </CardTitle>
              <CardDescription>
                Update the permission name, category, and description
              </CardDescription>
            </CardHeader>
            <CardContent>
              <form onSubmit={handleSubmit} className="space-y-6">
                <div className="grid gap-4 md:grid-cols-2">
                  <div className="space-y-2">
                    <Label htmlFor="category">Category *</Label>
                    <Select 
                      value={data.category} 
                      onValueChange={handleCategoryChange}
                      required
                    >
                      <SelectTrigger>
                        <SelectValue placeholder="Select category" />
                      </SelectTrigger>
                      <SelectContent>
                        {Object.entries(categories).map(([key, label]) => (
                          <SelectItem key={key} value={key}>{label}</SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    {errors.category && (
                      <p className="text-sm text-red-600">{errors.category}</p>
                    )}
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="display_name">Display Name *</Label>
                    <Input
                      id="display_name"
                      type="text"
                      value={data.display_name}
                      onChange={(e) => handleDisplayNameChange(e.target.value)}
                      placeholder="e.g., View Tournaments"
                      required
                    />
                    {errors.display_name && (
                      <p className="text-sm text-red-600">{errors.display_name}</p>
                    )}
                  </div>
                </div>

                <div className="space-y-2">
                  <Label htmlFor="name">Permission Name *</Label>
                  <Input
                    id="name"
                    type="text"
                    value={data.name}
                    onChange={(e) => setData('name', e.target.value)}
                    placeholder="e.g., tournaments.view"
                    className="font-mono"
                    required
                  />
                  <p className="text-xs text-muted-foreground">
                    Use format: category.action (e.g., tournaments.view, users.create)
                  </p>
                  {errors.name && (
                    <p className="text-sm text-red-600">{errors.name}</p>
                  )}
                </div>

                <div className="space-y-2">
                  <Label htmlFor="description">Description</Label>
                  <Textarea
                    id="description"
                    value={data.description}
                    onChange={(e) => setData('description', e.target.value)}
                    placeholder="Describe what this permission allows users to do..."
                    rows={3}
                  />
                  {errors.description && (
                    <p className="text-sm text-red-600">{errors.description}</p>
                  )}
                </div>

                <div className="space-y-2">
                  <Label htmlFor="sort_order">Sort Order</Label>
                  <Input
                    id="sort_order"
                    type="number"
                    value={data.sort_order}
                    onChange={(e) => setData('sort_order', parseInt(e.target.value) || 0)}
                    placeholder="0"
                    min="0"
                  />
                  <p className="text-xs text-muted-foreground">
                    Controls the display order within the category (0 = first)
                  </p>
                  {errors.sort_order && (
                    <p className="text-sm text-red-600">{errors.sort_order}</p>
                  )}
                </div>

                <div className="flex justify-end gap-4">
                  <Button type="button" variant="outline" asChild>
                    <Link href={route('admin.permissions.show', permission.id)}>
                      Cancel
                    </Link>
                  </Button>
                  <Button type="submit" disabled={processing}>
                    <Save className="h-4 w-4 mr-2" />
                    {processing ? 'Updating...' : 'Update Permission'}
                  </Button>
                </div>
              </form>
            </CardContent>
          </Card>

          {/* Settings Sidebar */}
          <div className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle>Settings</CardTitle>
                <CardDescription>
                  Configure permission behavior
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-center justify-between">
                  <div className="space-y-0.5">
                    <Label htmlFor="is_active">Active</Label>
                    <p className="text-xs text-muted-foreground">
                      Enable this permission
                    </p>
                  </div>
                  <Switch
                    id="is_active"
                    checked={data.is_active}
                    onCheckedChange={(checked) => setData('is_active', checked)}
                  />
                </div>

                <div className="flex items-center justify-between">
                  <div className="space-y-0.5">
                    <Label htmlFor="is_system">System Permission</Label>
                    <p className="text-xs text-muted-foreground">
                      Core system permission (cannot be deleted)
                    </p>
                  </div>
                  <Switch
                    id="is_system"
                    checked={data.is_system}
                    onCheckedChange={(checked) => setData('is_system', checked)}
                  />
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Permission Info</CardTitle>
              </CardHeader>
              <CardContent className="text-sm space-y-2">
                <div>
                  <strong>Created:</strong>
                  <p className="text-muted-foreground">
                    {new Date(permission.created_at).toLocaleDateString()}
                  </p>
                </div>
                <div>
                  <strong>Last Modified:</strong>
                  <p className="text-muted-foreground">
                    {new Date(permission.updated_at).toLocaleDateString()}
                  </p>
                </div>
                <div>
                  <strong>ID:</strong>
                  <p className="text-muted-foreground font-mono">#{permission.id}</p>
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Quick Actions</CardTitle>
                <CardDescription>
                  Common permission actions
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="grid gap-2">
                  {Object.entries(actions).map(([key, label]) => (
                    <Button
                      key={key}
                      variant="outline"
                      size="sm"
                      onClick={() => {
                        if (data.category) {
                          setData('name', `${data.category}.${key}`);
                          setData('display_name', `${label} ${categories[data.category] || data.category}`);
                        }
                      }}
                      className="justify-start"
                    >
                      {label}
                    </Button>
                  ))}
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </AppLayout>
  );
};

export default EditPermission;