import React from 'react';
import { Head, Link } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { 
  ArrowLeft,
  User,
  Users,
  CreditCard,
  Calendar,
  Hash,
  FileText,
  CheckCircle,
  DollarSign
} from 'lucide-react';

interface Payment {
  id: number;
  user: {
    id: number;
    name: string;
    email: string;
    phone?: string;
  };
  team?: {
    id: number;
    name: string;
  };
  amount: number;
  currency: string;
  payment_method: string;
  reference_number: string;
  checkout_request_id: string;
  merchant_request_id: string;
  description: string;
  payment_data?: any;
  paid_at: string;
  created_at: string;
  updated_at: string;
}

interface Props {
  payment: Payment;
}

const PaymentShow: React.FC<Props> = ({ payment }) => {
  const formatCurrency = (amount: number, currency: string = 'KES') => {
    return new Intl.NumberFormat('en-KE', {
      style: 'currency',
      currency: currency,
    }).format(amount);
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'long',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
      second: '2-digit'
    });
  };

  return (
    <AppLayout breadcrumbs={[
      { title: 'Payments', href: '/admin/payments' },
      { title: `Payment #${payment.id}`, href: `/admin/payments/${payment.id}` }
    ]}>
      <Head title={`Payment #${payment.id}`} />
      
      <div className="space-y-6 p-4 flex flex-col gap-4">
        {/* Header */}
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-4">
            <Button variant="outline" size="sm" asChild>
              <Link href={route('admin.payments.index')}>
                <ArrowLeft className="h-4 w-4 mr-2" />
                Back to Payments
              </Link>
            </Button>
            <div>
              <h1 className="text-3xl font-bold">Payment #{payment.id}</h1>
              <p className="text-muted-foreground">View payment details and transaction information</p>
            </div>
          </div>
          <Badge variant="default" className="bg-green-100 text-green-800 border-green-200">
            <CheckCircle className="h-4 w-4 mr-1" />
            Completed
          </Badge>
        </div>

        <div className="grid gap-6 md:grid-cols-2">
          {/* Payment Information */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <CreditCard className="h-5 w-5" />
                Payment Information
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Amount</label>
                  <div className="flex items-center gap-2 mt-1">
                    <DollarSign className="h-4 w-4 text-green-600" />
                    <span className="text-2xl font-bold text-green-600">
                      {formatCurrency(payment.amount, payment.currency)}
                    </span>
                  </div>
                </div>
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Payment Method</label>
                  <div className="mt-1">
                    <Badge variant="secondary" className="capitalize">
                      {payment.payment_method}
                    </Badge>
                  </div>
                </div>
              </div>

              <div>
                <label className="text-sm font-medium text-muted-foreground">Description</label>
                <div className="flex items-center gap-2 mt-1">
                  <FileText className="h-4 w-4 text-muted-foreground" />
                  <span>{payment.description}</span>
                </div>
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Reference Number</label>
                  <div className="flex items-center gap-2 mt-1">
                    <Hash className="h-4 w-4 text-muted-foreground" />
                    <span className="font-mono text-sm">{payment.reference_number || 'N/A'}</span>
                  </div>
                </div>
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Currency</label>
                  <div className="mt-1">
                    <span className="font-medium">{payment.currency}</span>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Customer Information */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <User className="h-5 w-5" />
                Customer Information
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div>
                <label className="text-sm font-medium text-muted-foreground">User</label>
                <div className="mt-1">
                  <Link 
                    href={route('admin.users.show', payment.user.id)}
                    className="flex items-center gap-2 text-blue-600 hover:underline"
                  >
                    <User className="h-4 w-4" />
                    <div>
                      <div className="font-medium">{payment.user.name}</div>
                      <div className="text-sm text-muted-foreground">{payment.user.email}</div>
                    </div>
                  </Link>
                </div>
              </div>

              {payment.user.phone && (
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Phone</label>
                  <div className="mt-1">
                    <span className="font-mono">{payment.user.phone}</span>
                  </div>
                </div>
              )}

              {payment.team && (
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Team</label>
                  <div className="mt-1">
                    <Link 
                      href={route('admin.teams.show', payment.team.id)}
                      className="flex items-center gap-2 text-blue-600 hover:underline"
                    >
                      <Users className="h-4 w-4" />
                      <span className="font-medium">{payment.team.name}</span>
                    </Link>
                  </div>
                </div>
              )}
            </CardContent>
          </Card>

          {/* Transaction Details */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Hash className="h-5 w-5" />
                Transaction Details
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div>
                <label className="text-sm font-medium text-muted-foreground">Checkout Request ID</label>
                <div className="mt-1">
                  <span className="font-mono text-sm bg-muted px-2 py-1 rounded">
                    {payment.checkout_request_id}
                  </span>
                </div>
              </div>

              {payment.merchant_request_id && (
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Merchant Request ID</label>
                  <div className="mt-1">
                    <span className="font-mono text-sm bg-muted px-2 py-1 rounded">
                      {payment.merchant_request_id}
                    </span>
                  </div>
                </div>
              )}

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Created At</label>
                  <div className="flex items-center gap-2 mt-1">
                    <Calendar className="h-4 w-4 text-muted-foreground" />
                    <span className="text-sm">{formatDate(payment.created_at)}</span>
                  </div>
                </div>
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Paid At</label>
                  <div className="flex items-center gap-2 mt-1">
                    <Calendar className="h-4 w-4 text-green-600" />
                    <span className="text-sm font-medium text-green-600">
                      {formatDate(payment.paid_at)}
                    </span>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Payment Data */}
          {payment.payment_data && (
            <Card>
              <CardHeader>
                <CardTitle>Raw Payment Data</CardTitle>
                <CardDescription>
                  Technical details from the payment processor
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="bg-muted p-4 rounded-lg overflow-x-auto">
                  <pre className="text-sm">
                    {JSON.stringify(payment.payment_data, null, 2)}
                  </pre>
                </div>
              </CardContent>
            </Card>
          )}
        </div>
      </div>
    </AppLayout>
  );
};

export default PaymentShow;