import React, { useState } from 'react';
import { Head, Link, router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { 
  CreditCard, 
  DollarSign, 
  Users, 
  TrendingUp, 
  TrendingDown,
  Calendar,
  Search,
  Filter,
  Eye,
  Download,
  BarChart3
} from 'lucide-react';

interface Payment {
  id: number;
  user: {
    id: number;
    name: string;
    email: string;
  };
  team?: {
    id: number;
    name: string;
  };
  amount: number;
  currency: string;
  payment_method: string;
  reference_number: string;
  checkout_request_id: string;
  description: string;
  paid_at: string;
  created_at: string;
}

interface PaymentStats {
  total_payments: number;
  total_amount: number;
  average_amount: number;
  payment_methods: Array<{
    payment_method: string;
    count: number;
    total: number;
  }>;
  daily_stats: Array<{
    date: string;
    count: number;
    total: number;
  }>;
  monthly_stats: Array<{
    month: number;
    year: number;
    count: number;
    total: number;
  }>;
  current_month?: {
    count: number;
    amount: number;
  };
  previous_month?: {
    count: number;
    amount: number;
  };
  growth?: {
    count: number;
    amount: number;
  };
}

interface Props {
  payments: {
    data: Payment[];
    current_page: number;
    last_page: number;
    per_page: number;
    total: number;
  };
  stats: PaymentStats;
  paymentMethods: string[];
  filters: {
    search: string;
    payment_method: string;
    date_range: string;
    sort_by: string;
    sort_order: string;
  };
}

const PaymentsIndex: React.FC<Props> = ({ payments, stats, paymentMethods, filters }) => {
  const [searchTerm, setSearchTerm] = useState(filters.search || '');

  const formatCurrency = (amount: number, currency: string = 'KES') => {
    return new Intl.NumberFormat('en-KE', {
      style: 'currency',
      currency: currency,
    }).format(amount);
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    router.get(route('admin.payments.index'), {
      ...filters,
      search: searchTerm,
    });
  };

  const handleFilter = (key: string, value: string) => {
    router.get(route('admin.payments.index'), {
      ...filters,
      [key]: value,
    });
  };

  const handleSort = (column: string) => {
    const newOrder = filters.sort_by === column && filters.sort_order === 'asc' ? 'desc' : 'asc';
    router.get(route('admin.payments.index'), {
      ...filters,
      sort_by: column,
      sort_order: newOrder,
    });
  };

  const getGrowthIcon = (growth: number) => {
    if (growth > 0) return <TrendingUp className="h-4 w-4 text-green-600" />;
    if (growth < 0) return <TrendingDown className="h-4 w-4 text-red-600" />;
    return null;
  };

  const getGrowthColor = (growth: number) => {
    if (growth > 0) return 'text-green-600';
    if (growth < 0) return 'text-red-600';
    return 'text-gray-600';
  };

  // Statistics cards
  const statCards = [
    {
      title: 'Total Payments',
      value: stats.total_payments.toLocaleString(),
      icon: CreditCard,
      description: stats.current_month ? `${stats.current_month.count} this month` : 'Completed payments',
      color: 'text-blue-600',
      growth: stats.growth?.count,
    },
    {
      title: 'Total Revenue',
      value: formatCurrency(stats.total_amount),
      icon: DollarSign,
      description: stats.current_month ? `${formatCurrency(stats.current_month.amount)} this month` : 'Total collected',
      color: 'text-green-600',
      growth: stats.growth?.amount,
    },
    {
      title: 'Average Payment',
      value: formatCurrency(stats.average_amount),
      icon: BarChart3,
      description: 'Per transaction',
      color: 'text-purple-600',
    },
    {
      title: 'Payment Methods',
      value: stats.payment_methods.length,
      icon: Users,
      description: `${stats.payment_methods.find(pm => pm.payment_method === 'mpesa')?.count || 0} M-Pesa`,
      color: 'text-orange-600',
    },
  ];

  return (
    <AppLayout breadcrumbs={[{ title: 'Payments', href: '/admin/payments' }]}>
      <Head title="Payments Management" />
      
      <div className="space-y-6 p-4 flex flex-col gap-4">
        {/* Header */}
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-3xl font-bold">Payments Management</h1>
            <p className="text-muted-foreground">View and manage completed payments with detailed statistics</p>
          </div>
          <div className="flex gap-2">
            <Button variant="outline" className="gap-2">
              <Download className="h-4 w-4" />
              Export
            </Button>
          </div>
        </div>

        {/* Statistics Grid */}
        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
          {statCards.map((stat) => {
            const IconComponent = stat.icon;
            return (
              <Card key={stat.title}>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">
                    {stat.title}
                  </CardTitle>
                  <IconComponent className={`h-4 w-4 ${stat.color}`} />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{stat.value}</div>
                  <div className="flex items-center gap-1 text-xs text-muted-foreground">
                    {stat.growth !== undefined && (
                      <>
                        {getGrowthIcon(stat.growth)}
                        <span className={getGrowthColor(stat.growth)}>
                          {stat.growth > 0 ? '+' : ''}{stat.growth.toFixed(1)}%
                        </span>
                      </>
                    )}
                    <span className={stat.growth !== undefined ? 'ml-1' : ''}>
                      {stat.description}
                    </span>
                  </div>
                </CardContent>
              </Card>
            );
          })}
        </div>

        {/* Payment Methods Breakdown */}
        {stats.payment_methods.length > 0 && (
          <Card>
            <CardHeader>
              <CardTitle>Payment Methods Breakdown</CardTitle>
              <CardDescription>Revenue and transaction count by payment method</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
                {stats.payment_methods.map((method) => (
                  <div key={method.payment_method} className="flex items-center justify-between p-4 border rounded-lg">
                    <div>
                      <p className="font-medium capitalize">{method.payment_method}</p>
                      <p className="text-sm text-muted-foreground">{method.count} transactions</p>
                    </div>
                    <div className="text-right">
                      <p className="font-bold">{formatCurrency(method.total)}</p>
                      <p className="text-sm text-muted-foreground">
                        {((method.total / stats.total_amount) * 100).toFixed(1)}%
                      </p>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        )}

        {/* Filters and Search */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Filter className="h-5 w-5" />
              Filters & Search
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="flex flex-col sm:flex-row gap-4">
              <form onSubmit={handleSearch} className="flex gap-2 flex-1">
                <div className="relative flex-1">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
                  <Input
                    placeholder="Search by reference, user, team, or description..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="pl-10"
                  />
                </div>
                <Button type="submit">Search</Button>
              </form>

              <div className="flex gap-2">
                <Select
                  value={filters.payment_method || 'all'}
                  onValueChange={(value) => handleFilter('payment_method', value)}
                >
                  <SelectTrigger className="w-40">
                    <SelectValue placeholder="Payment Method" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Methods</SelectItem>
                    {paymentMethods.map((method) => (
                      <SelectItem key={method} value={method}>
                        {method.charAt(0).toUpperCase() + method.slice(1)}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>

                <Select
                  value={filters.date_range || 'all'}
                  onValueChange={(value) => handleFilter('date_range', value)}
                >
                  <SelectTrigger className="w-32">
                    <SelectValue placeholder="Date Range" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Time</SelectItem>
                    <SelectItem value="today">Today</SelectItem>
                    <SelectItem value="week">This Week</SelectItem>
                    <SelectItem value="month">This Month</SelectItem>
                    <SelectItem value="year">This Year</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Payments Table */}
        <Card>
          <CardHeader>
            <CardTitle>Completed Payments</CardTitle>
            <CardDescription>
              Showing {payments.data.length} of {payments.total} payments
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="overflow-x-auto">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead 
                      className="cursor-pointer hover:bg-muted/50"
                      onClick={() => handleSort('reference_number')}
                    >
                      Reference
                    </TableHead>
                    <TableHead 
                      className="cursor-pointer hover:bg-muted/50"
                      onClick={() => handleSort('user_id')}
                    >
                      User
                    </TableHead>
                    <TableHead>Team</TableHead>
                    <TableHead 
                      className="cursor-pointer hover:bg-muted/50 text-right"
                      onClick={() => handleSort('amount')}
                    >
                      Amount
                    </TableHead>
                    <TableHead 
                      className="cursor-pointer hover:bg-muted/50"
                      onClick={() => handleSort('payment_method')}
                    >
                      Method
                    </TableHead>
                    <TableHead>Description</TableHead>
                    <TableHead 
                      className="cursor-pointer hover:bg-muted/50"
                      onClick={() => handleSort('paid_at')}
                    >
                      Paid At
                    </TableHead>
                    <TableHead>Actions</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {payments.data.map((payment) => (
                    <TableRow key={payment.id}>
                      <TableCell>
                        <div className="font-mono text-sm">
                          {payment.reference_number || payment.checkout_request_id?.slice(-8)}
                        </div>
                      </TableCell>
                      <TableCell>
                        <div>
                          <div className="font-medium">{payment.user.name}</div>
                          <div className="text-sm text-muted-foreground">{payment.user.email}</div>
                        </div>
                      </TableCell>
                      <TableCell>
                        {payment.team ? (
                          <Link 
                            href={route('admin.teams.show', payment.team.id)}
                            className="text-blue-600 hover:underline"
                          >
                            {payment.team.name}
                          </Link>
                        ) : (
                          <span className="text-muted-foreground">-</span>
                        )}
                      </TableCell>
                      <TableCell className="text-right font-medium">
                        {formatCurrency(payment.amount, payment.currency)}
                      </TableCell>
                      <TableCell>
                        <Badge variant="secondary" className="capitalize">
                          {payment.payment_method}
                        </Badge>
                      </TableCell>
                      <TableCell>
                        <div className="max-w-xs truncate" title={payment.description}>
                          {payment.description}
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-1">
                          <Calendar className="h-4 w-4 text-muted-foreground" />
                          {formatDate(payment.paid_at)}
                        </div>
                      </TableCell>
                      <TableCell>
                        <Button
                          variant="ghost"
                          size="sm"
                          asChild
                        >
                          <Link href={route('admin.payments.show', payment.id)}>
                            <Eye className="h-4 w-4" />
                          </Link>
                        </Button>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>

            {/* Pagination */}
            {payments.last_page > 1 && (
              <div className="flex items-center justify-between mt-4">
                <div className="text-sm text-muted-foreground">
                  Showing {((payments.current_page - 1) * payments.per_page) + 1} to{' '}
                  {Math.min(payments.current_page * payments.per_page, payments.total)} of{' '}
                  {payments.total} payments
                </div>
                <div className="flex gap-2">
                  {payments.current_page > 1 && (
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => router.get(route('admin.payments.index'), {
                        ...filters,
                        page: payments.current_page - 1,
                      })}
                    >
                      Previous
                    </Button>
                  )}
                  {payments.current_page < payments.last_page && (
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => router.get(route('admin.payments.index'), {
                        ...filters,
                        page: payments.current_page + 1,
                      })}
                    >
                      Next
                    </Button>
                  )}
                </div>
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </AppLayout>
  );
};

export default PaymentsIndex;