import React from 'react';
import { Head, Link, router } from '@inertiajs/react';
import { route } from 'ziggy-js';
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { 
  ArrowLeft, 
  Edit, 
  Trash2, 
  Download, 
  Copy,
  Calendar,
  User,
  FileText,
  HardDrive,
  Eye,
  Tag,
  Folder,
  Image as ImageIcon,
  Video,
  Music
} from 'lucide-react';
import { toast } from 'react-hot-toast';
import Swal from 'sweetalert2';

interface MediaFile {
  id: number;
  name: string;
  file_name: string;
  mime_type: string;
  size: number;
  url: string;
  human_readable_size: string;
  is_image: boolean;
  is_video: boolean;
  is_audio: boolean;
  is_document: boolean;
  alt_text?: string;
  description?: string;
  collection_name?: string;
  created_at: string;
  updated_at: string;
  user?: {
    id: number;
    name: string;
    email: string;
  };
}

interface Props {
  media: MediaFile;
}

const MediaShow = ({ media }: Props) => {
  const handleDelete = () => {
    Swal.fire({
      title: 'Are you sure?',
      text: 'This action cannot be undone!',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonColor: '#c22b2b',
      cancelButtonColor: '#256630',
      confirmButtonText: 'Yes, delete it!'
    }).then((result) => {
      if (result.isConfirmed) {
        router.delete(route('admin.media.destroy', media.id), {
          onSuccess: () => {
            toast.success('File deleted successfully');
            Swal.fire('Deleted!', 'The file has been deleted.', 'success');
          },
          onError: () => {
            toast.error('Failed to delete file');
          }
        });
      }
    });
  };

  const copyToClipboard = (text: string) => {
    navigator.clipboard.writeText(text).then(() => {
      toast.success('URL copied to clipboard');
    }).catch(() => {
      toast.error('Failed to copy URL');
    });
  };

  const getFileIcon = () => {
    if (media.is_image) return <ImageIcon className="h-12 w-12 text-blue-500" />;
    if (media.is_video) return <Video className="h-12 w-12 text-red-500" />;
    if (media.is_audio) return <Music className="h-12 w-12 text-green-500" />;
    return <FileText className="h-12 w-12 text-gray-500" />;
  };

  const getTypeColor = () => {
    if (media.is_image) return 'bg-blue-100 text-blue-800';
    if (media.is_video) return 'bg-red-100 text-red-800';
    if (media.is_audio) return 'bg-green-100 text-green-800';
    return 'bg-gray-100 text-gray-800';
  };

  const getFileType = () => {
    if (media.is_image) return 'Image';
    if (media.is_video) return 'Video';
    if (media.is_audio) return 'Audio';
    return 'Document';
  };

  return (
    <AppLayout>
      <Head title={`Media: ${media.name}`} />
      
      <div className="space-y-6">
        {/* Header */}
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-4">
            <Link href={route('admin.media.index')}>
              <Button variant="ghost">
                <ArrowLeft className="h-4 w-4 mr-2" />
                Back to Media
              </Button>
            </Link>
            <div>
              <h1 className="text-2xl font-bold text-gray-900">{media.name}</h1>
              <p className="text-gray-600">{media.file_name}</p>
            </div>
          </div>
          
          <div className="flex items-center space-x-2">
            <Button 
              variant="outline"
              onClick={() => copyToClipboard(media.url)}
            >
              <Copy className="h-4 w-4 mr-2" />
              Copy URL
            </Button>
            <Button asChild variant="outline">
              <a href={media.url} download target="_blank" rel="noopener noreferrer">
                <Download className="h-4 w-4 mr-2" />
                Download
              </a>
            </Button>
            <Link href={route('admin.media.edit', media.id)}>
              <Button>
                <Edit className="h-4 w-4 mr-2" />
                Edit
              </Button>
            </Link>
            <Button 
              variant="destructive"
              onClick={handleDelete}
            >
              <Trash2 className="h-4 w-4 mr-2" />
              Delete
            </Button>
          </div>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          {/* Preview */}
          <div className="lg:col-span-2">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center">
                  <Eye className="h-5 w-5 mr-2" />
                  Preview
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="flex items-center justify-center bg-gray-50 rounded-lg p-8 min-h-96">
                  {media.is_image ? (
                    <img
                      src={media.url}
                      alt={media.alt_text || media.name}
                      className="max-w-full max-h-96 object-contain rounded-lg shadow-lg"
                    />
                  ) : media.is_video ? (
                    <video
                      src={media.url}
                      controls
                      className="max-w-full max-h-96 rounded-lg shadow-lg"
                    >
                      Your browser does not support the video tag.
                    </video>
                  ) : media.is_audio ? (
                    <div className="text-center">
                      {getFileIcon()}
                      <audio
                        src={media.url}
                        controls
                        className="mt-4"
                      >
                        Your browser does not support the audio tag.
                      </audio>
                    </div>
                  ) : (
                    <div className="text-center">
                      {getFileIcon()}
                      <p className="mt-4 text-gray-600">Document preview not available</p>
                      <Button asChild variant="outline" className="mt-2">
                        <a href={media.url} target="_blank" rel="noopener noreferrer">
                          <Download className="h-4 w-4 mr-2" />
                          Download to View
                        </a>
                      </Button>
                    </div>
                  )}
                </div>
              </CardContent>
            </Card>

            {/* Description */}
            {media.description && (
              <Card className="mt-6">
                <CardHeader>
                  <CardTitle className="flex items-center">
                    <FileText className="h-5 w-5 mr-2" />
                    Description
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <p className="text-gray-700">{media.description}</p>
                </CardContent>
              </Card>
            )}
          </div>

          {/* Details */}
          <div className="space-y-6">
            {/* File Info */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center">
                  <HardDrive className="h-5 w-5 mr-2" />
                  File Information
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-center justify-between">
                  <span className="text-sm font-medium text-gray-600">Type</span>
                  <Badge className={getTypeColor()}>
                    {getFileType()}
                  </Badge>
                </div>
                
                <div className="flex items-center justify-between">
                  <span className="text-sm font-medium text-gray-600">Size</span>
                  <span className="text-sm text-gray-900">{media.human_readable_size}</span>
                </div>
                
                <div className="flex items-center justify-between">
                  <span className="text-sm font-medium text-gray-600">MIME Type</span>
                  <span className="text-sm text-gray-900 font-mono">{media.mime_type}</span>
                </div>
                
                <div className="flex items-center justify-between">
                  <span className="text-sm font-medium text-gray-600">File Name</span>
                  <span className="text-sm text-gray-900 font-mono break-all">{media.file_name}</span>
                </div>

                {media.collection_name && (
                  <div className="flex items-center justify-between">
                    <span className="text-sm font-medium text-gray-600">Collection</span>
                    <Badge variant="secondary" className="flex items-center">
                      <Folder className="h-3 w-3 mr-1" />
                      {media.collection_name}
                    </Badge>
                  </div>
                )}
              </CardContent>
            </Card>

            {/* Meta Information */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center">
                  <Tag className="h-5 w-5 mr-2" />
                  Meta Information
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                {media.alt_text && (
                  <div>
                    <span className="text-sm font-medium text-gray-600">Alt Text</span>
                    <p className="text-sm text-gray-900 mt-1">{media.alt_text}</p>
                  </div>
                )}

                <div>
                  <span className="text-sm font-medium text-gray-600">URL</span>
                  <div className="flex items-center mt-1">
                    <code className="text-xs bg-gray-100 px-2 py-1 rounded flex-1 mr-2 break-all">
                      {media.url}
                    </code>
                    <Button 
                      size="sm" 
                      variant="ghost"
                      onClick={() => copyToClipboard(media.url)}
                    >
                      <Copy className="h-3 w-3" />
                    </Button>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Upload Info */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center">
                  <Calendar className="h-5 w-5 mr-2" />
                  Upload Information
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-center justify-between">
                  <span className="text-sm font-medium text-gray-600">Uploaded</span>
                  <span className="text-sm text-gray-900">
                    {new Date(media.created_at).toLocaleDateString('en-US', {
                      year: 'numeric',
                      month: 'long',
                      day: 'numeric',
                      hour: '2-digit',
                      minute: '2-digit'
                    })}
                  </span>
                </div>

                {media.updated_at !== media.created_at && (
                  <div className="flex items-center justify-between">
                    <span className="text-sm font-medium text-gray-600">Modified</span>
                    <span className="text-sm text-gray-900">
                      {new Date(media.updated_at).toLocaleDateString('en-US', {
                        year: 'numeric',
                        month: 'long',
                        day: 'numeric',
                        hour: '2-digit',
                        minute: '2-digit'
                      })}
                    </span>
                  </div>
                )}

                {media.user && (
                  <div className="flex items-center justify-between">
                    <span className="text-sm font-medium text-gray-600">Uploaded by</span>
                    <div className="flex items-center">
                      <User className="h-4 w-4 text-gray-400 mr-1" />
                      <span className="text-sm text-gray-900">{media.user.name}</span>
                    </div>
                  </div>
                )}
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </AppLayout>
  );
};

export default MediaShow; 