import React, { useState } from 'react';
import { Head, Link, router } from '@inertiajs/react';
import { route } from 'ziggy-js';
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Checkbox } from '@/components/ui/checkbox';
import { 
  Plus, 
  Search, 
  Filter, 
  Download, 
  Trash2, 
  Edit, 
  Eye, 
  Grid3X3, 
  List,
  Upload,
  HardDrive,
  Image as ImageIcon,
  Video,
  FileText,
  Music
} from 'lucide-react';
import { toast } from 'react-hot-toast';
import Swal from 'sweetalert2';
import MediaManager from '@/components/MediaManager';

interface MediaFile {
  id: number;
  name: string;
  file_name: string;
  mime_type: string;
  size: number;
  url: string;
  human_readable_size: string;
  is_image: boolean;
  is_video: boolean;
  is_audio: boolean;
  is_document: boolean;
  alt_text?: string;
  description?: string;
  collection_name?: string;
  created_at: string;
  user?: {
    name: string;
  };
}

interface Props {
  media: {
    data: MediaFile[];
    links: any;
    meta: any;
  };
  collections: Array<{
    value: string;
    label: string;
    count: number;
  }>;
  filters: {
    search?: string;
    type?: string;
    collection?: string;
  };
  stats: {
    total: number;
    images: number;
    videos: number;
    documents: number;
    audio: number;
    size: string;
  };
}

const MediaIndex = ({ media, collections, filters, stats }: Props) => {
  const [selectedFiles, setSelectedFiles] = useState<number[]>([]);
  const [viewMode, setViewMode] = useState<'grid' | 'list'>('grid');
  const [uploadManagerOpen, setUploadManagerOpen] = useState(false);

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    const formData = new FormData(e.target as HTMLFormElement);
    router.get(route('admin.media.index'), Object.fromEntries(formData), {
      preserveState: true,
    });
  };

  const handleSelectFile = (fileId: number) => {
    setSelectedFiles(prev => 
      prev.includes(fileId) 
        ? prev.filter(id => id !== fileId)
        : [...prev, fileId]
    );
  };

  const handleSelectAll = () => {
    if (selectedFiles.length === media.data.length) {
      setSelectedFiles([]);
    } else {
      setSelectedFiles(media.data.map(file => file.id));
    }
  };

  const handleBulkDelete = () => {
    if (selectedFiles.length === 0) {
      toast.error('Please select files to delete');
      return;
    }

    Swal.fire({
      title: 'Are you sure?',
      text: `This will permanently delete ${selectedFiles.length} file(s). This action cannot be undone!`,
      icon: 'warning',
      showCancelButton: true,
      confirmButtonColor: '#c22b2b',
      cancelButtonColor: '#256630',
      confirmButtonText: 'Yes, delete them!'
    }).then((result) => {
      if (result.isConfirmed) {
        router.delete(route('admin.media.bulk-destroy'), {
          data: { ids: selectedFiles },
          onSuccess: () => {
            setSelectedFiles([]);
            toast.success('Files deleted successfully');
            Swal.fire('Deleted!', 'The files have been deleted.', 'success');
          },
          onError: () => {
            toast.error('Failed to delete files');
          }
        });
      }
    });
  };

  const handleDelete = (file: MediaFile) => {
    Swal.fire({
      title: 'Are you sure?',
      text: 'This action cannot be undone!',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonColor: '#c22b2b',
      cancelButtonColor: '#256630',
      confirmButtonText: 'Yes, delete it!'
    }).then((result) => {
      if (result.isConfirmed) {
        router.delete(route('admin.media.destroy', file.id), {
          onSuccess: () => {
            toast.success('File deleted successfully');
            Swal.fire('Deleted!', 'The file has been deleted.', 'success');
          },
          onError: () => {
            toast.error('Failed to delete file');
          }
        });
      }
    });
  };

  const getFileIcon = (file: MediaFile) => {
    if (file.is_image) return <ImageIcon className="h-8 w-8 text-blue-500" />;
    if (file.is_video) return <Video className="h-8 w-8 text-red-500" />;
    if (file.is_audio) return <Music className="h-8 w-8 text-green-500" />;
    return <FileText className="h-8 w-8 text-gray-500" />;
  };

  const getTypeColor = (file: MediaFile) => {
    if (file.is_image) return 'bg-blue-100 text-blue-800';
    if (file.is_video) return 'bg-red-100 text-red-800';
    if (file.is_audio) return 'bg-green-100 text-green-800';
    return 'bg-gray-100 text-gray-800';
  };

  return (
    <AppLayout>
      <Head title="Media Manager" />
      
      <div className="space-y-6">
        {/* Header */}
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between">
          <div>
            <h1 className="text-2xl font-bold text-gray-900">Media Manager</h1>
            <p className="mt-2 text-gray-600">Manage all your media files and assets</p>
          </div>
          <div className="mt-4 sm:mt-0">
            <MediaManager
              trigger={
                <Button>
                  <Plus className="h-4 w-4 mr-2" />
                  Upload Files
                </Button>
              }
              multiple={true}
              onSelect={() => {
                // Refresh the page after upload
                router.reload();
              }}
            />
          </div>
        </div>

        {/* Stats Cards */}
        <div className="grid grid-cols-2 gap-4 sm:grid-cols-3 lg:grid-cols-6">
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <HardDrive className="h-8 w-8 text-gray-500" />
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-600">Total Files</p>
                  <p className="text-2xl font-bold">{stats.total}</p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <ImageIcon className="h-8 w-8 text-blue-500" />
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-600">Images</p>
                  <p className="text-2xl font-bold">{stats.images}</p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <Video className="h-8 w-8 text-red-500" />
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-600">Videos</p>
                  <p className="text-2xl font-bold">{stats.videos}</p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <FileText className="h-8 w-8 text-gray-500" />
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-600">Documents</p>
                  <p className="text-2xl font-bold">{stats.documents}</p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <Music className="h-8 w-8 text-green-500" />
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-600">Audio</p>
                  <p className="text-2xl font-bold">{stats.audio}</p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <Upload className="h-8 w-8 text-purple-500" />
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-600">Total Size</p>
                  <p className="text-2xl font-bold">{stats.size}</p>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Filters */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center">
              <Filter className="h-5 w-5 mr-2" />
              Filters
            </CardTitle>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleSearch} className="flex flex-col sm:flex-row gap-4">
              <div className="flex-1">
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
                  <Input
                    name="search"
                    placeholder="Search files..."
                    defaultValue={filters.search}
                    className="pl-10"
                  />
                </div>
              </div>
              
              <Select name="type" defaultValue={filters.type || 'all'}>
                <SelectTrigger className="w-full sm:w-48">
                  <SelectValue placeholder="File Type" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Types</SelectItem>
                  <SelectItem value="images">Images</SelectItem>
                  <SelectItem value="videos">Videos</SelectItem>
                  <SelectItem value="documents">Documents</SelectItem>
                  <SelectItem value="audio">Audio</SelectItem>
                </SelectContent>
              </Select>
              
              <Select name="collection" defaultValue={filters.collection || 'all'}>
                <SelectTrigger className="w-full sm:w-48">
                  <SelectValue placeholder="Collection" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Collections</SelectItem>
                  {collections.map((collection) => (
                    <SelectItem key={collection.value} value={collection.value}>
                      {collection.label} ({collection.count})
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              
              <Button type="submit">
                <Search className="h-4 w-4 mr-2" />
                Search
              </Button>
            </form>
          </CardContent>
        </Card>

        {/* Actions Bar */}
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between">
          <div className="flex items-center space-x-4">
            <div className="flex items-center space-x-2">
              <Checkbox
                checked={selectedFiles.length === media.data.length && media.data.length > 0}
                onCheckedChange={handleSelectAll}
              />
              <span className="text-sm text-gray-600">
                {selectedFiles.length > 0 ? `${selectedFiles.length} selected` : 'Select all'}
              </span>
            </div>
            
            {selectedFiles.length > 0 && (
              <Button 
                variant="destructive" 
                size="sm"
                onClick={handleBulkDelete}
              >
                <Trash2 className="h-4 w-4 mr-2" />
                Delete Selected
              </Button>
            )}
          </div>
          
          <div className="flex items-center space-x-2 mt-4 sm:mt-0">
            <Button
              variant={viewMode === 'grid' ? 'default' : 'outline'}
              size="sm"
              onClick={() => setViewMode('grid')}
            >
              <Grid3X3 className="h-4 w-4" />
            </Button>
            <Button
              variant={viewMode === 'list' ? 'default' : 'outline'}
              size="sm"
              onClick={() => setViewMode('list')}
            >
              <List className="h-4 w-4" />
            </Button>
          </div>
        </div>

        {/* Media Grid/List */}
        {media.data.length === 0 ? (
          <Card>
            <CardContent className="text-center py-12">
              <Upload className="h-12 w-12 text-gray-300 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">No media files found</h3>
              <p className="text-gray-500 mb-4">
                {filters.search || filters.type !== 'all' || filters.collection !== 'all'
                  ? 'Try adjusting your search criteria'
                  : 'Upload some files to get started'
                }
              </p>
            </CardContent>
          </Card>
        ) : (
          <div className={
            viewMode === 'grid' 
              ? 'grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 xl:grid-cols-6 gap-4'
              : 'space-y-2'
          }>
            {media.data.map((file) => (
              <Card 
                key={file.id} 
                className={`cursor-pointer transition-all hover:shadow-md ${
                  selectedFiles.includes(file.id) ? 'ring-2 ring-primary' : ''
                }`}
              >
                <CardContent className={viewMode === 'grid' ? 'p-3' : 'p-4'}>
                  {viewMode === 'grid' ? (
                    <div>
                      <div className="flex items-center justify-between mb-2">
                        <Checkbox
                          checked={selectedFiles.includes(file.id)}
                          onCheckedChange={() => handleSelectFile(file.id)}
                        />
                        <div className="flex items-center space-x-1">
                          <Link href={route('admin.media.show', file.id)}>
                            <Button variant="ghost" size="sm">
                              <Eye className="h-3 w-3" />
                            </Button>
                          </Link>
                          <Link href={route('admin.media.edit', file.id)}>
                            <Button variant="ghost" size="sm">
                              <Edit className="h-3 w-3" />
                            </Button>
                          </Link>
                          <Button 
                            variant="ghost" 
                            size="sm"
                            onClick={() => handleDelete(file)}
                          >
                            <Trash2 className="h-3 w-3 text-red-500" />
                          </Button>
                        </div>
                      </div>
                      
                      <div className="aspect-square bg-gray-100 rounded-lg flex items-center justify-center mb-2">
                        {file.is_image ? (
                          <img
                            src={file.url}
                            alt={file.alt_text || file.name}
                            className="w-full h-full object-cover rounded-lg"
                          />
                        ) : (
                          getFileIcon(file)
                        )}
                      </div>
                      
                      <div className="text-xs text-gray-600 truncate" title={file.name}>
                        {file.name}
                      </div>
                      <div className="text-xs text-gray-400">
                        {file.human_readable_size}
                      </div>
                      
                      {file.collection_name && (
                        <Badge variant="secondary" className="text-xs mt-1">
                          {file.collection_name}
                        </Badge>
                      )}
                    </div>
                  ) : (
                    <div className="flex items-center space-x-4">
                      <Checkbox
                        checked={selectedFiles.includes(file.id)}
                        onCheckedChange={() => handleSelectFile(file.id)}
                      />
                      
                      <div className="flex-shrink-0">
                        {file.is_image ? (
                          <img
                            src={file.url}
                            alt={file.alt_text || file.name}
                            className="w-12 h-12 object-cover rounded"
                          />
                        ) : (
                          getFileIcon(file)
                        )}
                      </div>
                      
                      <div className="flex-1 min-w-0">
                        <div className="text-sm font-medium text-gray-900 truncate">
                          {file.name}
                        </div>
                        <div className="text-xs text-gray-500">
                          {file.human_readable_size} • {file.mime_type}
                        </div>
                        <div className="text-xs text-gray-400">
                          Uploaded {new Date(file.created_at).toLocaleDateString()} by {file.user?.name || 'Unknown'}
                        </div>
                      </div>
                      
                      <div className="flex items-center space-x-1">
                        <Badge className={getTypeColor(file)}>
                          {file.is_image ? 'Image' : file.is_video ? 'Video' : file.is_audio ? 'Audio' : 'Document'}
                        </Badge>
                        
                        {file.collection_name && (
                          <Badge variant="secondary">
                            {file.collection_name}
                          </Badge>
                        )}
                      </div>
                      
                      <div className="flex items-center space-x-1">
                        <Link href={route('admin.media.show', file.id)}>
                          <Button variant="ghost" size="sm">
                            <Eye className="h-4 w-4" />
                          </Button>
                        </Link>
                        <Link href={route('admin.media.edit', file.id)}>
                          <Button variant="ghost" size="sm">
                            <Edit className="h-4 w-4" />
                          </Button>
                        </Link>
                        <Button 
                          variant="ghost" 
                          size="sm"
                          onClick={() => handleDelete(file)}
                        >
                          <Trash2 className="h-4 w-4 text-red-500" />
                        </Button>
                      </div>
                    </div>
                  )}
                </CardContent>
              </Card>
            ))}
          </div>
        )}

        {/* Pagination */}
        {media.links && (
          <div className="flex items-center justify-between">
            <div className="text-sm text-gray-700">
              Showing {media?.meta?.from || 0} to {media?.meta?.to || 0} of {media?.meta?.total || 0} results
            </div>
            <div className="flex items-center space-x-2">
              {media.links.map((link: any, index: number) => (
                <Button
                  key={index}
                  variant={link.active ? 'default' : 'outline'}
                  size="sm"
                  disabled={!link.url}
                  onClick={() => link.url && router.get(link.url)}
                  dangerouslySetInnerHTML={{ __html: link.label }}
                />
              ))}
            </div>
          </div>
        )}
      </div>
    </AppLayout>
  );
};

export default MediaIndex; 