import React, { useState } from 'react';
import { Head, Link, router } from '@inertiajs/react';
import { route } from 'ziggy-js';
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { 
  ArrowLeft, 
  Save, 
  Eye,
  Image as ImageIcon,
  Video,
  Music,
  FileText
} from 'lucide-react';
import { toast } from 'react-hot-toast';

interface MediaFile {
  id: number;
  name: string;
  file_name: string;
  mime_type: string;
  size: number;
  url: string;
  human_readable_size: string;
  is_image: boolean;
  is_video: boolean;
  is_audio: boolean;
  is_document: boolean;
  alt_text?: string;
  description?: string;
  collection_name?: string;
  created_at: string;
  updated_at: string;
  user?: {
    id: number;
    name: string;
    email: string;
  };
}

interface Props {
  media: MediaFile;
  collections: string[];
}

const MediaEdit = ({ media, collections }: Props) => {
  const [form, setForm] = useState({
    name: media.name || '',
    alt_text: media.alt_text || '',
    description: media.description || '',
    collection_name: media.collection_name || '',
  });
  const [loading, setLoading] = useState(false);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    router.put(route('admin.media.update', media.id), form as any, {
      onSuccess: () => {
        toast.success('Media updated successfully');
      },
      onError: (errors) => {
        toast.error('Please check the form for errors');
      },
      onFinish: () => setLoading(false)
    });
  };

  const handleInputChange = (field: string, value: string) => {
    setForm(prev => ({ ...prev, [field]: value }));
  };

  const getFileIcon = () => {
    if (media.is_image) return <ImageIcon className="h-12 w-12 text-blue-500" />;
    if (media.is_video) return <Video className="h-12 w-12 text-red-500" />;
    if (media.is_audio) return <Music className="h-12 w-12 text-green-500" />;
    return <FileText className="h-12 w-12 text-gray-500" />;
  };

  return (
    <AppLayout>
      <Head title={`Edit Media: ${media.name}`} />
      
      <div className="space-y-6">
        {/* Header */}
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-4">
            <Link href={route('admin.media.show', media.id)}>
              <Button variant="ghost">
                <ArrowLeft className="h-4 w-4 mr-2" />
                Back
              </Button>
            </Link>
            <div>
              <h1 className="text-2xl font-bold text-gray-900">Edit Media</h1>
              <p className="text-gray-600">{media.file_name}</p>
            </div>
          </div>
          
          <div className="flex items-center space-x-2">
            <Link href={route('admin.media.show', media.id)}>
              <Button variant="outline">
                <Eye className="h-4 w-4 mr-2" />
                View
              </Button>
            </Link>
          </div>
        </div>

        <form onSubmit={handleSubmit} className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          {/* Preview */}
          <div className="lg:col-span-2">
            <Card>
              <CardHeader>
                <CardTitle>Preview</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="flex items-center justify-center bg-gray-50 rounded-lg p-8 min-h-64">
                  {media.is_image ? (
                    <img
                      src={media.url}
                      alt={form.alt_text || form.name}
                      className="max-w-full max-h-64 object-contain rounded-lg shadow-lg"
                    />
                  ) : media.is_video ? (
                    <video
                      src={media.url}
                      controls
                      className="max-w-full max-h-64 rounded-lg shadow-lg"
                    >
                      Your browser does not support the video tag.
                    </video>
                  ) : media.is_audio ? (
                    <div className="text-center">
                      {getFileIcon()}
                      <audio
                        src={media.url}
                        controls
                        className="mt-4"
                      >
                        Your browser does not support the audio tag.
                      </audio>
                    </div>
                  ) : (
                    <div className="text-center">
                      {getFileIcon()}
                      <p className="mt-4 text-gray-600">Document preview not available</p>
                    </div>
                  )}
                </div>
              </CardContent>
            </Card>
          </div>

          {/* Form */}
          <div className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle>Edit Details</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <Label htmlFor="name">Name *</Label>
                  <Input
                    id="name"
                    value={form.name}
                    onChange={(e) => handleInputChange('name', e.target.value)}
                    placeholder="Enter media name"
                    required
                  />
                </div>

                {media.is_image && (
                  <div>
                    <Label htmlFor="alt_text">Alt Text</Label>
                    <Input
                      id="alt_text"
                      value={form.alt_text}
                      onChange={(e) => handleInputChange('alt_text', e.target.value)}
                      placeholder="Describe the image for accessibility"
                    />
                    <p className="text-xs text-gray-500 mt-1">
                      Alternative text for screen readers and SEO
                    </p>
                  </div>
                )}

                <div>
                  <Label htmlFor="description">Description</Label>
                  <Textarea
                    id="description"
                    value={form.description}
                    onChange={(e) => handleInputChange('description', e.target.value)}
                    placeholder="Add a description for this media file"
                    rows={3}
                  />
                </div>

                <div>
                  <Label htmlFor="collection_name">Collection</Label>
                  <Select 
                    value={form.collection_name} 
                    onValueChange={(value) => handleInputChange('collection_name', value)}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="Select a collection" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="">No Collection</SelectItem>
                      {collections.map((collection) => (
                        <SelectItem key={collection} value={collection}>
                          {collection.charAt(0).toUpperCase() + collection.slice(1).replace('-', ' ')}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  <p className="text-xs text-gray-500 mt-1">
                    Group media files by collection for better organization
                  </p>
                </div>
              </CardContent>
            </Card>

            {/* File Information (Read-only) */}
            <Card>
              <CardHeader>
                <CardTitle>File Information</CardTitle>
              </CardHeader>
              <CardContent className="space-y-3">
                <div className="flex justify-between">
                  <span className="text-sm font-medium text-gray-600">File Name</span>
                  <span className="text-sm text-gray-900 font-mono">{media.file_name}</span>
                </div>
                
                <div className="flex justify-between">
                  <span className="text-sm font-medium text-gray-600">Size</span>
                  <span className="text-sm text-gray-900">{media.human_readable_size}</span>
                </div>
                
                <div className="flex justify-between">
                  <span className="text-sm font-medium text-gray-600">Type</span>
                  <span className="text-sm text-gray-900">{media.mime_type}</span>
                </div>
                
                <div className="flex justify-between">
                  <span className="text-sm font-medium text-gray-600">Uploaded</span>
                  <span className="text-sm text-gray-900">
                    {new Date(media.created_at).toLocaleDateString()}
                  </span>
                </div>

                {media.user && (
                  <div className="flex justify-between">
                    <span className="text-sm font-medium text-gray-600">Uploaded by</span>
                    <span className="text-sm text-gray-900">{media.user.name}</span>
                  </div>
                )}
              </CardContent>
            </Card>

            {/* Actions */}
            <Card>
              <CardContent className="pt-6">
                <div className="flex items-center space-x-2">
                  <Button 
                    type="submit" 
                    disabled={loading}
                    className="flex-1"
                  >
                    <Save className="h-4 w-4 mr-2" />
                    {loading ? 'Saving...' : 'Save Changes'}
                  </Button>
                  <Link href={route('admin.media.show', media.id)}>
                    <Button variant="outline" type="button">
                      Cancel
                    </Button>
                  </Link>
                </div>
              </CardContent>
            </Card>
          </div>
        </form>
      </div>
    </AppLayout>
  );
};

export default MediaEdit; 