import React, { useState } from 'react';
import { Head } from '@inertiajs/react';
import { router } from '@inertiajs/react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Progress } from '@/components/ui/progress';
import { 
  ArrowLeft, 
  Edit, 
  Play, 
  Pause, 
  Square,
  CheckCircle,
  Clock,
  Users,
  Trophy,
  Eye,
  Trash2,
  Calendar,
  MapPin,
  Monitor,
  Shield,
  Activity,
  BarChart3,
  TrendingUp
} from 'lucide-react';
import AdminLayout from '@/layouts/admin-layout';

interface Props {
  match: {
    id: number;
    name: string;
    match_code: string;
    match_type: string;
    format: string;
    best_of_games: number;
    time_limit_minutes: number | null;
    description: string;
    notes: string;
    status: string;
    progress_status: string;
    result_type: string;
    result_notes: string;
    verification_status: string;
    scheduled_time: string;
    started_at: string | null;
    completed_at: string | null;
    estimated_duration_minutes: number | null;
    actual_duration_minutes: number | null;
    player1_score: number;
    player2_score: number;
    game_scores: any[];
    round_scores: any[];
    is_streamed: boolean;
    stream_url: string;
    twitch_url: string;
    youtube_url: string;
    allow_spectators: boolean;
    spectator_count: number;
    peak_spectator_count: number;
    match_quality_score: number;
    total_actions: number;
    average_action_per_minute: number;
    created_at: string;
    updated_at: string;
    tournament?: {
      id: number;
      name: string;
    };
    bracket?: {
      id: number;
      name: string;
    };
    player1?: {
      id: number;
      display_name: string;
      game_tag: string;
    };
    player2?: {
      id: number;
      display_name: string;
      game_tag: string;
    };
    team1?: {
      id: number;
      name: string;
      tag: string;
    };
    team2?: {
      id: number;
      name: string;
      tag: string;
    };
    winner?: {
      id: number;
      display_name?: string;
      name?: string;
    };
    referee?: {
      id: number;
      name: string;
    };
  };
  detailedStats: any;
}

const ShowMatch: React.FC<Props> = ({ match, detailedStats }) => {
  const [isSubmitting, setIsSubmitting] = useState(false);

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'scheduled': return 'bg-blue-100 text-blue-800';
      case 'in_progress': return 'bg-yellow-100 text-yellow-800';
      case 'completed': return 'bg-green-100 text-green-800';
      case 'cancelled': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getProgressStatusColor = (status: string) => {
    switch (status) {
      case 'not_started': return 'bg-gray-100 text-gray-800';
      case 'in_progress': return 'bg-yellow-100 text-yellow-800';
      case 'paused': return 'bg-orange-100 text-orange-800';
      case 'completed': return 'bg-green-100 text-green-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getVerificationColor = (status: string) => {
    switch (status) {
      case 'verified': return 'bg-green-100 text-green-800';
      case 'disputed': return 'bg-red-100 text-red-800';
      case 'under_review': return 'bg-yellow-100 text-yellow-800';
      case 'pending': return 'bg-gray-100 text-gray-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const handleMatchAction = (action: string, data?: any) => {
    setIsSubmitting(true);
    
    const routes: Record<string, string> = {
      'start': 'admin.matches.start',
      'pause': 'admin.matches.pause',
      'resume': 'admin.matches.resume',
      'complete': 'admin.matches.complete',
    };

    if (routes[action]) {
      router.post(route(routes[action], match.id), data || {}, {
        onFinish: () => setIsSubmitting(false),
      });
    }
  };

  const handleDelete = () => {
    if (confirm('Are you sure you want to delete this match? This action cannot be undone.')) {
      router.delete(route('admin.matches.destroy', match.id));
    }
  };

  const getParticipantName = (participantType: 'player1' | 'player2' | 'team1' | 'team2') => {
    const participant = match[participantType];
    if (participantType.startsWith('player')) {
      return participant?.display_name || 'Unknown Player';
    } else {
      return participant?.name || 'Unknown Team';
    }
  };

  const getMatchProgress = () => {
    if (match.status === 'completed') return 100;
    if (match.progress_status === 'in_progress') return 60;
    if (match.progress_status === 'paused') return 40;
    if (match.status === 'scheduled') return 0;
    return 20;
  };

  return (
    <AdminLayout>
      <Head title={`Match - ${match.name}`} />
      
      <div className="space-y-6">
        {/* Header */}
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-4">
            <Button
              variant="outline"
              size="sm"
              onClick={() => router.get(route('admin.matches.index'))}
            >
              <ArrowLeft className="w-4 h-4 mr-2" />
              Back to Matches
            </Button>
            <div>
              <h1 className="text-3xl font-bold flex items-center gap-2">
                <Trophy className="w-8 h-8" />
                {match.name}
              </h1>
              <p className="text-muted-foreground">{match.match_code}</p>
            </div>
          </div>
          
          <div className="flex items-center gap-2">
            {match.status === 'scheduled' && (
              <Button
                onClick={() => handleMatchAction('start')}
                disabled={isSubmitting}
                className="bg-green-600 hover:bg-green-700"
              >
                <Play className="w-4 h-4 mr-2" />
                Start Match
              </Button>
            )}
            
            {match.progress_status === 'in_progress' && (
              <>
                <Button
                  onClick={() => handleMatchAction('pause')}
                  disabled={isSubmitting}
                  variant="outline"
                >
                  <Pause className="w-4 h-4 mr-2" />
                  Pause
                </Button>
                <Button
                  onClick={() => handleMatchAction('complete')}
                  disabled={isSubmitting}
                  className="bg-blue-600 hover:bg-blue-700"
                >
                  <CheckCircle className="w-4 h-4 mr-2" />
                  Complete
                </Button>
              </>
            )}
            
            {match.progress_status === 'paused' && (
              <Button
                onClick={() => handleMatchAction('resume')}
                disabled={isSubmitting}
                className="bg-yellow-600 hover:bg-yellow-700"
              >
                <Play className="w-4 h-4 mr-2" />
                Resume
              </Button>
            )}
            
            <Button
              onClick={() => router.get(route('admin.matches.edit', match.id))}
              variant="outline"
            >
              <Edit className="w-4 h-4 mr-2" />
              Edit
            </Button>
          </div>
        </div>

        {/* Status Overview */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center gap-2">
                <Activity className="w-5 h-5 text-blue-600" />
                <div>
                  <p className="text-sm text-muted-foreground">Status</p>
                  <Badge className={getStatusColor(match.status)}>
                    {match.status.replace('_', ' ')}
                  </Badge>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center gap-2">
                <Clock className="w-5 h-5 text-yellow-600" />
                <div>
                  <p className="text-sm text-muted-foreground">Progress</p>
                  <Badge className={getProgressStatusColor(match.progress_status)}>
                    {match.progress_status.replace('_', ' ')}
                  </Badge>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center gap-2">
                <Shield className="w-5 h-5 text-green-600" />
                <div>
                  <p className="text-sm text-muted-foreground">Verification</p>
                  <Badge className={getVerificationColor(match.verification_status)}>
                    {match.verification_status.replace('_', ' ')}
                  </Badge>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center gap-2">
                <TrendingUp className="w-5 h-5 text-purple-600" />
                <div>
                  <p className="text-sm text-muted-foreground">Quality Score</p>
                  <p className="text-lg font-semibold">
                    {match.match_quality_score ? `${match.match_quality_score}/10` : 'N/A'}
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Match Progress */}
        <Card>
          <CardHeader>
            <CardTitle>Match Progress</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              <Progress value={getMatchProgress()} className="h-3" />
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4 text-sm">
                <div>
                  <p className="text-muted-foreground">Scheduled</p>
                  <p className="font-medium">
                    {match.scheduled_time ? new Date(match.scheduled_time).toLocaleString() : 'Not scheduled'}
                  </p>
                </div>
                <div>
                  <p className="text-muted-foreground">Started</p>
                  <p className="font-medium">
                    {match.started_at ? new Date(match.started_at).toLocaleString() : 'Not started'}
                  </p>
                </div>
                <div>
                  <p className="text-muted-foreground">Completed</p>
                  <p className="font-medium">
                    {match.completed_at ? new Date(match.completed_at).toLocaleString() : 'Not completed'}
                  </p>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          {/* Match Details */}
          <div className="lg:col-span-2 space-y-6">
            {/* Participants & Score */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Users className="w-5 h-5" />
                  Participants & Score
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-6">
                  {/* Score Display */}
                  <div className="text-center">
                    <div className="flex items-center justify-center gap-8">
                      <div className="text-center">
                        <p className="text-2xl font-bold">{match.player1_score}</p>
                        <p className="text-lg font-medium">
                          {getParticipantName('player1') || getParticipantName('team1')}
                        </p>
                      </div>
                      <div className="text-4xl font-bold text-muted-foreground">VS</div>
                      <div className="text-center">
                        <p className="text-2xl font-bold">{match.player2_score}</p>
                        <p className="text-lg font-medium">
                          {getParticipantName('player2') || getParticipantName('team2')}
                        </p>
                      </div>
                    </div>
                  </div>

                  {/* Winner Display */}
                  {match.winner && (
                    <div className="text-center p-4 bg-green-50 rounded-lg">
                      <Trophy className="w-8 h-8 mx-auto mb-2 text-green-600" />
                      <p className="text-lg font-semibold text-green-800">
                        Winner: {match.winner.display_name || match.winner.name}
                      </p>
                    </div>
                  )}

                  {/* Game Scores */}
                  {match.game_scores && match.game_scores.length > 0 && (
                    <div>
                      <h4 className="font-medium mb-2">Game Breakdown</h4>
                      <div className="space-y-2">
                        {match.game_scores.map((score: any, index: number) => (
                          <div key={index} className="flex justify-between items-center p-2 bg-muted rounded">
                            <span>Game {index + 1}</span>
                            <span className="font-medium">{score.player1} - {score.player2}</span>
                          </div>
                        ))}
                      </div>
                    </div>
                  )}
                </div>
              </CardContent>
            </Card>

            {/* Match Information */}
            <Card>
              <CardHeader>
                <CardTitle>Match Information</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">Tournament</label>
                    <p className="text-lg">{match.tournament?.name || 'N/A'}</p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">Bracket</label>
                    <p className="text-lg">{match.bracket?.name || 'N/A'}</p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">Match Type</label>
                    <p className="text-lg capitalize">{match.match_type}</p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">Format</label>
                    <p className="text-lg">{match.format.replace('_', ' ')}</p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">Best of Games</label>
                    <p className="text-lg">{match.best_of_games}</p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">Time Limit</label>
                    <p className="text-lg">{match.time_limit_minutes ? `${match.time_limit_minutes} min` : 'No limit'}</p>
                  </div>
                </div>

                {match.description && (
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">Description</label>
                    <p className="text-lg">{match.description}</p>
                  </div>
                )}

                {match.notes && (
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">Notes</label>
                    <p className="text-lg">{match.notes}</p>
                  </div>
                )}
              </CardContent>
            </Card>

            {/* Streaming Information */}
            {match.is_streamed && (
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Monitor className="w-5 h-5" />
                    Streaming Information
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    {match.stream_url && (
                      <div>
                        <label className="text-sm font-medium text-muted-foreground">Stream URL</label>
                        <a href={match.stream_url} target="_blank" rel="noopener noreferrer" 
                           className="text-blue-600 hover:underline block">
                          {match.stream_url}
                        </a>
                      </div>
                    )}
                    {match.twitch_url && (
                      <div>
                        <label className="text-sm font-medium text-muted-foreground">Twitch URL</label>
                        <a href={match.twitch_url} target="_blank" rel="noopener noreferrer" 
                           className="text-purple-600 hover:underline block">
                          {match.twitch_url}
                        </a>
                      </div>
                    )}
                    {match.youtube_url && (
                      <div>
                        <label className="text-sm font-medium text-muted-foreground">YouTube URL</label>
                        <a href={match.youtube_url} target="_blank" rel="noopener noreferrer" 
                           className="text-red-600 hover:underline block">
                          {match.youtube_url}
                        </a>
                      </div>
                    )}
                  </div>
                </CardContent>
              </Card>
            )}
          </div>

          {/* Sidebar */}
          <div className="space-y-6">
            {/* Statistics */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <BarChart3 className="w-5 h-5" />
                  Statistics
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Duration</label>
                  <p className="text-lg font-semibold">
                    {match.actual_duration_minutes ? `${match.actual_duration_minutes} min` : 'N/A'}
                  </p>
                </div>
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Spectators</label>
                  <p className="text-lg font-semibold">
                    {match.spectator_count || 0} (Peak: {match.peak_spectator_count || 0})
                  </p>
                </div>
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Total Actions</label>
                  <p className="text-lg font-semibold">{match.total_actions || 0}</p>
                </div>
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Actions/Min</label>
                  <p className="text-lg font-semibold">{match.average_action_per_minute || 0}</p>
                </div>
              </CardContent>
            </Card>

            {/* Staff */}
            <Card>
              <CardHeader>
                <CardTitle>Staff</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Referee</label>
                  <p className="text-lg">{match.referee?.name || 'Not assigned'}</p>
                </div>
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Spectators Allowed</label>
                  <Badge className={match.allow_spectators ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'}>
                    {match.allow_spectators ? 'Yes' : 'No'}
                  </Badge>
                </div>
              </CardContent>
            </Card>

            {/* Quick Actions */}
            <Card>
              <CardHeader>
                <CardTitle>Quick Actions</CardTitle>
              </CardHeader>
              <CardContent className="space-y-2">
                <Button
                  variant="outline"
                  className="w-full justify-start"
                  onClick={() => router.get(route('admin.matches.edit', match.id))}
                >
                  <Edit className="w-4 h-4 mr-2" />
                  Edit Match
                </Button>
                <Button
                  variant="outline"
                  className="w-full justify-start text-red-600 hover:text-red-700"
                  onClick={handleDelete}
                >
                  <Trash2 className="w-4 h-4 mr-2" />
                  Delete Match
                </Button>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </AdminLayout>
  );
};

export default ShowMatch; 