import React from 'react';
import { Head } from '@inertiajs/react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Plus, Filter, Download, Eye, Pencil } from 'lucide-react';
import { router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';

interface Match {
  id: number;
  name: string;
  match_code: string;
  match_type: string;
  format: string;
  best_of_games: number;
  tournament_id: number;
  bracket_id: number;
  round: number;
  match_number: number;
  player1_id: number;
  player2_id: number;
  team1_id: number;
  team2_id: number;
  player1_score: number;
  player2_score: number;
  status: string;
  progress_status: string;
  scheduled_time: string;
  started_at: string;
  completed_at: string;
  winner_id: number;
  is_streamed: boolean;
  allow_spectators: boolean;
  tournament?: {
    id: number;
    name: string;
  };
  bracket?: {
    id: number;
    name: string;
  };
  player1?: {
    id: number;
    display_name: string;
  };
  player2?: {
    id: number;
    display_name: string;
  };
  team1?: {
    id: number;
    name: string;
  };
  team2?: {
    id: number;
    name: string;
  };
  winner?: {
    id: number;
    display_name?: string;
    name?: string;
  };
}

interface Props {
  matches: {
    data: Match[];
    current_page: number;
    last_page: number;
    per_page: number;
    total: number;
  };
  filters: {
    search: string;
    status: string;
    progress_status: string;
    match_type: string;
    format: string;
    tournament_id: string;
    bracket_id: string;
    sort_by: string;
    sort_order: string;
  };
}

const MatchesIndex: React.FC<Props> = ({ matches, filters }) => {
  const handleFilterChange = (key: string, value: string) => {
    router.get(route('admin.matches.index'), {
      ...filters,
      [key]: value,
    }, {
      preserveState: true,
      preserveScroll: true,
    });
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'scheduled': return 'bg-blue-100 text-blue-800';
      case 'in_progress': return 'bg-yellow-100 text-yellow-800';
      case 'completed': return 'bg-green-100 text-green-800';
      case 'cancelled': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getProgressStatusColor = (status: string) => {
    switch (status) {
      case 'not_started': return 'bg-gray-100 text-gray-800';
      case 'in_progress': return 'bg-yellow-100 text-yellow-800';
      case 'paused': return 'bg-orange-100 text-orange-800';
      case 'completed': return 'bg-green-100 text-green-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getParticipantName = (match: Match, participantType: 'player1' | 'player2' | 'team1' | 'team2') => {
    if (participantType.startsWith('player')) {
      const player = match[participantType as keyof Match] as any;
      return player?.display_name || 'Unknown Player';
    } else {
      const team = match[participantType as keyof Match] as any;
      return team?.name || 'Unknown Team';
    }
  };

  return (
    <AppLayout breadcrumbs={[{ title: 'Matches', href: '/admin/matches' }]}> 
      <Head title="Matches" />
      <div className="space-y-6 p-4 flex flex-col gap-4">
        {/* Header */}
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold">Matches</h1>
            <p className="text-muted-foreground">Manage tournament matches</p>
          </div>
          <div className="flex items-center gap-2">
            <Button variant="outline" size="sm">
              <Filter className="w-4 h-4 mr-2" />
              Filters
            </Button>
            <Button variant="outline" size="sm">
              <Download className="w-4 h-4 mr-2" />
              Export
            </Button>
            <Button onClick={() => router.get(route('admin.matches.create'))}>
              <Plus className="w-4 h-4 mr-2" />
              Create Match
            </Button>
          </div>
        </div>

        {/* Stats Cards */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium">Total Matches</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{matches.total}</div>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium">Scheduled</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">
                {matches.data.filter(m => m.status === 'scheduled').length}
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium">In Progress</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">
                {matches.data.filter(m => m.status === 'in_progress').length}
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium">Completed</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">
                {matches.data.filter(m => m.status === 'completed').length}
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Matches Table */}
        <Card>
          <CardHeader>
            <CardTitle>All Matches</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead>
                  <tr className="border-b">
                    <th className="text-left p-2 w-2/6">Match</th>
                    <th className="text-left p-2 w-2/6">Participants</th>
                    <th className="text-left p-2 w-1/6">Score</th>
                    <th className="text-left p-2 w-1/6">Actions</th>
                  </tr>
                </thead>
                <tbody>
                  {matches.data.map((match) => (
                    <tr key={match.id} className="border-b hover:bg-muted/50">
                      <td className="p-2">
                        <div>
                          <div className="font-medium">{match.name}</div>
                          <div className="text-sm text-muted-foreground">
                            <span className="font-bold">{match.tournament?.name}</span> <br /> 
                            <span className="text-sm p-2">Round {match.round} Match {match.match_number}</span> <br />
                            <span className="text-sm p-2">Type: {match.match_type.charAt(0).toUpperCase() + match.match_type.slice(1).replace('_', ' ')}</span> <br />
                            <span className="text-sm p-2">Date: {match.scheduled_time ? new Date(match.scheduled_time).toLocaleDateString() : 'TBD'}</span>
                            <br />
                            <span className="text-sm p-2">Status: <span className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-medium ${getStatusColor(match.status)}`}>
                              {match.status.replace('_', ' ')} </span>
                            </span>
                          </div>
                        </div>
                      </td>
                      <td className="p-2">
                        <div className="text-sm flex items-center justify-between px-4 py-4">
                          <div>{match.team1?.name}</div>
                          <div className="text-black font-bold text-lg">vs</div>
                          <div>{match.team2?.name}</div>
                        </div>
                      </td>
                      <td className="p-2">
                        <div className="text-sm font-medium">
                          {match.player1_score} - {match.player2_score}
                        </div>
                      </td>
                      
                                            
                      <td className="p-2">
                        
                        <div className="flex items-center gap-2">
                          <Button
                            variant="ghost"
                            size="sm"
                            className="text-black"
                            onClick={() => router.get(route('admin.matches.show', match.id))}
                          >
                            <Eye className="w-4 h-4" /> View
                          </Button>
                          <Button
                            variant="ghost"
                            size="sm"
                            className="text-black"
                            onClick={() => router.get(route('admin.matches.edit', match.id))}
                          >
                            <Pencil className="w-4 h-4" /> Edit
                          </Button>
                        </div>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>

            {/* Pagination */}
            <div className="flex items-center justify-between mt-4">
              <div className="text-sm text-muted-foreground">
                Showing {((matches.current_page - 1) * matches.per_page) + 1} to{' '}
                {Math.min(matches.current_page * matches.per_page, matches.total)} of{' '}
                {matches.total} results
              </div>
              <div className="flex items-center gap-2">
                <Button
                  variant="outline"
                  size="sm"
                  disabled={matches.current_page === 1}
                  onClick={() => handleFilterChange('page', String(matches.current_page - 1))}
                >
                  Previous
                </Button>
                <span className="text-sm">
                  Page {matches.current_page} of {matches.last_page}
                </span>
                <Button
                  variant="outline"
                  size="sm"
                  disabled={matches.current_page === matches.last_page}
                  onClick={() => handleFilterChange('page', String(matches.current_page + 1))}
                >
                  Next
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
      </AppLayout>
  );
};

export default MatchesIndex; 