import React, { useState } from 'react';
import { Head } from '@inertiajs/react';
import { router } from '@inertiajs/react';
import { route } from 'ziggy-js';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { ArrowLeft, Save, Edit } from 'lucide-react';
import AdminLayout from '@/layouts/admin-layout';
import MatchFormSteps from '@/components/matches/match-form-steps';
import BasicInfoStep from '@/components/matches/form-steps/basic-info-step';
import TournamentBracketStep from '@/components/matches/form-steps/tournament-bracket-step';
import ParticipantsStep from '@/components/matches/form-steps/participants-step';
import StreamingSettingsStep from '@/components/matches/form-steps/streaming-settings-step';
import AppLayout from '@/layouts/app-layout';

interface Props {
  match: {
    id: number;
    name: string;
    match_code: string;
    match_type: string;
    format: string;
    best_of_games: number;
    time_limit_minutes: number | null;
    description: string;
    notes: string;
    tournament_id: number;
    bracket_id: number | null;
    round: number;
    match_number: number;
    scheduled_time: string;
    estimated_duration_minutes: number | null;
    referee_id: number | null;
    allow_spectators: boolean;
    player1_id: number | null;
    player2_id: number | null;
    team1_id: number | null;
    team2_id: number | null;
    player1_score: number;
    player2_score: number;
    is_streamed: boolean;
    stream_url: string;
    twitch_url: string;
    youtube_url: string;
    broadcaster_notes: string;
    status: string;
    progress_status: string;
  };
  tournaments: any[];
  brackets: any[];
  players: any[];
  teams: any[];
  referees: any[];
}

const EditMatch: React.FC<Props> = ({ match, tournaments, brackets, players, teams, referees }) => {
  const [currentStep, setCurrentStep] = useState(0);
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [errors, setErrors] = useState<any>({});
  const [form, setForm] = useState({
    name: match.name,
    match_code: match.match_code,
    match_type: match.match_type,
    format: match.format,
    best_of_games: match.best_of_games,
    time_limit_minutes: match.time_limit_minutes,
    description: match.description,
    notes: match.notes,
    tournament_id: match.tournament_id?.toString() || '',
    bracket_id: match.bracket_id?.toString() || '',
    round: match.round?.toString() || '',
    match_number: match.match_number?.toString() || '',
    scheduled_time: match.scheduled_time ? match.scheduled_time.slice(0, 16) : '',
    estimated_duration_minutes: match.estimated_duration_minutes,
    referee_id: match.referee_id?.toString() || '',
    allow_spectators: match.allow_spectators,
    player1_id: match.player1_id?.toString() || '',
    player2_id: match.player2_id?.toString() || '',
    team1_id: match.team1_id?.toString() || '',
    team2_id: match.team2_id?.toString() || '',
    player1_score: match.player1_score,
    player2_score: match.player2_score,
    is_streamed: match.is_streamed,
    stream_url: match.stream_url,
    twitch_url: match.twitch_url,
    youtube_url: match.youtube_url,
    broadcaster_notes: match.broadcaster_notes,
  });

  const steps = [
    {
      id: 'basic-info',
      title: 'Basic Info',
      description: 'Match name, type, and format',
      icon: <span className="text-lg">📝</span>,
    },
    {
      id: 'tournament-bracket',
      title: 'Tournament & Bracket',
      description: 'Tournament and bracket details',
      icon: <span className="text-lg">🏆</span>,
    },
    {
      id: 'participants',
      title: 'Participants',
      description: 'Players or teams and scores',
      icon: <span className="text-lg">👥</span>,
    },
    {
      id: 'streaming-settings',
      title: 'Streaming & Settings',
      description: 'Streaming options and final settings',
      icon: <span className="text-lg">📺</span>,
    },
  ];

  const handleStepChange = (step: number) => {
    setCurrentStep(step);
  };

  const handleNext = () => {
    if (currentStep < steps.length - 1) {
      setCurrentStep(currentStep + 1);
    }
  };

  const handlePrevious = () => {
    if (currentStep > 0) {
      setCurrentStep(currentStep - 1);
    }
  };

  const handleSubmit = () => {
    setIsSubmitting(true);
    setErrors({});

    router.put(route('admin.matches.update', match.id), form, {
      onSuccess: () => {
        setIsSubmitting(false);
      },
      onError: (errors) => {
        setErrors(errors);
        setIsSubmitting(false);
      },
    });
  };

  const canProceed = () => {
    switch (currentStep) {
      case 0:
        // Basic info: just require name and match type
        return form?.name?.trim() !== '' && form?.match_type !== '';
      case 1:
        // Tournament bracket: require tournament and basic scheduling
        return form?.bracket_id !== '' && String(form?.round).trim() !== '';
      case 2:
        // Participants: require at least one participant on each side
        return (form?.player1_id !== '' || form?.team1_id !== '') && (form?.player2_id !== '' || form?.team2_id !== '');
      case 3:
        // Final step: always allow
        return true;
      default:
        return false;
    }
  };

  const renderCurrentStep = () => {
    switch (currentStep) {
      case 0:
        return (
          <BasicInfoStep
            form={form}
            setForm={setForm}
            errors={errors}
          />
        );
      case 1:
        return (
          <TournamentBracketStep
            form={form}
            setForm={setForm}
            errors={errors}
            tournaments={tournaments}
            brackets={brackets}
          />
        );
      case 2:
        return (
          <ParticipantsStep
            form={form}
            setForm={setForm}
            errors={errors}
            players={players}
            teams={teams}
            isEditing={true}
          />
        );
      case 3:
        return (
          <StreamingSettingsStep
            form={form}
            setForm={setForm}
            errors={errors}
          />
        );
      default:
        return null;
    }
  };

  return (
    <AppLayout breadcrumbs={[{ title: 'Matches', href: '/admin/matches' }]}> 
      <Head title={`Edit Match - ${match.name}`} />
      
      <div className="space-y-6 p-4 flex flex-col gap-4">
        {/* Header */}
        <div className="flex items-center gap-4">
          <Button
            variant="outline"
            size="sm"
            onClick={() => router.get(route('admin.matches.index'))}
          >
            <ArrowLeft className="w-4 h-4 mr-2" />
            Back to Matches
          </Button>
          <div>
            <h1 className="text-3xl font-bold flex items-center gap-2">
              <Edit className="w-8 h-8" />
              Edit Match
            </h1>
            <p className="text-muted-foreground">Update match information and settings</p>
          </div>
        </div>

        {/* Match Status Info */}
        <Card>
          <CardHeader>
            <CardTitle>Current Match Status</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              <div>
                <label className="text-sm font-medium text-muted-foreground">Status</label>
                <p className="text-lg font-semibold capitalize">{match.status.replace('_', ' ')}</p>
              </div>
              <div>
                <label className="text-sm font-medium text-muted-foreground">Progress</label>
                <p className="text-lg font-semibold capitalize">{match.progress_status.replace('_', ' ')}</p>
              </div>
              <div>
                <label className="text-sm font-medium text-muted-foreground">Current Score</label>
                <p className="text-lg font-semibold">{match.player1_score} - {match.player2_score}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Form Steps */}
        <MatchFormSteps
          steps={steps}
          currentStep={currentStep}
          onStepChange={handleStepChange}
          onNext={handleNext}
          onPrevious={handlePrevious}
          onSubmit={handleSubmit}
          isSubmitting={isSubmitting}
          canProceed={canProceed() as boolean}
          isLastStep={currentStep === steps.length - 1}
        >
          {renderCurrentStep()}
        </MatchFormSteps>
      </div>
    </AppLayout>
  );
};

export default EditMatch; 