import React, { useState } from 'react';
import { Head } from '@inertiajs/react';
import { router } from '@inertiajs/react';
import { route } from 'ziggy-js';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { ArrowLeft } from 'lucide-react';
import MatchFormSteps from '@/components/matches/match-form-steps';
import BasicInfoStep from '@/components/matches/form-steps/basic-info-step';
import TournamentBracketStep from '@/components/matches/form-steps/tournament-bracket-step';
import ParticipantsStep from '@/components/matches/form-steps/participants-step';
import StreamingSettingsStep from '@/components/matches/form-steps/streaming-settings-step';
import AppLayout from '@/layouts/app-layout';

interface Props {
  tournaments: any[];
  brackets: any[];
  players: any[];
  teams: any[];
}

const CreateMatch: React.FC<Props> = ({ tournaments, brackets, players, teams }) => {
  const [currentStep, setCurrentStep] = useState(0);
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [errors, setErrors] = useState<any>({});
  const [form, setForm] = useState({
    name: '',
    match_code: '',
    match_type: '',
    format: '',
    best_of_games: 1,
    time_limit_minutes: null,
    description: '',
    notes: '',
    tournament_id: '',
    bracket_id: '',
    round: '',
    match_number: '',
    scheduled_time: '',
    estimated_duration_minutes: null,
    referee_id: '',
    allow_spectators: true,
    player1_id: '',
    player2_id: '',
    team1_id: '',
    team2_id: '',
    is_streamed: false,
    stream_url: '',
    twitch_url: '',
    youtube_url: '',
    broadcaster_notes: '',
  });

  const steps = [
    {
      id: 'basic-info',
      title: 'Basic Info',
      description: 'Match name, type, and format',
      icon: <span className="text-lg">📝</span>,
    },
    {
      id: 'tournament-bracket',
      title: 'Tournament & Bracket',
      description: 'Select tournament and bracket',
      icon: <span className="text-lg">🏆</span>,
    },
    {
      id: 'participants',
      title: 'Participants',
      description: 'Select players or teams',
      icon: <span className="text-lg">👥</span>,
    },
    {
      id: 'streaming-settings',
      title: 'Streaming & Settings',
      description: 'Streaming options and final settings',
      icon: <span className="text-lg">📺</span>,
    },
  ];

  const handleStepChange = (step: number) => {
    setCurrentStep(step);
  };

  const handleNext = () => {
    try {
      if (currentStep < steps.length - 1) {
        setCurrentStep(currentStep + 1);
      }
    } catch (error) {
    }
  };

  const handlePrevious = () => {
    if (currentStep > 0) {
      setCurrentStep(currentStep - 1);
    }
  };

  const handleSubmit = () => {
    setIsSubmitting(true);
    setErrors({});

    router.post(route('admin.matches.store'), form, {
      onSuccess: () => {
        setIsSubmitting(false);
      },
      onError: (errors) => {
        setErrors(errors);
        setIsSubmitting(false);
      },
    });
  };

  const canProceed = () => {
    try {
      switch (currentStep) {
        case 0:
          // Basic info: just require name and match type
          return (form.name && form.name.trim() !== '') && (form.match_type && form.match_type !== '');
        case 1:
          // Tournament bracket: require tournament and basic scheduling
          return (form.tournament_id && form.tournament_id !== '') && (form.round && String(form.round).trim() !== '');
        case 2:
          // Participants: require at least one participant on each side
          return ((form.player1_id && form.player1_id !== '') || (form.team1_id && form.team1_id !== '')) && 
                 ((form.player2_id && form.player2_id !== '') || (form.team2_id && form.team2_id !== ''));
        case 3:
          // Final step: always allow
          return true;
        default:
          return false;
      }
    } catch (error) {
      return false;
    }
  };

  const renderCurrentStep = () => {
    try {
      switch (currentStep) {
        case 0:
          return (
            <BasicInfoStep
              form={form}
              setForm={setForm}
              errors={errors}
            />
          );
        case 1:
          return (
            <TournamentBracketStep
              form={form}
              setForm={setForm}
              errors={errors}
              tournaments={tournaments}
              brackets={brackets}
            />
          );
        case 2:
          return (
            <ParticipantsStep
              form={form}
              setForm={setForm}
              errors={errors}
              players={players}
              teams={teams}
            />
          );
        case 3:
          return (
            <StreamingSettingsStep
              form={form}
              setForm={setForm}
              errors={errors}
            />
          );
        default:
          return <div>Invalid step</div>;
      }
    } catch (error) {
      return <div>Error loading step content</div>;
    }
  };

  return (
    <AppLayout breadcrumbs={[{ title: 'Matches', href: '/admin/matches' }]}> 
      <Head title="Create Match" />
      
      <div className="space-y-6 p-4 flex flex-col gap-4">
        {/* Header */}
        <div className="flex items-center gap-4">
          <Button
            variant="outline"
            size="sm"
            onClick={() => router.get(route('admin.matches.index'))}
          >
            <ArrowLeft className="w-4 h-4 mr-2" />
            Back to Matches
          </Button>
          <div>
            <h1 className="text-3xl font-bold">Create Match</h1>
            <p className="text-muted-foreground">Create a new tournament match</p>
          </div>
        </div>

        {/* Form Steps */}
        <MatchFormSteps
          steps={steps}
          currentStep={currentStep}
          onStepChange={handleStepChange}
          onNext={handleNext}
          onPrevious={handlePrevious}
          onSubmit={handleSubmit}
          isSubmitting={isSubmitting}
          canProceed={Boolean(canProceed())}
          isLastStep={currentStep === steps.length - 1}
        >
          {renderCurrentStep()}
        </MatchFormSteps>
      </div>
    </AppLayout>
  );
};

export default CreateMatch; 