import React, { useState } from 'react';
import { Head, Link, router } from '@inertiajs/react';
import { route } from 'ziggy-js';
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Plus, Search, Edit, Trash2, Play } from 'lucide-react';
import { toast } from 'react-hot-toast';
import Swal from 'sweetalert2';

interface VlogVideo {
  id: number;
  title: string;
  description?: string;
  platform: string;
  platform_icon: string;
  video_url: string;
  status: string;
  category: string;
  created_at: string;
}

interface Props {
  vlogVideos: VlogVideo[];
}

const VlogVideoIndex = ({ vlogVideos }: Props) => {
  const [searchTerm, setSearchTerm] = useState('');

  const filteredVideos = vlogVideos.filter(video =>
    video.title.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const handleDelete = (id: number) => {
    Swal.fire({
      title: 'Are you sure?',
      text: 'You won\'t be able to revert this!',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonColor: '#3085d6',
      cancelButtonColor: '#d33',
      confirmButtonText: 'Yes, delete it!'
    }).then((result) => {
      if (result.isConfirmed) {
        router.delete(route('admin.frontend.vlog.destroy', id), {
          onSuccess: (response: any) => [
            toast.success(response.message),
            router.visit(route('admin.frontend.vlog.index'))
          ],
          onError: (response: any) => [
            toast.error(response.message),
            router.visit(route('admin.frontend.vlog.index'))
          ]
        });
      }
    });
  };

  return (
    <AppLayout>
      <Head title="Vlog Videos" />
      
      <div className="container mx-auto p-6">
        <div className="flex justify-between items-center mb-6">
          <div>
            <h1 className="text-2xl font-bold">Vlog Videos</h1>
            <p className="text-muted-foreground">Manage vlog videos for the frontend</p>
          </div>
          <Link href={route('admin.frontend.vlog.create')}>
            <Button style={{ backgroundColor: '#256630' }} className="hover:opacity-90">
              <Plus className="h-4 w-4 mr-2" />
              Add Video
            </Button>
          </Link>
        </div>

        <Card className="mb-6">
          <CardContent className="p-4">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
              <Input
                placeholder="Search videos..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10"
              />
            </div>
          </CardContent>
        </Card>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {filteredVideos.map((video) => (
            <Card key={video.id}>
              <CardHeader>
                <CardTitle className="text-lg">{video.title}</CardTitle>
                {video.description && (
                  <p className="text-sm text-gray-600">{video.description}</p>
                )}
              </CardHeader>
              <CardContent>
                <div className="flex items-center gap-2 mb-3">
                  <Badge variant={video.status === 'published' ? 'default' : 'secondary'}>
                    {video.status}
                  </Badge>
                  <Badge variant="outline">{video.category}</Badge>
                </div>
                <div className="flex items-center gap-2">
                  <Link href={route('admin.frontend.vlog.edit', video.id)}>
                    <Button variant="outline" size="sm">
                      <Edit className="h-4 w-4" />
                    </Button>
                  </Link>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => window.open(video.video_url, '_blank')}
                  >
                    <Play className="h-4 w-4" />
                  </Button>
                  <Button
                    variant="destructive"
                    size="sm"
                    onClick={() => handleDelete(video.id)}
                  >
                    <Trash2 className="h-4 w-4" />
                  </Button>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>

        {filteredVideos.length === 0 && (
          <Card>
            <CardContent className="p-8 text-center">
              <Play className="h-12 w-12 mx-auto mb-4 text-gray-300" />
              <h3 className="text-lg font-semibold mb-2">No videos found</h3>
              <p className="text-sm text-gray-500 mb-4">
                {searchTerm ? 'Try adjusting your search terms.' : 'Get started by adding your first video.'}
              </p>
              {!searchTerm && (
                <Link href={route('admin.frontend.vlog.create')}>
                  <Button>
                    <Plus className="h-4 w-4 mr-2" />
                    Add Video
                  </Button>
                </Link>
              )}
            </CardContent>
          </Card>
        )}
      </div>
    </AppLayout>
  );
};

export default VlogVideoIndex; 