import React, { useState } from 'react';
import { Head, router } from '@inertiajs/react';
import { route } from 'ziggy-js';
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Switch } from '@/components/ui/switch';
import { ArrowLeft, Save } from 'lucide-react';
import { toast } from 'react-hot-toast';
import { TournamentHighlight } from '@/types';

interface Props {
  highlight?: TournamentHighlight;
  isEdit?: boolean;
}

const TournamentHighlightForm = ({ highlight, isEdit = false }: Props) => {
  const [form, setForm] = useState<TournamentHighlight>({
    icon: highlight?.icon || '🏆',
    title: highlight?.title || '',
    description: highlight?.description || '',
    value: highlight?.value || '',
    color: highlight?.color || 'text-primary',
    sort_order: highlight?.sort_order || 0,
    active: highlight?.active ?? true,
    status: highlight?.status || 'draft'
  });

  const [loading, setLoading] = useState(false);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    const url = isEdit 
      ? route('admin.frontend.tournament-highlights.update', highlight?.id)
      : route('admin.frontend.tournament-highlights.store');
    
    const method = isEdit ? 'put' : 'post';

    router[method](url, form as any, {
      onSuccess: () => {
        toast.success(`Tournament highlight ${isEdit ? 'updated' : 'created'} successfully`);
        router.get(route('admin.frontend.tournament-highlights.index'));
      },
      onError: (errors) => {
        toast.error('Please check the form for errors');
      },
      onFinish: () => setLoading(false)
    });
  };

  const handleInputChange = (field: keyof TournamentHighlight, value: any) => {
    setForm(prev => ({ ...prev, [field]: value }));
  };

  const colorOptions = [
    { value: 'text-primary', label: 'Primary', preview: '#256630' },
    { value: 'text-secondary', label: 'Secondary', preview: '#6b7280' },
    { value: 'text-green-600', label: 'Green', preview: '#059669' },
    { value: 'text-blue-600', label: 'Blue', preview: '#2563eb' },
    { value: 'text-red-600', label: 'Red', preview: '#dc2626' },
    { value: 'text-yellow-600', label: 'Yellow', preview: '#d97706' },
    { value: 'text-purple-600', label: 'Purple', preview: '#9333ea' },
    { value: 'text-pink-600', label: 'Pink', preview: '#db2777' },
    { value: 'text-indigo-600', label: 'Indigo', preview: '#4f46e5' },
    { value: 'text-orange-600', label: 'Orange', preview: '#ea580c' },
  ];

  const iconOptions = [
    '🏆', '⚽', '💰', '🎯', '👥', '🏅', '🥇', '🥈', '🥉', '⭐', '🔥', '💎', '🚀', '🎉', '⚡'
  ];

  return (
    <AppLayout>
      <Head title={`${isEdit ? 'Edit' : 'Create'} Tournament Highlight`} />
      
      <div className="space-y-6">
        <div className="flex items-center space-x-4">
          <Button variant="ghost" onClick={() => router.get(route('admin.frontend.tournament-highlights.index'))}>
            <ArrowLeft className="h-4 w-4 mr-2" />
            Back
          </Button>
          <div>
            <h1 className="text-2xl font-bold">{isEdit ? 'Edit' : 'Create'} Tournament Highlight</h1>
            <p className="text-muted-foreground">
              {isEdit ? 'Update tournament highlight details' : 'Add a new tournament highlight'}
            </p>
          </div>
        </div>

        <form onSubmit={handleSubmit} className="space-y-6">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            {/* Basic Information */}
            <Card>
              <CardHeader>
                <CardTitle>Basic Information</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="icon">Icon</Label>
                  <div className="grid grid-cols-8 gap-2">
                    {iconOptions.map((icon) => (
                      <button
                        key={icon}
                        type="button"
                        onClick={() => handleInputChange('icon', icon)}
                        className={`p-2 rounded border text-lg hover:bg-gray-50 transition-colors ${
                          form.icon === icon ? 'border-primary bg-primary/10' : 'border-gray-200'
                        }`}
                      >
                        {icon}
                      </button>
                    ))}
                  </div>
                  <Input
                    id="icon"
                    value={form.icon}
                    onChange={(e) => handleInputChange('icon', e.target.value)}
                    placeholder="Or enter custom icon"
                    className="mt-2"
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="title">Title</Label>
                  <Input
                    id="title"
                    value={form.title}
                    onChange={(e) => handleInputChange('title', e.target.value)}
                    placeholder="e.g., 4th Edition"
                    required
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="description">Description</Label>
                  <Textarea
                    id="description"
                    value={form.description}
                    onChange={(e) => handleInputChange('description', e.target.value)}
                    placeholder="e.g., The biggest tournament yet"
                    rows={3}
                    required
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="value">Value</Label>
                  <Input
                    id="value"
                    value={form.value}
                    onChange={(e) => handleInputChange('value', e.target.value)}
                    placeholder="e.g., 2024, 32+, €50K"
                    required
                  />
                </div>
              </CardContent>
            </Card>

            {/* Styling and Settings */}
            <Card>
              <CardHeader>
                <CardTitle>Styling & Settings</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="color">Color</Label>
                  <div className="grid grid-cols-5 gap-2">
                    {colorOptions.map((color) => (
                      <button
                        key={color.value}
                        type="button"
                        onClick={() => handleInputChange('color', color.value)}
                        className={`p-2 rounded border text-sm hover:bg-gray-50 transition-colors ${
                          form.color === color.value ? 'border-primary bg-primary/10' : 'border-gray-200'
                        }`}
                        title={color.label}
                      >
                        <div 
                          className="w-4 h-4 rounded-full mx-auto mb-1"
                          style={{ backgroundColor: color.preview }}
                        />
                        <span className="text-xs">{color.label}</span>
                      </button>
                    ))}
                  </div>
                  <Input
                    value={form.color}
                    onChange={(e) => handleInputChange('color', e.target.value)}
                    placeholder="Or enter custom color class"
                    className="mt-2"
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="sort_order">Sort Order</Label>
                  <Input
                    id="sort_order"
                    type="number"
                    value={form.sort_order}
                    onChange={(e) => handleInputChange('sort_order', parseInt(e.target.value) || 0)}
                    placeholder="0"
                    min="0"
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="status">Status</Label>
                  <Select value={form.status} onValueChange={(value) => handleInputChange('status', value)}>
                    <SelectTrigger>
                      <SelectValue placeholder="Select status" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="draft">Draft</SelectItem>
                      <SelectItem value="published">Published</SelectItem>
                      <SelectItem value="archived">Archived</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                <div className="flex items-center space-x-2">
                  <Switch
                    id="active"
                    checked={form.active}
                    onCheckedChange={(checked) => handleInputChange('active', checked)}
                  />
                  <Label htmlFor="active">Active</Label>
                </div>
              </CardContent>
            </Card>
          </div>

          {/* Preview */}
          <Card>
            <CardHeader>
              <CardTitle>Preview</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="bg-gradient-to-br from-primary/5 to-secondary/5 p-6 rounded-lg">
                <div className="max-w-sm mx-auto">
                  <div className="bg-white/80 backdrop-blur-sm rounded-lg p-6 text-center shadow-lg">
                    <div className="mx-auto mb-4 p-3 rounded-full bg-white/50 w-fit">
                      <div className="text-2xl">{form.icon}</div>
                    </div>
                    <h3 className="text-xl font-semibold text-foreground mb-2">{form.title || 'Title'}</h3>
                    <p className="text-muted-foreground mb-3">{form.description || 'Description'}</p>
                    <div className={`text-3xl font-bold ${form.color}`}>
                      {form.value || 'Value'}
                    </div>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Actions */}
          <div className="flex justify-end space-x-4">
            <Button
              type="button"
              variant="outline"
              onClick={() => router.get(route('admin.frontend.tournament-highlights.index'))}
            >
              Cancel
            </Button>
            <Button
              type="submit"
              disabled={loading}
              style={{ backgroundColor: '#256630' }}
              className="hover:opacity-90"
            >
              <Save className="h-4 w-4 mr-2" />
              {loading ? 'Saving...' : (isEdit ? 'Update' : 'Create')}
            </Button>
          </div>
        </form>
      </div>
    </AppLayout>
  );
};

export default TournamentHighlightForm; 