import React from 'react';
import { Head, Link, router } from '@inertiajs/react';
import { route } from 'ziggy-js';
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Plus, Edit, Trash2, Star, StarOff } from 'lucide-react';
import { toast } from 'react-hot-toast';
import Swal from 'sweetalert2';
import { Testimonial } from '@/types';

interface Props {
  testimonials: {
    data: Testimonial[];
    links: any;
    meta: any;
  };
  filters: {
    search?: string;
    rating?: string;
    status?: string;
    featured?: string;
  };
}

const TestimonialsIndex = ({ testimonials, filters }: Props) => {
  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    const formData = new FormData(e.target as HTMLFormElement);
    router.get(route('admin.frontend.testimonials.index'), Object.fromEntries(formData), {
      preserveState: true,
    });
  };

  const handleDelete = (testimonial: Testimonial) => {
    Swal.fire({
      title: 'Are you sure?',
      text: 'This action cannot be undone!',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonColor: '#c22b2b',
      cancelButtonColor: '#256630',
      confirmButtonText: 'Yes, delete it!'
    }).then((result) => {
      if (result.isConfirmed) {
        router.delete(route('admin.frontend.testimonials.destroy', testimonial.id), {
          onSuccess: () => {
            toast.success('Testimonial deleted successfully');
            Swal.fire('Deleted!', 'The testimonial has been deleted.', 'success');
          },
          onError: () => {
            toast.error('Failed to delete testimonial');
          }
        });
      }
    });
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'published': return 'bg-green-100 text-green-800';
      case 'draft': return 'bg-yellow-100 text-yellow-800';
      case 'archived': return 'bg-gray-100 text-gray-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const renderStars = (rating: number) => {
    return Array.from({ length: 5 }, (_, i) => (
      <Star
        key={i}
        className={`h-4 w-4 ${i < rating ? 'text-yellow-500 fill-current' : 'text-gray-300'}`}
      />
    ));
  };

  return (
    <AppLayout>
      <Head title="Testimonials" />
      
      <div className="space-y-6">
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-2xl font-bold">Testimonials</h1>
            <p className="text-muted-foreground">Manage testimonials for the frontend</p>
          </div>
          <Link href={route('admin.frontend.testimonials.create')}>
            <Button style={{ backgroundColor: '#256630' }} className="hover:opacity-90">
              <Plus className="h-4 w-4 mr-2" />
              Add Testimonial
            </Button>
          </Link>
        </div>

        <Card>
          <CardHeader>
            <CardTitle>Filters</CardTitle>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleSearch} className="flex gap-4">
              <div className="flex-1">
                <Input
                  name="search"
                  placeholder="Search testimonials..."
                  defaultValue={filters.search}
                />
              </div>
              <select
                name="rating"
                className="px-3 py-2 border border-input rounded-md"
                defaultValue={filters.rating || 'all'}
              >
                <option value="all">All Ratings</option>
                <option value="5">5 Stars</option>
                <option value="4">4+ Stars</option>
                <option value="3">3+ Stars</option>
                <option value="2">2+ Stars</option>
                <option value="1">1+ Stars</option>
              </select>
              <select
                name="status"
                className="px-3 py-2 border border-input rounded-md"
                defaultValue={filters.status || 'all'}
              >
                <option value="all">All Status</option>
                <option value="draft">Draft</option>
                <option value="published">Published</option>
                <option value="archived">Archived</option>
              </select>
              <select
                name="featured"
                className="px-3 py-2 border border-input rounded-md"
                defaultValue={filters.featured || 'all'}
              >
                <option value="all">All</option>
                <option value="true">Featured</option>
                <option value="false">Not Featured</option>
              </select>
              <Button type="submit" style={{ backgroundColor: '#256630' }} className="hover:opacity-90">
                Filter
              </Button>
            </form>
          </CardContent>
        </Card>

        <div className="grid gap-6">
          {testimonials.data.map((testimonial) => (
            <Card key={testimonial.id} className="hover:shadow-md transition-shadow">
              <CardContent className="p-6">
                <div className="flex gap-6">
                  <div className="flex-shrink-0">
                    {testimonial.image ? (
                      <img
                        src={testimonial.image}
                        alt={testimonial.name}
                        className="w-16 h-16 rounded-full object-cover"
                      />
                    ) : (
                      <div className="w-16 h-16 rounded-full bg-gray-200 flex items-center justify-center">
                        <span className="text-gray-500 text-xl font-semibold">
                          {testimonial.name.charAt(0)}
                        </span>
                      </div>
                    )}
                  </div>
                  
                  <div className="flex-1">
                    <div className="flex justify-between items-start mb-3">
                      <div>
                        <div className="flex items-center gap-2 mb-1">
                          <h3 className="text-lg font-semibold">{testimonial.name}</h3>
                          {testimonial.featured && <Star className="h-4 w-4 text-yellow-500 fill-current" />}
                          {!testimonial.active && <span className="text-xs bg-red-100 text-red-800 px-2 py-0.5 rounded">Inactive</span>}
                        </div>
                        <p className="text-sm text-muted-foreground">
                          {testimonial.role}
                          {testimonial.team && ` • ${testimonial.team}`}
                        </p>
                        <div className="flex items-center gap-1 mt-1">
                          {renderStars(testimonial.rating)}
                          <span className="text-sm text-muted-foreground ml-2">
                            ({testimonial.rating}/5)
                          </span>
                        </div>
                      </div>
                      
                      <div className="flex items-center space-x-2">
                        <Link href={route('admin.frontend.testimonials.edit', testimonial.id)}>
                          <Button variant="ghost" size="sm">
                            <Edit className="h-4 w-4" />
                          </Button>
                        </Link>
                        <Button
                          variant="ghost"
                          size="sm"
                          onClick={() => handleDelete(testimonial)}
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      </div>
                    </div>
                    
                    <blockquote className="text-sm italic mb-3 border-l-4 border-gray-200 pl-4">
                      "{testimonial.quote}"
                    </blockquote>
                    
                    <div className="flex items-center gap-4 text-xs text-muted-foreground">
                      {testimonial.achievement && (
                        <span>🏆 {testimonial.achievement}</span>
                      )}
                      <span>📅 {testimonial.year}</span>
                      <Badge className={getStatusColor(testimonial.status)}>
                        {testimonial.status}
                      </Badge>
                      <span>Order: #{testimonial.sort_order}</span>
                      <span>{new Date(testimonial.created_at).toLocaleDateString()}</span>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>

        {testimonials.data.length === 0 && (
          <Card>
            <CardContent className="p-6 text-center">
              <p className="text-muted-foreground">No testimonials found</p>
            </CardContent>
          </Card>
        )}
      </div>
    </AppLayout>
  );
};

export default TestimonialsIndex; 