import React, { useState } from 'react';
import { Head, router } from '@inertiajs/react';
import { route } from 'ziggy-js';
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Switch } from '@/components/ui/switch';
import { ArrowLeft, Save, Star } from 'lucide-react';
import MediaManager from '@/components/MediaManager';
import { toast } from 'react-hot-toast';
import { Testimonial } from '@/types';

interface Props {
  testimonial?: Testimonial;
  isEdit?: boolean;
}

const TestimonialForm = ({ testimonial, isEdit = false }: Props) => {
  const [form, setForm] = useState<Testimonial>({
    name: testimonial?.name || '',
    role: testimonial?.role || '',
    team: testimonial?.team || '',
    image: testimonial?.image || '',
    rating: testimonial?.rating || 5,
    quote: testimonial?.quote || '',
    achievement: testimonial?.achievement || '',
    year: testimonial?.year || new Date().getFullYear(),
    sort_order: testimonial?.sort_order || 0,
    status: testimonial?.status || 'draft',
    active: testimonial?.active ?? true,
    featured: testimonial?.featured || false
  });

  const [loading, setLoading] = useState(false);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    const url = isEdit 
      ? route('admin.frontend.testimonials.update', testimonial?.id)
      : route('admin.frontend.testimonials.store');
    
    const method = isEdit ? 'put' : 'post';

    router[method](url, form as any, {
      onSuccess: () => {
        toast.success(`Testimonial ${isEdit ? 'updated' : 'created'} successfully`);
        router.get(route('admin.frontend.testimonials.index'));
      },
      onError: (errors) => {
        toast.error('Please check the form for errors');
      },
      onFinish: () => setLoading(false)
    });
  };

  const handleInputChange = (field: keyof Testimonial, value: any) => {
    setForm(prev => ({ ...prev, [field]: value }));
  };

  const renderRatingSelector = () => {
    return (
      <div className="flex gap-1">
        {[1, 2, 3, 4, 5].map((rating) => (
          <button
            key={rating}
            type="button"
            onClick={() => handleInputChange('rating', rating)}
            className={`p-1 rounded transition-colors ${
              rating <= form.rating ? 'text-yellow-500' : 'text-gray-300 hover:text-yellow-300'
            }`}
          >
            <Star className="h-6 w-6 fill-current" />
          </button>
        ))}
        <span className="ml-2 text-sm text-muted-foreground">
          {form.rating}/5 stars
        </span>
      </div>
    );
  };

  return (
    <AppLayout>
      <Head title={`${isEdit ? 'Edit' : 'Create'} Testimonial`} />
      
      <div className="space-y-6">
        <div className="flex items-center space-x-4">
          <Button variant="ghost" onClick={() => router.get(route('admin.frontend.testimonials.index'))}>
            <ArrowLeft className="h-4 w-4 mr-2" />
            Back
          </Button>
          <div>
            <h1 className="text-2xl font-bold">{isEdit ? 'Edit' : 'Create'} Testimonial</h1>
            <p className="text-muted-foreground">{isEdit ? 'Update the' : 'Add a new'} testimonial</p>
          </div>
        </div>

        <form onSubmit={handleSubmit}>
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            <div className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle>Person Information</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <Label htmlFor="name">Full Name *</Label>
                    <Input
                      id="name"
                      value={form.name}
                      onChange={(e) => handleInputChange('name', e.target.value)}
                      placeholder="Enter full name"
                      required
                    />
                  </div>

                  <div>
                    <Label htmlFor="role">Role/Position *</Label>
                    <Input
                      id="role"
                      value={form.role}
                      onChange={(e) => handleInputChange('role', e.target.value)}
                      placeholder="e.g., Team Captain, Player, Coach"
                      required
                    />
                  </div>

                  <div>
                    <Label htmlFor="team">Team/Organization</Label>
                    <Input
                      id="team"
                      value={form.team}
                      onChange={(e) => handleInputChange('team', e.target.value)}
                      placeholder="Team or organization name"
                    />
                  </div>

                  <div>
                    <Label>Profile Image</Label>
                    <MediaManager
                      trigger={
                        <Button type="button" variant="outline" className="w-full">
                          {form.image ? 'Change Image' : 'Select Image'}
                        </Button>
                      }
                      onSelect={(files) => {
                        const file = Array.isArray(files) ? files[0] : files;
                        if (file) {
                          handleInputChange('image', file.url);
                        }
                      }}
                      acceptedTypes={['image/*']}
                    />
                    {form.image && (
                      <div className="mt-2">
                        <img 
                          src={form.image} 
                          alt="Profile preview" 
                          className="w-20 h-20 rounded-full object-cover"
                        />
                      </div>
                    )}
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>Testimonial Content</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <Label htmlFor="quote">Quote/Testimonial *</Label>
                    <Textarea
                      id="quote"
                      value={form.quote}
                      onChange={(e) => handleInputChange('quote', e.target.value)}
                      placeholder="Enter the testimonial quote..."
                      rows={4}
                      required
                    />
                  </div>

                  <div>
                    <Label htmlFor="rating">Rating *</Label>
                    <div className="mt-2">
                      {renderRatingSelector()}
                    </div>
                  </div>

                  <div>
                    <Label htmlFor="achievement">Achievement/Context</Label>
                    <Input
                      id="achievement"
                      value={form.achievement}
                      onChange={(e) => handleInputChange('achievement', e.target.value)}
                      placeholder="e.g., Tournament Winner, MVP"
                    />
                  </div>

                  <div>
                    <Label htmlFor="year">Year</Label>
                    <Input
                      id="year"
                      type="number"
                      value={form.year}
                      onChange={(e) => handleInputChange('year', parseInt(e.target.value) || new Date().getFullYear())}
                      min="2020"
                      max={new Date().getFullYear() + 1}
                    />
                  </div>
                </CardContent>
              </Card>
            </div>

            <div className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle>Display Settings</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <Label htmlFor="sort_order">Sort Order</Label>
                    <Input
                      id="sort_order"
                      type="number"
                      value={form.sort_order}
                      onChange={(e) => handleInputChange('sort_order', parseInt(e.target.value) || 0)}
                      placeholder="0"
                      min="0"
                    />
                    <p className="text-xs text-muted-foreground mt-1">
                      Lower numbers appear first
                    </p>
                  </div>

                  <div>
                    <Label htmlFor="status">Status *</Label>
                    <Select value={form.status} onValueChange={(value) => handleInputChange('status', value)}>
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="draft">Draft</SelectItem>
                        <SelectItem value="published">Published</SelectItem>
                        <SelectItem value="archived">Archived</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>

                  <div className="space-y-4">
                    <div className="flex items-center space-x-2">
                      <Switch
                        id="active"
                        checked={form.active}
                        onCheckedChange={(checked) => handleInputChange('active', checked)}
                      />
                      <Label htmlFor="active">Active</Label>
                    </div>

                    <div className="flex items-center space-x-2">
                      <Switch
                        id="featured"
                        checked={form.featured}
                        onCheckedChange={(checked) => handleInputChange('featured', checked)}
                      />
                      <Label htmlFor="featured">Featured Testimonial</Label>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>Preview</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="bg-gray-50 p-4 rounded-lg">
                    <div className="flex gap-4">
                      <div className="flex-shrink-0">
                        {form.image ? (
                          <img
                            src={form.image}
                            alt="Preview"
                            className="w-12 h-12 rounded-full object-cover"
                          />
                        ) : (
                          <div className="w-12 h-12 rounded-full bg-gray-200 flex items-center justify-center">
                            <span className="text-gray-500 text-sm font-semibold">
                              {form.name.charAt(0) || '?'}
                            </span>
                          </div>
                        )}
                      </div>
                      <div className="flex-1">
                        <div className="flex items-center gap-1 mb-1">
                          {[1, 2, 3, 4, 5].map((star) => (
                            <Star
                              key={star}
                              className={`h-3 w-3 ${
                                star <= form.rating ? 'text-yellow-500 fill-current' : 'text-gray-300'
                              }`}
                            />
                          ))}
                        </div>
                        <p className="text-sm italic mb-2">
                          "{form.quote || 'Your testimonial quote will appear here...'}"
                        </p>
                        <p className="text-xs text-muted-foreground">
                          <strong>{form.name || 'Name'}</strong>
                          {form.role && `, ${form.role}`}
                          {form.team && ` • ${form.team}`}
                          {form.achievement && ` • ${form.achievement}`}
                        </p>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <div className="flex gap-4">
                <Button 
                  type="button" 
                  variant="outline" 
                  onClick={() => router.get(route('admin.frontend.testimonials.index'))}
                  disabled={loading}
                >
                  Cancel
                </Button>
                <Button 
                  type="submit" 
                  style={{ backgroundColor: '#256630' }}
                  className="hover:opacity-90 flex-1"
                  disabled={loading}
                >
                  <Save className="h-4 w-4 mr-2" />
                  {loading ? 'Saving...' : isEdit ? 'Update Testimonial' : 'Create Testimonial'}
                </Button>
              </div>
            </div>
          </div>
        </form>
      </div>
    </AppLayout>
  );
};

export default TestimonialForm; 