import React from 'react';
import { Head, Link, router } from '@inertiajs/react';
import { route } from 'ziggy-js';
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Plus, Edit, Trash2, Eye, EyeOff, ExternalLink, Mail, Phone } from 'lucide-react';
import { toast } from 'react-hot-toast';
import Swal from 'sweetalert2';
import { Sponsor } from '@/types';

interface Props {
  sponsors: {
    data: Sponsor[];
    links: any;
    meta: any;
  };
  filters: {
    search?: string;
    category?: string;
    tier?: string;
    status?: string;
  };
  categories: Record<string, string>;
  tiers: Record<string, string>;
}

const SponsorsIndex = ({ sponsors, filters, categories, tiers }: Props) => {
  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    const formData = new FormData(e.target as HTMLFormElement);
    router.get(route('admin.frontend.sponsors.index'), Object.fromEntries(formData), {
      preserveState: true,
    });
  };

  const handleDelete = (sponsor: Sponsor) => {
    Swal.fire({
      title: 'Are you sure?',
      text: 'This action cannot be undone!',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonColor: '#c22b2b',
      cancelButtonColor: '#256630',
      confirmButtonText: 'Yes, delete it!'
    }).then((result) => {
      if (result.isConfirmed) {
        router.delete(route('admin.frontend.sponsors.destroy', sponsor.id), {
          onSuccess: () => {
            toast.success('Sponsor deleted successfully');
            Swal.fire('Deleted!', 'The sponsor has been deleted.', 'success');
          },
          onError: () => {
            toast.error('Failed to delete sponsor');
          }
        });
      }
    });
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'published': return 'bg-green-100 text-green-800';
      case 'draft': return 'bg-yellow-100 text-yellow-800';
      case 'archived': return 'bg-gray-100 text-gray-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getTierColor = (tier: string) => {
    switch (tier) {
      case 'platinum': return 'bg-purple-100 text-purple-800 border-purple-200';
      case 'gold': return 'bg-yellow-100 text-yellow-800 border-yellow-200';
      case 'silver': return 'bg-gray-100 text-gray-800 border-gray-200';
      case 'bronze': return 'bg-orange-100 text-orange-800 border-orange-200';
      default: return 'bg-gray-100 text-gray-800 border-gray-200';
    }
  };

  return (
    <AppLayout>
      <Head title="Sponsors" />
      
      <div className="space-y-6">
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-2xl font-bold">Sponsors</h1>
            <p className="text-muted-foreground">Manage sponsors for the frontend</p>
          </div>
          <Link href={route('admin.frontend.sponsors.create')}>
            <Button style={{ backgroundColor: '#256630' }} className="hover:opacity-90">
              <Plus className="h-4 w-4 mr-2" />
              Add Sponsor
            </Button>
          </Link>
        </div>

        <Card>
          <CardHeader>
            <CardTitle>Filters</CardTitle>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleSearch} className="flex gap-4">
              <div className="flex-1">
                <Input
                  name="search"
                  placeholder="Search sponsors..."
                  defaultValue={filters.search}
                />
              </div>
              <select
                name="category"
                className="px-3 py-2 border border-input rounded-md"
                defaultValue={filters.category || 'all'}
              >
                <option value="all">All Categories</option>
                {Object.entries(categories).map(([key, label]) => (
                  <option key={key} value={key}>{label}</option>
                ))}
              </select>
              <select
                name="tier"
                className="px-3 py-2 border border-input rounded-md"
                defaultValue={filters.tier || 'all'}
              >
                <option value="all">All Tiers</option>
                {Object.entries(tiers).map(([key, label]) => (
                  <option key={key} value={key}>{label}</option>
                ))}
              </select>
              <select
                name="status"
                className="px-3 py-2 border border-input rounded-md"
                defaultValue={filters.status || 'all'}
              >
                <option value="all">All Status</option>
                <option value="draft">Draft</option>
                <option value="published">Published</option>
                <option value="archived">Archived</option>
              </select>
              <Button type="submit" style={{ backgroundColor: '#256630' }} className="hover:opacity-90">
                Filter
              </Button>
            </form>
          </CardContent>
        </Card>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {sponsors.data.map((sponsor) => (
            <Card key={sponsor.id} className="hover:shadow-md transition-shadow">
              <CardContent className="p-6">
                <div className="space-y-4">
                  <div className="flex items-start gap-4">
                    <div className="flex-shrink-0">
                      {sponsor.logo ? (
                        <img
                          src={sponsor.logo}
                          alt={sponsor.name}
                          className="w-16 h-16 object-contain rounded border"
                        />
                      ) : (
                        <div className="w-16 h-16 bg-gray-100 rounded border flex items-center justify-center">
                          <span className="text-gray-400 text-xs">No Logo</span>
                        </div>
                      )}
                    </div>
                    
                    <div className="flex-1 min-w-0">
                      <div className="flex items-center gap-2 mb-1">
                        <h3 className="font-semibold truncate">{sponsor.name}</h3>
                        {!sponsor.active && (
                          <span className="text-xs bg-red-100 text-red-800 px-2 py-0.5 rounded">
                            Inactive
                          </span>
                        )}
                      </div>
                      
                      <div className="flex flex-wrap gap-1 mb-2">
                        <Badge 
                          className={`text-xs border ${getTierColor(sponsor.tier)}`}
                          variant="outline"
                        >
                          {tiers[sponsor.tier] || sponsor.tier}
                        </Badge>
                        <Badge variant="secondary" className="text-xs">
                          {categories[sponsor.category] || sponsor.category}
                        </Badge>
                        <Badge className={`text-xs ${getStatusColor(sponsor.status)}`}>
                          {sponsor.status}
                        </Badge>
                      </div>
                      
                      {sponsor.description && (
                        <p className="text-sm text-muted-foreground line-clamp-2 mb-2">
                          {sponsor.description}
                        </p>
                      )}
                      
                      <div className="flex items-center gap-3 text-xs text-muted-foreground">
                        {sponsor.website_url && (
                          <a
                            href={sponsor.website_url}
                            target="_blank"
                            rel="noopener noreferrer"
                            className="flex items-center gap-1 hover:text-blue-600"
                          >
                            <ExternalLink className="h-3 w-3" />
                            Website
                          </a>
                        )}
                        {sponsor.contact_email && (
                          <div className="flex items-center gap-1">
                            <Mail className="h-3 w-3" />
                            <span className="truncate max-w-24">{sponsor.contact_email}</span>
                          </div>
                        )}
                        {sponsor.contact_phone && (
                          <div className="flex items-center gap-1">
                            <Phone className="h-3 w-3" />
                            <span>{sponsor.contact_phone}</span>
                          </div>
                        )}
                      </div>
                    </div>
                  </div>
                  
                  <div className="flex justify-between items-center pt-2 border-t">
                    <div className="text-xs text-muted-foreground">
                      Order: #{sponsor.sort_order} • {new Date(sponsor.created_at).toLocaleDateString()}
                    </div>
                    <div className="flex items-center space-x-2">
                      <Link href={route('admin.frontend.sponsors.edit', sponsor.id)}>
                        <Button variant="ghost" size="sm">
                          <Edit className="h-4 w-4" />
                        </Button>
                      </Link>
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => handleDelete(sponsor)}
                      >
                        <Trash2 className="h-4 w-4" />
                      </Button>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>

        {sponsors.data.length === 0 && (
          <Card>
            <CardContent className="p-6 text-center">
              <p className="text-muted-foreground">No sponsors found</p>
            </CardContent>
          </Card>
        )}
      </div>
    </AppLayout>
  );
};

export default SponsorsIndex; 