import React, { useState } from 'react';
import { Head, router } from '@inertiajs/react';
import { route } from 'ziggy-js';
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Switch } from '@/components/ui/switch';
import { ArrowLeft, Save, ExternalLink } from 'lucide-react';
import MediaManager from '@/components/MediaManager';
import { toast } from 'react-hot-toast';
import { Sponsor } from '@/types';

interface Props {
  sponsor?: Sponsor;
  categories: Record<string, string>;
  tiers: Record<string, string>;
  isEdit?: boolean;
}

const SponsorForm = ({ sponsor, categories, tiers, isEdit = false }: Props) => {
  const [form, setForm] = useState<Sponsor>({
    name: sponsor?.name || '',
    category: sponsor?.category || 'equipment',
    logo: sponsor?.logo || '',
    description: sponsor?.description || '',
    tier: sponsor?.tier || 'bronze',
    website_url: sponsor?.website_url || '',
    contact_email: sponsor?.contact_email || '',
    contact_phone: sponsor?.contact_phone || '',
    sort_order: sponsor?.sort_order || 0,
    status: sponsor?.status || 'draft',
    active: sponsor?.active ?? true
  });

  const [loading, setLoading] = useState(false);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    const url = isEdit 
      ? route('admin.frontend.sponsors.update', sponsor?.id)
      : route('admin.frontend.sponsors.store');
    
    const method = isEdit ? 'put' : 'post';

    router[method](url, form as any, {
      onSuccess: () => {
        toast.success(`Sponsor ${isEdit ? 'updated' : 'created'} successfully`);
        router.get(route('admin.frontend.sponsors.index'));
      },
      onError: (errors) => {
        toast.error('Please check the form for errors');
      },
      onFinish: () => setLoading(false)
    });
  };

  const handleInputChange = (field: keyof Sponsor, value: any) => {
    setForm(prev => ({ ...prev, [field]: value }));
  };

  const getTierBadgeStyle = (tier: string) => {
    switch (tier) {
      case 'platinum': return 'bg-purple-100 text-purple-800 border-purple-200';
      case 'gold': return 'bg-yellow-100 text-yellow-800 border-yellow-200';
      case 'silver': return 'bg-gray-100 text-gray-800 border-gray-200';
      case 'bronze': return 'bg-orange-100 text-orange-800 border-orange-200';
      default: return 'bg-gray-100 text-gray-800 border-gray-200';
    }
  };

  return (
    <AppLayout>
      <Head title={`${isEdit ? 'Edit' : 'Create'} Sponsor`} />
      
      <div className="space-y-6">
        <div className="flex items-center space-x-4">
          <Button variant="ghost" onClick={() => router.get(route('admin.frontend.sponsors.index'))}>
            <ArrowLeft className="h-4 w-4 mr-2" />
            Back
          </Button>
          <div>
            <h1 className="text-2xl font-bold">{isEdit ? 'Edit' : 'Create'} Sponsor</h1>
            <p className="text-muted-foreground">{isEdit ? 'Update the' : 'Add a new'} sponsor</p>
          </div>
        </div>

        <form onSubmit={handleSubmit}>
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            <div className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle>Basic Information</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <Label htmlFor="name">Sponsor Name *</Label>
                    <Input
                      id="name"
                      value={form.name}
                      onChange={(e) => handleInputChange('name', e.target.value)}
                      placeholder="Enter sponsor name"
                      required
                    />
                  </div>

                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <Label htmlFor="category">Category *</Label>
                      <Select value={form.category} onValueChange={(value) => handleInputChange('category', value)}>
                        <SelectTrigger>
                          <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                          {Object.entries(categories).map(([key, label]) => (
                            <SelectItem key={key} value={key}>{label}</SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                    </div>
                    <div>
                      <Label htmlFor="tier">Sponsorship Tier *</Label>
                      <Select value={form.tier} onValueChange={(value) => handleInputChange('tier', value)}>
                        <SelectTrigger>
                          <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                          {Object.entries(tiers).map(([key, label]) => (
                            <SelectItem key={key} value={key}>
                              <span className={`inline-flex items-center px-2 py-1 rounded text-xs font-medium border ${getTierBadgeStyle(key)}`}>
                                {label}
                              </span>
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                    </div>
                  </div>

                  <div>
                    <Label htmlFor="description">Description</Label>
                    <Textarea
                      id="description"
                      value={form.description}
                      onChange={(e) => handleInputChange('description', e.target.value)}
                      placeholder="Brief description of the sponsor"
                      rows={3}
                    />
                  </div>

                  <div>
                    <Label>Logo</Label>
                    <MediaManager
                      trigger={
                        <Button type="button" variant="outline" className="w-full">
                          {form.logo ? 'Change Logo' : 'Select Logo'}
                        </Button>
                      }
                      onSelect={(files) => {
                        const file = Array.isArray(files) ? files[0] : files;
                        if (file) {
                          handleInputChange('logo', file.url);
                        }
                      }}
                      acceptedTypes={['image/*']}
                    />
                    {form.logo && (
                      <div className="mt-2">
                        <img 
                          src={form.logo} 
                          alt="Logo preview" 
                          className="w-32 h-20 object-contain rounded border"
                        />
                      </div>
                    )}
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>Contact Information</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <Label htmlFor="website_url">Website URL</Label>
                    <div className="relative">
                      <Input
                        id="website_url"
                        type="url"
                        value={form.website_url}
                        onChange={(e) => handleInputChange('website_url', e.target.value)}
                        placeholder="https://example.com"
                        className="pr-10"
                      />
                      {form.website_url && (
                        <a
                          href={form.website_url}
                          target="_blank"
                          rel="noopener noreferrer"
                          className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 hover:text-blue-600"
                        >
                          <ExternalLink className="h-4 w-4" />
                        </a>
                      )}
                    </div>
                  </div>

                  <div>
                    <Label htmlFor="contact_email">Contact Email</Label>
                    <Input
                      id="contact_email"
                      type="email"
                      value={form.contact_email}
                      onChange={(e) => handleInputChange('contact_email', e.target.value)}
                      placeholder="contact@sponsor.com"
                    />
                  </div>

                  <div>
                    <Label htmlFor="contact_phone">Contact Phone</Label>
                    <Input
                      id="contact_phone"
                      type="tel"
                      value={form.contact_phone}
                      onChange={(e) => handleInputChange('contact_phone', e.target.value)}
                      placeholder="07XX XXX XXX"
                    />
                  </div>
                </CardContent>
              </Card>
            </div>

            <div className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle>Display Settings</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <Label htmlFor="sort_order">Sort Order</Label>
                    <Input
                      id="sort_order"
                      type="number"
                      value={form.sort_order}
                      onChange={(e) => handleInputChange('sort_order', parseInt(e.target.value) || 0)}
                      placeholder="0"
                      min="0"
                    />
                    <p className="text-xs text-muted-foreground mt-1">
                      Lower numbers appear first
                    </p>
                  </div>

                  <div>
                    <Label htmlFor="status">Status *</Label>
                    <Select value={form.status} onValueChange={(value) => handleInputChange('status', value)}>
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="draft">Draft</SelectItem>
                        <SelectItem value="published">Published</SelectItem>
                        <SelectItem value="archived">Archived</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>

                  <div className="flex items-center space-x-2">
                    <Switch
                      id="active"
                      checked={form.active}
                      onCheckedChange={(checked) => handleInputChange('active', checked)}
                    />
                    <Label htmlFor="active">Active</Label>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>Preview</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="bg-gray-50 p-4 rounded-lg">
                    <div className="flex items-start gap-4">
                      <div className="flex-shrink-0">
                        {form.logo ? (
                          <img
                            src={form.logo}
                            alt="Preview"
                            className="w-16 h-16 object-contain rounded border bg-white"
                          />
                        ) : (
                          <div className="w-16 h-16 bg-gray-200 rounded border flex items-center justify-center">
                            <span className="text-gray-400 text-xs">Logo</span>
                          </div>
                        )}
                      </div>
                      <div className="flex-1">
                        <h3 className="font-semibold text-sm mb-1">
                          {form.name || 'Sponsor Name'}
                        </h3>
                        <div className="flex gap-2 mb-2">
                          <span className={`inline-flex items-center px-2 py-1 rounded text-xs font-medium border ${getTierBadgeStyle(form.tier)}`}>
                            {tiers[form.tier] || form.tier}
                          </span>
                          <span className="inline-flex items-center px-2 py-1 rounded text-xs font-medium bg-blue-100 text-blue-800 border border-blue-200">
                            {categories[form.category] || form.category}
                          </span>
                        </div>
                        {form.description && (
                          <p className="text-xs text-muted-foreground mb-2">
                            {form.description}
                          </p>
                        )}
                        <div className="flex gap-3 text-xs text-muted-foreground">
                          {form.website_url && (
                            <span className="flex items-center gap-1">
                              <ExternalLink className="h-3 w-3" />
                              Website
                            </span>
                          )}
                          {form.contact_email && (
                            <span>📧 Email</span>
                          )}
                          {form.contact_phone && (
                            <span>📱 Phone</span>
                          )}
                        </div>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <div className="flex gap-4">
                <Button 
                  type="button" 
                  variant="outline" 
                  onClick={() => router.get(route('admin.frontend.sponsors.index'))}
                  disabled={loading}
                >
                  Cancel
                </Button>
                <Button 
                  type="submit" 
                  style={{ backgroundColor: '#256630' }}
                  className="hover:opacity-90 flex-1"
                  disabled={loading}
                >
                  <Save className="h-4 w-4 mr-2" />
                  {loading ? 'Saving...' : isEdit ? 'Update Sponsor' : 'Create Sponsor'}
                </Button>
              </div>
            </div>
          </div>
        </form>
      </div>
    </AppLayout>
  );
};

export default SponsorForm; 