import React from 'react';
import { Head, Link, router } from '@inertiajs/react';
import { route } from 'ziggy-js';
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Plus, Edit, Trash2, Eye, EyeOff, Star, StarOff } from 'lucide-react';
import { toast } from 'react-hot-toast';
import Swal from 'sweetalert2';
import { NewsArticle } from '@/types';

interface Props {
  articles: {
    data: NewsArticle[];
    links: any;
    meta: any;
  };
  filters: {
    search?: string;
    category?: string;
    status?: string;
  };
  categories: Record<string, string>;
}

const NewsArticlesIndex = ({ articles, filters, categories }: Props) => {
  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    const formData = new FormData(e.target as HTMLFormElement);
    router.get(route('admin.frontend.news-articles.index'), Object.fromEntries(formData), {
      preserveState: true,
    });
  };

  const handleDelete = (article: NewsArticle) => {
    Swal.fire({
      title: 'Are you sure?',
      text: 'This action cannot be undone!',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonColor: '#c22b2b',
      cancelButtonColor: '#256630',
      confirmButtonText: 'Yes, delete it!'
    }).then((result) => {
      if (result.isConfirmed) {
        router.delete(route('admin.frontend.news-articles.destroy', article.id), {
          onSuccess: () => {
            toast.success('News article deleted successfully');
            Swal.fire('Deleted!', 'The news article has been deleted.', 'success');
          },
          onError: () => {
            toast.error('Failed to delete news article');
          }
        });
      }
    });
  };

  const handleToggleFeatured = (article: NewsArticle) => {
    router.patch(route('admin.frontend.news-articles.toggle-featured', article.id), {}, {
      onSuccess: () => {
        toast.success(`Article ${article.featured ? 'removed from' : 'added to'} featured`);
      },
      onError: () => {
        toast.error('Failed to update featured status');
      }
    });
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'published': return 'bg-green-100 text-green-800';
      case 'draft': return 'bg-yellow-100 text-yellow-800';
      case 'archived': return 'bg-gray-100 text-gray-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  return (
    <AppLayout>
      <Head title="News Articles" />
      
      <div className="space-y-6">
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-2xl font-bold">News Articles</h1>
            <p className="text-muted-foreground">Manage news articles for the frontend</p>
          </div>
          <Link href={route('admin.frontend.news-articles.create')}>
            <Button style={{ backgroundColor: '#256630' }} className="hover:opacity-90">
              <Plus className="h-4 w-4 mr-2" />
              Add Article
            </Button>
          </Link>
        </div>

        <Card>
          <CardHeader>
            <CardTitle>Filters</CardTitle>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleSearch} className="flex gap-4">
              <div className="flex-1">
                <Input
                  name="search"
                  placeholder="Search articles..."
                  defaultValue={filters.search}
                />
              </div>
              <select
                name="category"
                className="px-3 py-2 border border-input rounded-md"
                defaultValue={filters.category || 'all'}
              >
                {Object.entries(categories).map(([key, label]) => (
                  <option key={key} value={key}>{label}</option>
                ))}
              </select>
              <select
                name="status"
                className="px-3 py-2 border border-input rounded-md"
                defaultValue={filters.status || 'all'}
              >
                <option value="all">All Status</option>
                <option value="draft">Draft</option>
                <option value="published">Published</option>
                <option value="archived">Archived</option>
              </select>
              <Button type="submit" style={{ backgroundColor: '#256630' }} className="hover:opacity-90">
                Filter
              </Button>
            </form>
          </CardContent>
        </Card>

        <div className="grid gap-4">
          {articles.data.map((article) => (
            <Card key={article.id} className="hover:shadow-md transition-shadow">
              <CardContent className="p-6">
                <div className="flex justify-between items-start">
                  <div className="flex-1">
                    <div className="flex items-center gap-2 mb-2">
                      <h3 className="text-lg font-semibold">{article.title}</h3>
                      {article.featured && <Star className="h-4 w-4 text-yellow-500 fill-current" />}
                    </div>
                    <p className="text-muted-foreground mb-3 line-clamp-2">{article.excerpt}</p>
                    <div className="flex items-center gap-4 text-sm text-muted-foreground">
                      <span>By {article.author}</span>
                      <Badge variant="outline">{categories[article.category] || article.category}</Badge>
                      <Badge className={getStatusColor(article.status)}>
                        {article.status}
                      </Badge>
                      <span>Views: {article.views}</span>
                      <span>Likes: {article.likes}</span>
                      <span>{new Date(article.created_at).toLocaleDateString()}</span>
                    </div>
                  </div>
                  <div className="flex items-center space-x-2">
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => handleToggleFeatured(article)}
                      title={article.featured ? 'Remove from featured' : 'Mark as featured'}
                    >
                      {article.featured ? 
                        <Star className="h-4 w-4 text-yellow-500 fill-current" /> : 
                        <StarOff className="h-4 w-4" />
                      }
                    </Button>
                    <Link href={route('admin.frontend.news-articles.edit', article.id)}>
                      <Button variant="ghost" size="sm">
                        <Edit className="h-4 w-4" />
                      </Button>
                    </Link>
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => handleDelete(article)}
                    >
                      <Trash2 className="h-4 w-4" />
                    </Button>
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>

        {articles.data.length === 0 && (
          <Card>
            <CardContent className="p-6 text-center">
              <p className="text-muted-foreground">No articles found</p>
            </CardContent>
          </Card>
        )}
      </div>
    </AppLayout>
  );
};

export default NewsArticlesIndex; 