import React, { useState } from 'react';
import { Head, router } from '@inertiajs/react';
import { route } from 'ziggy-js';
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Switch } from '@/components/ui/switch';
import { ArrowLeft, Save } from 'lucide-react';
import MediaManager from '@/components/MediaManager';
import { toast } from 'react-hot-toast';
import { NewsArticle } from '@/types';

interface Props {
  article?: NewsArticle;
  categories: Record<string, string>;
  isEdit?: boolean;
}

const NewsArticleForm = ({ article, categories, isEdit = false }: Props) => {
  const [form, setForm] = useState<NewsArticle>({
    title: article?.title || '',
    excerpt: article?.excerpt || '',
    content: article?.content || '',
    category: article?.category || 'tournament',
    author: article?.author || '',
    author_image: article?.author_image || '',
    date: article?.date || new Date().toISOString().split('T')[0],
    read_time: article?.read_time || 5,
    image: article?.image || '',
    tags: article?.tags || [],
    featured: article?.featured || false,
    status: article?.status || 'draft',
    published_at: article?.published_at || '',
    meta_description: article?.meta_description || '',
    meta_keywords: article?.meta_keywords || ''
  });

  const [loading, setLoading] = useState(false);
  const [tagInput, setTagInput] = useState('');

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    const url = isEdit 
      ? route('admin.frontend.news-articles.update', article?.id)
      : route('admin.frontend.news-articles.store');
    
    const method = isEdit ? 'put' : 'post';

    router[method](url, form as any, {
      onSuccess: () => {
        toast.success(`News article ${isEdit ? 'updated' : 'created'} successfully`);
        router.get(route('admin.frontend.news-articles.index'));
      },
      onError: (errors) => {
        toast.error('Please check the form for errors');
      },
      onFinish: () => setLoading(false)
    });
  };

  const handleInputChange = (field: keyof NewsArticle, value: any) => {
    setForm(prev => ({ ...prev, [field]: value }));
  };

  const handleAddTag = () => {
    if (tagInput.trim() && !form.tags.includes(tagInput.trim())) {
      setForm(prev => ({
        ...prev,
        tags: [...prev.tags, tagInput.trim()]
      }));
      setTagInput('');
    }
  };

  const handleRemoveTag = (tagToRemove: string) => {
    setForm(prev => ({
      ...prev,
      tags: prev.tags.filter(tag => tag !== tagToRemove)
    }));
  };

  return (
    <AppLayout>
      <Head title={`${isEdit ? 'Edit' : 'Create'} News Article`} />
      
      <div className="space-y-6">
        <div className="flex items-center space-x-4">
          <Button variant="ghost" onClick={() => router.get(route('admin.frontend.news-articles.index'))}>
            <ArrowLeft className="h-4 w-4 mr-2" />
            Back
          </Button>
          <div>
            <h1 className="text-2xl font-bold">{isEdit ? 'Edit' : 'Create'} News Article</h1>
            <p className="text-muted-foreground">{isEdit ? 'Update the' : 'Add a new'} news article</p>
          </div>
        </div>

        <form onSubmit={handleSubmit}>
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            <div className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle>Basic Information</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <Label htmlFor="title">Title *</Label>
                    <Input
                      id="title"
                      value={form.title}
                      onChange={(e) => handleInputChange('title', e.target.value)}
                      placeholder="Enter article title"
                      required
                    />
                  </div>

                  <div>
                    <Label htmlFor="excerpt">Excerpt</Label>
                    <Textarea
                      id="excerpt"
                      value={form.excerpt}
                      onChange={(e) => handleInputChange('excerpt', e.target.value)}
                      placeholder="Brief description of the article"
                      rows={3}
                    />
                  </div>

                  <div>
                    <Label htmlFor="category">Category *</Label>
                    <Select value={form.category} onValueChange={(value) => handleInputChange('category', value)}>
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        {Object.entries(categories).map(([key, label]) => (
                          key !== 'all' && (
                            <SelectItem key={key} value={key}>{label}</SelectItem>
                          )
                        ))}
                      </SelectContent>
                    </Select>
                  </div>

                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <Label htmlFor="author">Author *</Label>
                      <Input
                        id="author"
                        value={form.author}
                        onChange={(e) => handleInputChange('author', e.target.value)}
                        placeholder="Author name"
                        required
                      />
                    </div>
                    <div>
                      <Label htmlFor="read_time">Read Time (minutes)</Label>
                      <Input
                        id="read_time"
                        type="number"
                        value={form.read_time}
                        onChange={(e) => handleInputChange('read_time', parseInt(e.target.value) || 0)}
                        placeholder="5"
                        min="1"
                      />
                    </div>
                  </div>

                  <div>
                    <Label htmlFor="date">Publication Date</Label>
                    <Input
                      id="date"
                      type="date"
                      value={form.date}
                      onChange={(e) => handleInputChange('date', e.target.value)}
                    />
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>Content</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <Label htmlFor="content">Article Content *</Label>
                    <Textarea
                      id="content"
                      value={form.content}
                      onChange={(e) => handleInputChange('content', e.target.value)}
                      placeholder="Write your article content here..."
                      rows={12}
                      required
                    />
                  </div>
                </CardContent>
              </Card>
            </div>

            <div className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle>Media & Images</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <Label>Featured Image</Label>
                    <MediaManager
                      trigger={
                        <Button type="button" variant="outline" className="w-full">
                          {form.image ? 'Change Image' : 'Select Image'}
                        </Button>
                      }
                      onSelect={(files) => {
                        if (Array.isArray(files) && files.length > 0) {
                          handleInputChange('image', files[0].url);
                        }
                      }}
                      acceptedTypes={['image/*']}
                    />
                    {form.image && (
                      <div className="mt-2">
                        <img src={form.image} alt="Preview" className="w-full h-32 object-cover rounded" />
                      </div>
                    )}
                  </div>

                  <div>
                    <Label>Author Image</Label>
                    <MediaManager
                      trigger={
                        <Button type="button" variant="outline" className="w-full">
                          {form.author_image ? 'Change Author Image' : 'Select Author Image'}
                        </Button>
                      }
                      onSelect={(files) => {
                        if (Array.isArray(files) && files.length > 0) {
                          handleInputChange('author_image', files[0].url);
                        }
                      }}
                      acceptedTypes={['image/*']}
                    />
                    {form.author_image && (
                      <div className="mt-2">
                        <img src={form.author_image} alt="Author" className="w-16 h-16 object-cover rounded-full" />
                      </div>
                    )}
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>Tags</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="flex gap-2">
                    <Input
                      value={tagInput}
                      onChange={(e) => setTagInput(e.target.value)}
                      placeholder="Add a tag"
                      onKeyPress={(e) => e.key === 'Enter' && (e.preventDefault(), handleAddTag())}
                    />
                    <Button type="button" onClick={handleAddTag} variant="outline">
                      Add
                    </Button>
                  </div>
                  <div className="flex flex-wrap gap-2">
                    {form.tags.map((tag, index) => (
                      <span
                        key={index}
                        className="bg-blue-100 text-blue-800 px-2 py-1 rounded text-sm cursor-pointer hover:bg-blue-200"
                        onClick={() => handleRemoveTag(tag)}
                      >
                        {tag} ×
                      </span>
                    ))}
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>Publishing Options</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <Label htmlFor="status">Status *</Label>
                    <Select value={form.status} onValueChange={(value) => handleInputChange('status', value)}>
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="draft">Draft</SelectItem>
                        <SelectItem value="published">Published</SelectItem>
                        <SelectItem value="archived">Archived</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>

                  <div className="flex items-center space-x-2">
                    <Switch
                      id="featured"
                      checked={form.featured}
                      onCheckedChange={(checked) => handleInputChange('featured', checked)}
                    />
                    <Label htmlFor="featured">Featured Article</Label>
                  </div>

                  {form.status === 'published' && (
                    <div>
                      <Label htmlFor="published_at">Publish Date & Time</Label>
                      <Input
                        id="published_at"
                        type="datetime-local"
                        value={form.published_at}
                        onChange={(e) => handleInputChange('published_at', e.target.value)}
                      />
                    </div>
                  )}
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>SEO</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <Label htmlFor="meta_description">Meta Description</Label>
                    <Textarea
                      id="meta_description"
                      value={form.meta_description}
                      onChange={(e) => handleInputChange('meta_description', e.target.value)}
                      placeholder="Brief description for search engines (160 characters max)"
                      rows={3}
                      maxLength={160}
                    />
                    <p className="text-xs text-muted-foreground mt-1">
                      {form.meta_description.length}/160 characters
                    </p>
                  </div>

                  <div>
                    <Label htmlFor="meta_keywords">Meta Keywords</Label>
                    <Input
                      id="meta_keywords"
                      value={form.meta_keywords}
                      onChange={(e) => handleInputChange('meta_keywords', e.target.value)}
                      placeholder="Comma-separated keywords"
                    />
                  </div>
                </CardContent>
              </Card>

              <div className="flex gap-4">
                <Button 
                  type="button" 
                  variant="outline" 
                  onClick={() => router.get(route('admin.frontend.news-articles.index'))}
                  disabled={loading}
                >
                  Cancel
                </Button>
                <Button 
                  type="submit" 
                  style={{ backgroundColor: '#256630' }}
                  className="hover:opacity-90 flex-1"
                  disabled={loading}
                >
                  <Save className="h-4 w-4 mr-2" />
                  {loading ? 'Saving...' : isEdit ? 'Update Article' : 'Create Article'}
                </Button>
              </div>
            </div>
          </div>
        </form>
      </div>
    </AppLayout>
  );
};

export default NewsArticleForm; 