import React from 'react';
import { Head, Link, router } from '@inertiajs/react';
import { route } from 'ziggy-js';
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Plus, Edit, Trash2, Eye, EyeOff } from 'lucide-react';
import { toast } from 'react-hot-toast';

interface HeroSlide {
  id: number;
  title: string;
  subtitle: string;
  bg_image: string;
  cta_primary: string;
  cta_secondary: string;
  sort_order: number;
  status: string;
  active: boolean;
  created_at: string;
}

interface Props {
  slides: {
    data: HeroSlide[];
    current_page: number;
    last_page: number;
    per_page: number;
    total: number;
  };
  filters: {
    search?: string;
    status?: string;
  };
}

const HeroSlidesIndex = ({ slides, filters }: Props) => {
  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    const formData = new FormData(e.target as HTMLFormElement);
    router.get(route('admin.frontend.hero-slides.index'), Object.fromEntries(formData), {
      preserveState: true,
    });
  };

  const handleDelete = (slide: HeroSlide) => {
    if (confirm('Are you sure you want to delete this slide?')) {
      router.delete(route('admin.frontend.hero-slides.destroy', slide.id), {
        onSuccess: () => toast.success('Slide deleted successfully'),
      });
    }
  };

  const handleToggleActive = (slide: HeroSlide) => {
    router.patch(route('admin.frontend.hero-slides.update', slide.id), {
      active: !slide.active,
    }, {
      onSuccess: () => toast.success('Slide status updated'),
    });
  };

  return (
    <AppLayout>
      <Head title="Hero Slides" />
      
      <div className="space-y-6">
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-2xl font-bold">Hero Slides</h1>
            <p className="text-muted-foreground">Manage the hero section slides for the frontend</p>
          </div>
          <Link href={route('admin.frontend.hero-slides.create')}>
            <Button>
              <Plus className="h-4 w-4 mr-2" />
              Add Slide
            </Button>
          </Link>
        </div>

        <Card>
          <CardHeader>
            <CardTitle>Filters</CardTitle>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleSearch} className="flex gap-4">
              <div className="flex-1">
                <Input
                  name="search"
                  placeholder="Search slides..."
                  defaultValue={filters.search}
                />
              </div>
              <select
                name="status"
                className="px-3 py-2 border border-input rounded-md"
                defaultValue={filters.status || 'all'}
              >
                <option value="all">All Status</option>
                <option value="draft">Draft</option>
                <option value="published">Published</option>
                <option value="archived">Archived</option>
              </select>
              <Button type="submit">Filter</Button>
            </form>
          </CardContent>
        </Card>

        <div className="grid gap-4">
          {slides.data.map((slide) => (
            <Card key={slide.id}>
              <CardContent className="p-6">
                <div className="flex items-center justify-between">
                  <div className="flex items-center space-x-4">
                    {slide.bg_image && (
                      <img
                        src={slide.bg_image}
                        alt={slide.title}
                        className="w-16 h-12 object-cover rounded"
                      />
                    )}
                    <div>
                      <h3 className="font-semibold">{slide.title}</h3>
                      <p className="text-sm text-muted-foreground">{slide.subtitle}</p>
                      <div className="flex items-center space-x-2 mt-1">
                        <Badge variant={slide.status === 'published' ? 'default' : 'secondary'}>
                          {slide.status}
                        </Badge>
                        <Badge variant={slide.active ? 'default' : 'outline'}>
                          {slide.active ? 'Active' : 'Inactive'}
                        </Badge>
                        <span className="text-xs text-muted-foreground">
                          Order: {slide.sort_order}
                        </span>
                      </div>
                    </div>
                  </div>
                  <div className="flex items-center space-x-2">
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => handleToggleActive(slide)}
                    >
                      {slide.active ? <Eye className="h-4 w-4" /> : <EyeOff className="h-4 w-4" />}
                    </Button>
                    <Link href={route('admin.frontend.hero-slides.edit', slide.id)}>
                      <Button variant="ghost" size="sm">
                        <Edit className="h-4 w-4" />
                      </Button>
                    </Link>
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => handleDelete(slide)}
                    >
                      <Trash2 className="h-4 w-4" />
                    </Button>
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>

        {slides.data.length === 0 && (
          <Card>
            <CardContent className="p-6 text-center">
              <p className="text-muted-foreground">No slides found</p>
            </CardContent>
          </Card>
        )}
      </div>
    </AppLayout>
  );
};

export default HeroSlidesIndex; 