import React, { useState } from 'react';
import { Head, router } from '@inertiajs/react';
import { route } from 'ziggy-js';
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Switch } from '@/components/ui/switch';
import { ArrowLeft, Save } from 'lucide-react';
import MediaManager from '@/components/MediaManager';

const HeroSlideCreate = () => {
  const [form, setForm] = useState({
    title: '',
    subtitle: '',
    bg_image: '',
    cta_primary: '',
    cta_secondary: '',
    cta_primary_url: '',
    cta_secondary_url: '',
    sort_order: 0,
    status: 'draft',
    active: true
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    router.post(route('admin.frontend.hero-slides.store'), form as any, {
      onSuccess: () => {
        // Success handled by redirect
      },
    });
  };

  const handleInputChange = (field: string, value: any) => {
    setForm(prev => ({ ...prev, [field]: value }));
  };

  return (
    <AppLayout>
      <Head title="Create Hero Slide" />
      
      <div className="space-y-6">
        <div className="flex items-center space-x-4">
          <Button variant="ghost" onClick={() => router.get(route('admin.frontend.hero-slides.index'))}>
            <ArrowLeft className="h-4 w-4 mr-2" />
            Back
          </Button>
          <div>
            <h1 className="text-2xl font-bold">Create Hero Slide</h1>
            <p className="text-muted-foreground">Add a new slide to the hero section</p>
          </div>
        </div>

        <form onSubmit={handleSubmit}>
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            <Card>
              <CardHeader>
                <CardTitle>Basic Information</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <Label htmlFor="title">Title</Label>
                  <Input
                    id="title"
                    value={form.title}
                    onChange={(e) => handleInputChange('title', e.target.value)}
                    placeholder="Enter slide title"
                    required
                  />
                </div>

                <div>
                  <Label htmlFor="subtitle">Subtitle</Label>
                  <Input
                    id="subtitle"
                    value={form.subtitle}
                    onChange={(e) => handleInputChange('subtitle', e.target.value)}
                    placeholder="Enter slide subtitle"
                  />
                </div>

                <div>
                  <Label htmlFor="sort_order">Sort Order</Label>
                  <Input
                    id="sort_order"
                    type="number"
                    value={form.sort_order}
                    onChange={(e) => handleInputChange('sort_order', parseInt(e.target.value))}
                    placeholder="0"
                  />
                </div>

                <div>
                  <Label htmlFor="status">Status</Label>
                  <Select value={form.status} onValueChange={(value) => handleInputChange('status', value)}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="draft">Draft</SelectItem>
                      <SelectItem value="published">Published</SelectItem>
                      <SelectItem value="archived">Archived</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                <div className="flex items-center space-x-2">
                  <Switch
                    id="active"
                    checked={form.active}
                    onCheckedChange={(checked) => handleInputChange('active', checked)}
                  />
                  <Label htmlFor="active">Active</Label>
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Media & Actions</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <Label>Background Image</Label>
                  <MediaManager
                    trigger={<Button variant="outline" type="button">Select Image</Button>}
                    onSelect={(files) => {
                      const file = Array.isArray(files) ? files[0] : files;
                      if (file) {
                        handleInputChange('bg_image', file.url);
                      }
                    }}
                    acceptedTypes={['image/*']}
                    multiple={false}
                    collection="hero-slides"
                  />
                  {form.bg_image && (
                    <div className="mt-2">
                      <img src={form.bg_image} alt="Preview" className="w-full h-32 object-cover rounded" />
                    </div>
                  )}
                </div>

                <div>
                  <Label htmlFor="cta_primary">Primary CTA Text</Label>
                  <Input
                    id="cta_primary"
                    value={form.cta_primary}
                    onChange={(e) => handleInputChange('cta_primary', e.target.value)}
                    placeholder="e.g., Register Now"
                  />
                </div>

                <div>
                  <Label htmlFor="cta_primary_url">Primary CTA URL</Label>
                  <Input
                    id="cta_primary_url"
                    value={form.cta_primary_url}
                    onChange={(e) => handleInputChange('cta_primary_url', e.target.value)}
                    placeholder="https://example.com"
                  />
                </div>

                <div>
                  <Label htmlFor="cta_secondary">Secondary CTA Text</Label>
                  <Input
                    id="cta_secondary"
                    value={form.cta_secondary}
                    onChange={(e) => handleInputChange('cta_secondary', e.target.value)}
                    placeholder="e.g., Learn More"
                  />
                </div>

                <div>
                  <Label htmlFor="cta_secondary_url">Secondary CTA URL</Label>
                  <Input
                    id="cta_secondary_url"
                    value={form.cta_secondary_url}
                    onChange={(e) => handleInputChange('cta_secondary_url', e.target.value)}
                    placeholder="https://example.com"
                  />
                </div>
              </CardContent>
            </Card>
          </div>

          <div className="flex justify-end space-x-4 mt-6">
            <Button
              type="button"
              variant="outline"
              onClick={() => router.get(route('admin.frontend.hero-slides.index'))}
            >
              Cancel
            </Button>
            <Button type="submit">
              <Save className="h-4 w-4 mr-2" />
              Create Slide
            </Button>
          </div>
        </form>
      </div>
    </AppLayout>
  );
};

export default HeroSlideCreate; 