import React from 'react';
import { Head, Link, router } from '@inertiajs/react';
import { route } from 'ziggy-js';
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Plus, Edit, Trash2, Eye, Star, StarOff } from 'lucide-react';
import { toast } from 'react-hot-toast';
import Swal from 'sweetalert2';
import { GalleryImage } from '@/types';

interface Props {
  images: {
    data: GalleryImage[];
    links: any;
    meta: any;
  };
  filters: {
    search?: string;
    category?: string;
    edition?: string;
    status?: string;
  };
  categories: Record<string, string>;
}

const GalleryIndex = ({ images, filters, categories }: Props) => {
  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    const formData = new FormData(e.target as HTMLFormElement);
    router.get(route('admin.frontend.gallery.index'), Object.fromEntries(formData), {
      preserveState: true,
    });
  };

  const handleDelete = (image: GalleryImage) => {
    Swal.fire({
      title: 'Are you sure?',
      text: 'This action cannot be undone!',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonColor: '#c22b2b',
      cancelButtonColor: '#256630',
      confirmButtonText: 'Yes, delete it!'
    }).then((result) => {
      if (result.isConfirmed) {
        router.delete(route('admin.frontend.gallery.destroy', image.id), {
          onSuccess: () => {
            toast.success('Gallery image deleted successfully');
            Swal.fire('Deleted!', 'The gallery image has been deleted.', 'success');
          },
          onError: () => {
            toast.error('Failed to delete gallery image');
          }
        });
      }
    });
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'published': return 'bg-green-100 text-green-800';
      case 'draft': return 'bg-yellow-100 text-yellow-800';
      case 'archived': return 'bg-gray-100 text-gray-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  return (
    <AppLayout>
      <Head title="Gallery Images" />
      
      <div className="space-y-6">
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-2xl font-bold">Gallery Images</h1>
            <p className="text-muted-foreground">Manage gallery images for the frontend</p>
          </div>
          <div className="flex gap-2">
            <Link href={route('admin.frontend.gallery.bulk-create')}>
              <Button variant="outline" className="hover:opacity-90">
                <Plus className="h-4 w-4 mr-2" />
                Bulk Create
              </Button>
            </Link>
            <Link href={route('admin.frontend.gallery.create')}>
              <Button style={{ backgroundColor: '#256630' }} className="hover:opacity-90">
                <Plus className="h-4 w-4 mr-2" />
                Add Image
              </Button>
            </Link>
          </div>
        </div>

        <Card>
          <CardHeader>
            <CardTitle>Filters</CardTitle>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleSearch} className="flex gap-4">
              <div className="flex-1">
                <Input
                  name="search"
                  placeholder="Search images..."
                  defaultValue={filters.search}
                />
              </div>
              <select
                name="category"
                className="px-3 py-2 border border-input rounded-md"
                defaultValue={filters.category || 'all'}
              >
                {Object.entries(categories).map(([key, label]) => (
                  <option key={key} value={key}>{label}</option>
                ))}
              </select>
              <Input
                name="edition"
                placeholder="Edition"
                defaultValue={filters.edition}
                className="w-32"
              />
              <select
                name="status"
                className="px-3 py-2 border border-input rounded-md"
                defaultValue={filters.status || 'all'}
              >
                <option value="all">All Status</option>
                <option value="draft">Draft</option>
                <option value="published">Published</option>
                <option value="archived">Archived</option>
              </select>
              <Button type="submit" style={{ backgroundColor: '#256630' }} className="hover:opacity-90">
                Filter
              </Button>
            </form>
          </CardContent>
        </Card>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
          {images.data.map((image) => (
            <Card key={image.id} className="hover:shadow-md transition-shadow overflow-hidden">
              <div className="aspect-video relative">
                <img 
                  src={image.src} 
                  alt={image.alt}
                  className="w-full h-full object-cover"
                />
                {image.featured && (
                  <div className="absolute top-2 right-2">
                    <Star className="h-4 w-4 text-yellow-500 fill-current bg-white rounded-full p-0.5" />
                  </div>
                )}
              </div>
              <CardContent className="p-4">
                <div className="space-y-2">
                  <div className="flex items-center gap-2">
                    <h3 className="font-semibold truncate flex-1">{image.title}</h3>
                  </div>
                  
                  {image.description && (
                    <p className="text-sm text-muted-foreground line-clamp-2">
                      {image.description}
                    </p>
                  )}
                  
                  <div className="flex flex-wrap gap-1">
                    <Badge variant="outline" className="text-xs">
                      {categories[image.category] || image.category}
                    </Badge>
                    <Badge className={`text-xs ${getStatusColor(image.status)}`}>
                      {image.status}
                    </Badge>
                    {image.edition && (
                      <Badge variant="secondary" className="text-xs">
                        {image.edition}
                      </Badge>
                    )}
                  </div>
                  
                  <div className="text-xs text-muted-foreground space-y-1">
                    {image.photographer && (
                      <div>📸 {image.photographer}</div>
                    )}
                    {image.date_taken && (
                      <div>📅 {new Date(image.date_taken).toLocaleDateString()}</div>
                    )}
                  </div>
                  
                  <div className="flex justify-between items-center pt-2">
                    <span className="text-xs text-muted-foreground">
                      #{image.sort_order}
                    </span>
                    <div className="flex items-center space-x-1">
                      <Link href={route('admin.frontend.gallery.edit', image.id)}>
                        <Button variant="ghost" size="sm" className="h-8 w-8 p-0">
                          <Edit className="h-3 w-3" />
                        </Button>
                      </Link>
                      <Button
                        variant="ghost"
                        size="sm"
                        className="h-8 w-8 p-0"
                        onClick={() => handleDelete(image)}
                      >
                        <Trash2 className="h-3 w-3" />
                      </Button>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>

        {images.data.length === 0 && (
          <Card>
            <CardContent className="p-6 text-center">
              <p className="text-muted-foreground">No images found</p>
            </CardContent>
          </Card>
        )}
      </div>
    </AppLayout>
  );
};

export default GalleryIndex; 