import React, { useState } from 'react';
import { Head, router } from '@inertiajs/react';
import { route } from 'ziggy-js';
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Switch } from '@/components/ui/switch';
import { ArrowLeft, Save } from 'lucide-react';
import MediaManager from '@/components/MediaManager';
import { toast } from 'react-hot-toast';
import { GalleryImage } from '@/types';

interface Props {
  image?: GalleryImage;
  categories: Record<string, string>;
  isEdit?: boolean;
}

const GalleryImageForm = ({ image, categories, isEdit = false }: Props) => {
  const [form, setForm] = useState<GalleryImage>({
    src: image?.src || '',
    alt: image?.alt || '',
    title: image?.title || '',
    description: image?.description || '',
    category: image?.category || 'matches',
    edition: image?.edition || '',
    height: image?.height || 600,
    photographer: image?.photographer || '',
    date_taken: image?.date_taken || new Date().toISOString().split('T')[0],
    status: image?.status || 'draft',
    sort_order: image?.sort_order || 0,
    featured: image?.featured || false
  });

  const [loading, setLoading] = useState(false);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    const url = isEdit 
      ? route('admin.frontend.gallery.update', image?.id)
      : route('admin.frontend.gallery.store');
    
    const method = isEdit ? 'put' : 'post';

    router[method](url, form as any, {
      onSuccess: () => {
        toast.success(`Gallery image ${isEdit ? 'updated' : 'created'} successfully`);
        router.get(route('admin.frontend.gallery.index'));
      },
      onError: (errors) => {
        toast.error('Please check the form for errors');
      },
      onFinish: () => setLoading(false)
    });
  };

  const handleInputChange = (field: keyof GalleryImage, value: any) => {
    setForm(prev => ({ ...prev, [field]: value }));
  };

  return (
    <AppLayout>
      <Head title={`${isEdit ? 'Edit' : 'Create'} Gallery Image`} />
      
      <div className="space-y-6">
        <div className="flex items-center space-x-4">
          <Button variant="ghost" onClick={() => router.get(route('admin.frontend.gallery.index'))}>
            <ArrowLeft className="h-4 w-4 mr-2" />
            Back
          </Button>
          <div>
            <h1 className="text-2xl font-bold">{isEdit ? 'Edit' : 'Create'} Gallery Image</h1>
            <p className="text-muted-foreground">{isEdit ? 'Update the' : 'Add a new'} gallery image</p>
          </div>
        </div>

        <form onSubmit={handleSubmit}>
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            <div className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle>Image Upload</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <Label>Select Image *</Label>
                    <MediaManager
                      trigger={
                        <Button type="button" variant="outline" className="w-full">
                          {form.src ? 'Change Image' : 'Select Image'}
                        </Button>
                      }
                      onSelect={(files) => {
                        const file = Array.isArray(files) ? files[0] : files;
                        if (file) {
                          handleInputChange('src', file.url);
                          // Auto-fill alt text if empty
                          if (!form.alt) {
                            handleInputChange('alt', file.name || 'Gallery image');
                          }
                        }
                      }}
                      acceptedTypes={['image/*']}
                    />
                    {form.src && (
                      <div className="mt-4">
                        <img 
                          src={form.src} 
                          alt="Preview" 
                          className="w-full max-h-64 object-cover rounded border"
                        />
                      </div>
                    )}
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>Basic Information</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <Label htmlFor="title">Title *</Label>
                    <Input
                      id="title"
                      value={form.title}
                      onChange={(e) => handleInputChange('title', e.target.value)}
                      placeholder="Enter image title"
                      required
                    />
                  </div>

                  <div>
                    <Label htmlFor="alt">Alt Text *</Label>
                    <Input
                      id="alt"
                      value={form.alt}
                      onChange={(e) => handleInputChange('alt', e.target.value)}
                      placeholder="Describe the image for accessibility"
                      required
                    />
                  </div>

                  <div>
                    <Label htmlFor="description">Description</Label>
                    <Textarea
                      id="description"
                      value={form.description}
                      onChange={(e) => handleInputChange('description', e.target.value)}
                      placeholder="Brief description of the image"
                      rows={3}
                    />
                  </div>

                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <Label htmlFor="category">Category *</Label>
                      <Select value={form.category} onValueChange={(value) => handleInputChange('category', value)}>
                        <SelectTrigger>
                          <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                          {Object.entries(categories).map(([key, label]) => (
                            key !== 'all' && (
                              <SelectItem key={key} value={key}>{label}</SelectItem>
                            )
                          ))}
                        </SelectContent>
                      </Select>
                    </div>
                    <div>
                      <Label htmlFor="edition">Edition</Label>
                      <Input
                        id="edition"
                        value={form.edition}
                        onChange={(e) => handleInputChange('edition', e.target.value)}
                        placeholder="e.g., 2024, Season 1"
                      />
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>

            <div className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle>Photo Details</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <Label htmlFor="photographer">Photographer</Label>
                    <Input
                      id="photographer"
                      value={form.photographer}
                      onChange={(e) => handleInputChange('photographer', e.target.value)}
                      placeholder="Photographer name"
                    />
                  </div>

                  <div>
                    <Label htmlFor="date_taken">Date Taken</Label>
                    <Input
                      id="date_taken"
                      type="date"
                      value={form.date_taken}
                      onChange={(e) => handleInputChange('date_taken', e.target.value)}
                    />
                  </div>

                  <div>
                    <Label htmlFor="height">Image Height (px)</Label>
                    <Input
                      id="height"
                      type="number"
                      value={form.height}
                      onChange={(e) => handleInputChange('height', parseInt(e.target.value) || 0)}
                      placeholder="600"
                      min="100"
                    />
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>Display Settings</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <Label htmlFor="sort_order">Sort Order</Label>
                    <Input
                      id="sort_order"
                      type="number"
                      value={form.sort_order}
                      onChange={(e) => handleInputChange('sort_order', parseInt(e.target.value) || 0)}
                      placeholder="0"
                      min="0"
                    />
                    <p className="text-xs text-muted-foreground mt-1">
                      Lower numbers appear first
                    </p>
                  </div>

                  <div>
                    <Label htmlFor="status">Status *</Label>
                    <Select value={form.status} onValueChange={(value) => handleInputChange('status', value)}>
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="draft">Draft</SelectItem>
                        <SelectItem value="published">Published</SelectItem>
                        <SelectItem value="archived">Archived</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>

                  <div className="flex items-center space-x-2">
                    <Switch
                      id="featured"
                      checked={form.featured}
                      onCheckedChange={(checked) => handleInputChange('featured', checked)}
                    />
                    <Label htmlFor="featured">Featured Image</Label>
                  </div>
                </CardContent>
              </Card>

              <div className="flex gap-4">
                <Button 
                  type="button" 
                  variant="outline" 
                  onClick={() => router.get(route('admin.frontend.gallery.index'))}
                  disabled={loading}
                >
                  Cancel
                </Button>
                <Button 
                  type="submit" 
                  style={{ backgroundColor: '#256630' }}
                  className="hover:opacity-90 flex-1"
                  disabled={loading}
                >
                  <Save className="h-4 w-4 mr-2" />
                  {loading ? 'Saving...' : isEdit ? 'Update Image' : 'Create Image'}
                </Button>
              </div>
            </div>
          </div>
        </form>
      </div>
    </AppLayout>
  );
};

export default GalleryImageForm; 