import React, { useState } from 'react';
import { Head, router } from '@inertiajs/react';
import { route } from 'ziggy-js';
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Switch } from '@/components/ui/switch';
import { Badge } from '@/components/ui/badge';
import { ArrowLeft, Save, Plus, X, Image as ImageIcon } from 'lucide-react';
import MediaManager from '@/components/MediaManager';
import { toast } from 'react-hot-toast';

interface GalleryImage {
  id?: number;
  src: string;
  alt: string;
  title: string;
  description?: string;
  category: string;
  edition?: string;
  height: number;
  photographer?: string;
  date_taken?: string;
  status: string;
  sort_order: number;
  featured: boolean;
}

interface Props {
  categories: Record<string, string>;
  editions: Record<string, string>;
}

const GalleryImageBulkCreate = ({ categories, editions }: Props) => {
  const [selectedImages, setSelectedImages] = useState<GalleryImage[]>([]);
  const [commonSettings, setCommonSettings] = useState({
    category: 'matches',
    edition: '',
    height: 600,
    photographer: '',
    date_taken: new Date().toISOString().split('T')[0],
    status: 'draft',
    sort_order: 0,
    featured: false
  });
  const [loading, setLoading] = useState(false);

  const handleImageSelect = (files: any) => {
    const newImages = Array.isArray(files) ? files : [files];
    
    const imagesToAdd = newImages.map((file: any, index: number) => ({
      src: file.url,
      alt: file.name || `Gallery image ${selectedImages.length + index + 1}`,
      title: file.name || `Gallery image ${selectedImages.length + index + 1}`,
      description: '',
      category: commonSettings.category,
      edition: commonSettings.edition,
      height: commonSettings.height,
      photographer: commonSettings.photographer,
      date_taken: commonSettings.date_taken,
      status: commonSettings.status,
      sort_order: commonSettings.sort_order + selectedImages.length + index,
      featured: commonSettings.featured
    }));

    setSelectedImages(prev => [...prev, ...imagesToAdd]);
    toast.success(`Added ${newImages.length} image(s)`);
  };

  const removeImage = (index: number) => {
    setSelectedImages(prev => prev.filter((_, i) => i !== index));
  };

  const updateImage = (index: number, field: keyof GalleryImage, value: any) => {
    setSelectedImages(prev => prev.map((img, i) => 
      i === index ? { ...img, [field]: value } : img
    ));
  };

  const updateCommonSettings = (field: keyof typeof commonSettings, value: any) => {
    setCommonSettings(prev => ({ ...prev, [field]: value }));
    
    setSelectedImages(prev => prev.map(img => ({
      ...img,
      category: field === 'category' ? value : img.category,
      edition: field === 'edition' ? value : img.edition,
      height: field === 'height' ? value : img.height,
      photographer: field === 'photographer' ? value : img.photographer,
      date_taken: field === 'date_taken' ? value : img.date_taken,
      status: field === 'status' ? value : img.status,
      featured: field === 'featured' ? value : img.featured
    })));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (selectedImages.length === 0) {
      toast.error('Please select at least one image');
      return;
    }

    setLoading(true);

    router.post(route('admin.frontend.gallery.bulk-store'), {
      images: selectedImages as any
    }, {
      onSuccess: () => {
        toast.success(`Successfully created ${selectedImages.length} gallery images`);
        router.get(route('admin.frontend.gallery.index'));
      },
      onError: (errors) => {
        toast.error('Please check the form for errors');
      },
      onFinish: () => setLoading(false)
    });
  };

  return (
    <AppLayout>
      <Head title="Bulk Create Gallery Images" />
      
      <div className="space-y-6 p-4">
        <div className="flex items-center space-x-4">
          <Button variant="ghost" onClick={() => router.get(route('admin.frontend.gallery.index'))}>
            <ArrowLeft className="h-4 w-4 mr-2" />
            Back
          </Button>
          <div>
            <h1 className="text-2xl font-bold">Bulk Create Gallery Images</h1>
            <p className="text-muted-foreground">Select multiple images and set common properties</p>
          </div>
        </div>

        <form onSubmit={handleSubmit}>
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
            <div className="lg:col-span-1 space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle>Common Settings</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <Label>Select Images *</Label>
                    <MediaManager
                      trigger={
                        <Button type="button" variant="outline" className="w-full">
                          <Plus className="h-4 w-4 mr-2" />
                          Add Images
                        </Button>
                      }
                      onSelect={handleImageSelect}
                      multiple={true}
                      acceptedTypes={['image/*']}
                    />
                  </div>

                  <div>
                    <Label htmlFor="category">Category *</Label>
                    <Select value={commonSettings.category} onValueChange={(value) => updateCommonSettings('category', value)}>
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        {Object.entries(categories).map(([key, label]) => (
                          key !== 'all' && (
                            <SelectItem key={key} value={key}>{label}</SelectItem>
                          )
                        ))}
                      </SelectContent>
                    </Select>
                  </div>

                  <div>
                    <Label htmlFor="edition">Edition</Label>
                    <Select value={commonSettings.edition} onValueChange={(value) => updateCommonSettings('edition', value)}>
                      <SelectTrigger>
                        <SelectValue placeholder="Select edition" />
                      </SelectTrigger>
                      <SelectContent>
                        {Object.entries(editions).map(([key, label]) => (
                          key !== 'all' && (
                            <SelectItem key={key} value={key}>{label}</SelectItem>
                          )
                        ))}
                      </SelectContent>
                    </Select>
                  </div>

                  <div>
                    <Label htmlFor="height">Image Height (px)</Label>
                    <Input
                      id="height"
                      type="number"
                      value={commonSettings.height}
                      onChange={(e) => updateCommonSettings('height', parseInt(e.target.value) || 0)}
                      placeholder="600"
                      min="100"
                    />
                  </div>

                  <div>
                    <Label htmlFor="photographer">Photographer</Label>
                    <Input
                      id="photographer"
                      value={commonSettings.photographer}
                      onChange={(e) => updateCommonSettings('photographer', e.target.value)}
                      placeholder="Photographer name"
                    />
                  </div>

                  <div>
                    <Label htmlFor="date_taken">Date Taken</Label>
                    <Input
                      id="date_taken"
                      type="date"
                      value={commonSettings.date_taken}
                      onChange={(e) => updateCommonSettings('date_taken', e.target.value)}
                    />
                  </div>

                  <div>
                    <Label htmlFor="status">Status *</Label>
                    <Select value={commonSettings.status} onValueChange={(value) => updateCommonSettings('status', value)}>
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="draft">Draft</SelectItem>
                        <SelectItem value="published">Published</SelectItem>
                        <SelectItem value="archived">Archived</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>

                  <div>
                    <Label htmlFor="sort_order">Sort Order</Label>
                    <Input
                      id="sort_order"
                      type="number"
                      value={commonSettings.sort_order}
                      onChange={(e) => updateCommonSettings('sort_order', parseInt(e.target.value) || 0)}
                      placeholder="0"
                      min="0"
                    />
                  </div>

                  <div className="flex items-center space-x-2">
                    <Switch
                      id="featured"
                      checked={commonSettings.featured}
                      onCheckedChange={(checked) => updateCommonSettings('featured', checked)}
                    />
                    <Label htmlFor="featured">Featured Images</Label>
                  </div>
                </CardContent>
              </Card>

              <div className="flex gap-4">
                <Button 
                  type="button" 
                  variant="outline" 
                  onClick={() => router.get(route('admin.frontend.gallery.index'))}
                  disabled={loading}
                  className="flex-1"
                >
                  Cancel
                </Button>
                <Button 
                  type="submit" 
                  style={{ backgroundColor: '#256630' }}
                  className="hover:opacity-90 flex-1"
                  disabled={loading || selectedImages.length === 0}
                >
                  <Save className="h-4 w-4 mr-2" />
                  {loading ? 'Creating...' : `Create ${selectedImages.length} Images`}
                </Button>
              </div>
            </div>

            <div className="lg:col-span-2">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center justify-between">
                    Selected Images ({selectedImages.length})
                    {selectedImages.length > 0 && (
                      <Badge variant="secondary">
                        {selectedImages.length} image(s)
                      </Badge>
                    )}
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  {selectedImages.length === 0 ? (
                    <div className="text-center py-12 text-muted-foreground">
                      <ImageIcon className="h-12 w-12 mx-auto mb-4 opacity-50" />
                      <p>No images selected</p>
                      <p className="text-sm">Use the "Add Images" button to select images</p>
                    </div>
                  ) : (
                    <div className="space-y-4">
                      {selectedImages.map((image, index) => (
                        <Card key={index} className="p-4">
                          <div className="flex gap-4">
                            <div className="flex-shrink-0">
                              <img 
                                src={image.src} 
                                alt={image.alt}
                                className="w-20 h-20 object-cover rounded border"
                              />
                            </div>
                            <div className="flex-1 space-y-3">
                              <div className="flex items-start justify-between">
                                <div className="flex-1">
                                  <Input
                                    value={image.title}
                                    onChange={(e) => updateImage(index, 'title', e.target.value)}
                                    placeholder="Image title"
                                    className="mb-2"
                                  />
                                  <Input
                                    value={image.alt}
                                    onChange={(e) => updateImage(index, 'alt', e.target.value)}
                                    placeholder="Alt text for accessibility"
                                  />
                                </div>
                                <Button
                                  type="button"
                                  variant="ghost"
                                  size="sm"
                                  onClick={() => removeImage(index)}
                                  className="text-red-500 hover:text-red-700"
                                >
                                  <X className="h-4 w-4" />
                                </Button>
                              </div>
                              
                              <Textarea
                                value={image.description || ''}
                                onChange={(e) => updateImage(index, 'description', e.target.value)}
                                placeholder="Image description (optional)"
                                rows={2}
                              />
                              
                              <div className="flex items-center gap-2 text-sm text-muted-foreground">
                                <Badge variant="outline">{image.category}</Badge>
                                {image.edition && <Badge variant="outline">{image.edition}</Badge>}
                                <Badge variant={image.status === 'published' ? 'default' : 'secondary'}>
                                  {image.status}
                                </Badge>
                                {image.featured && <Badge variant="default">Featured</Badge>}
                              </div>
                            </div>
                          </div>
                        </Card>
                      ))}
                    </div>
                  )}
                </CardContent>
              </Card>
            </div>
          </div>
        </form>
      </div>
    </AppLayout>
  );
};

export default GalleryImageBulkCreate;