import React from 'react';
import { Head, Link, router } from '@inertiajs/react';
import { route } from 'ziggy-js';
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Plus, Edit, Trash2, Eye, EyeOff, ArrowUp, ArrowDown } from 'lucide-react';
import { toast } from 'react-hot-toast';
import Swal from 'sweetalert2';

interface TournamentHighlight {
  id: number;
  icon: string;
  title: string;
  description: string;
  value: string;
  color: string;
  sort_order: number;
  active: boolean;
  status?: string;
  created_at: string;
  updated_at: string;
}

interface Props {
  highlights: TournamentHighlight[];
  filters: {
    search?: string;
    status?: string;
    active?: string;
  };
}

const TournamentHighlightsIndex = ({ highlights, filters }: Props) => {
  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    const formData = new FormData(e.target as HTMLFormElement);
    router.get(route('admin.frontend.tournament-highlights.index'), Object.fromEntries(formData), {
      preserveState: true,
    });
  };

  const handleDelete = (highlight: TournamentHighlight) => {
    Swal.fire({
      title: 'Are you sure?',
      text: 'This action cannot be undone!',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonColor: '#c22b2b',
      cancelButtonColor: '#256630',
      confirmButtonText: 'Yes, delete it!'
    }).then((result) => {
      if (result.isConfirmed) {
        router.delete(route('admin.frontend.tournament-highlights.destroy', highlight.id), {
          onSuccess: () => {
            toast.success('Tournament highlight deleted successfully');
            Swal.fire('Deleted!', 'The tournament highlight has been deleted.', 'success');
          },
          onError: () => {
            toast.error('Failed to delete tournament highlight');
          }
        });
      }
    });
  };

  const handleToggleStatus = (highlight: TournamentHighlight) => {
    router.patch(route('admin.frontend.tournament-highlights.toggle-status', highlight.id), {}, {
      onSuccess: () => {
        toast.success('Tournament highlight status updated successfully');
      },
      onError: () => {
        toast.error('Failed to update tournament highlight status');
      }
    });
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'published': return 'bg-green-100 text-green-800';
      case 'draft': return 'bg-yellow-100 text-yellow-800';
      case 'archived': return 'bg-gray-100 text-gray-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getColorPreview = (color: string) => {
    return (
      <div 
        className="w-4 h-4 rounded-full border border-gray-300"
        style={{ backgroundColor: color.replace('text-', '').includes('-') ? '' : color }}
      />
    );
  };

  return (
    <AppLayout>
      <Head title="Tournament Highlights" />
      
      <div className="space-y-6">
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-2xl font-bold">Tournament Highlights</h1>
            <p className="text-muted-foreground">Manage tournament highlights for the frontend</p>
          </div>
          <Link href={route('admin.frontend.tournament-highlights.create')}>
            <Button style={{ backgroundColor: '#256630' }} className="hover:opacity-90">
              <Plus className="h-4 w-4 mr-2" />
              Add Highlight
            </Button>
          </Link>
        </div>

        <Card>
          <CardHeader>
            <CardTitle>Filters</CardTitle>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleSearch} className="flex gap-4">
              <div className="flex-1">
                <Input
                  name="search"
                  placeholder="Search highlights..."
                  defaultValue={filters?.search}
                />
              </div>
              <select
                name="status"
                className="px-3 py-2 border border-input rounded-md"
                defaultValue={filters?.status || 'all'}
              >
                <option value="all">All Status</option>
                <option value="draft">Draft</option>
                <option value="published">Published</option>
                <option value="archived">Archived</option>
              </select>
              <select
                name="active"
                className="px-3 py-2 border border-input rounded-md"
                defaultValue={filters?.active || 'all'}
              >
                <option value="all">All</option>
                <option value="true">Active</option>
                <option value="false">Inactive</option>
              </select>
              <Button type="submit" style={{ backgroundColor: '#256630' }} className="hover:opacity-90">
                Filter
              </Button>
            </form>
          </CardContent>
        </Card>

        <div className="grid gap-6">
          {highlights.map((highlight) => (
            <Card key={highlight.id} className="hover:shadow-md transition-shadow">
              <CardContent className="p-6">
                <div className="flex gap-6">
                  <div className="flex-shrink-0">
                    <div className="w-16 h-16 rounded-full bg-gray-100 flex items-center justify-center text-2xl">
                      {highlight.icon}
                    </div>
                  </div>
                  
                  <div className="flex-1">
                    <div className="flex justify-between items-start mb-3">
                      <div>
                        <div className="flex items-center gap-2 mb-1">
                          <h3 className="text-lg font-semibold">{highlight.title}</h3>
                          {highlight.active ? (
                            <Eye className="h-4 w-4 text-green-500" />
                          ) : (
                            <EyeOff className="h-4 w-4 text-red-500" />
                          )}
                        </div>
                        <p className="text-sm text-muted-foreground mb-2">
                          {highlight.description}
                        </p>
                        <div className="flex items-center gap-4">
                          <div className="flex items-center gap-2">
                            <span className="text-2xl font-bold" style={{ color: highlight.color }}>
                              {highlight.value}
                            </span>
                            {getColorPreview(highlight.color)}
                          </div>
                        </div>
                      </div>
                      
                      <div className="flex items-center space-x-2">
                        <Button
                          variant="ghost"
                          size="sm"
                          onClick={() => handleToggleStatus(highlight)}
                          title={highlight.active ? 'Deactivate' : 'Activate'}
                        >
                          {highlight.active ? <EyeOff className="h-4 w-4" /> : <Eye className="h-4 w-4" />}
                        </Button>
                        <Link href={route('admin.frontend.tournament-highlights.edit', highlight.id)}>
                          <Button variant="ghost" size="sm">
                            <Edit className="h-4 w-4" />
                          </Button>
                        </Link>
                        <Button
                          variant="ghost"
                          size="sm"
                          onClick={() => handleDelete(highlight)}
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      </div>
                    </div>
                    
                    <div className="flex items-center gap-4 text-xs text-muted-foreground">
                      <span>📊 Order: #{highlight.sort_order}</span>
                      {highlight.status && (
                        <Badge className={getStatusColor(highlight.status)}>
                          {highlight.status}
                        </Badge>
                      )}
                      <span>📅 {new Date(highlight.created_at).toLocaleDateString()}</span>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>

        {highlights.length === 0 && (
          <Card>
            <CardContent className="p-6 text-center">
              <p className="text-muted-foreground">No tournament highlights found</p>
            </CardContent>
          </Card>
        )}
      </div>
    </AppLayout>
  );
};

export default TournamentHighlightsIndex; 