import { Head, Link } from "@inertiajs/react";
import AppLayout from "@/layouts/app-layout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { ArrowLeft, Edit, HelpCircle, Users, Trophy, Calendar, Shield, Gift } from "lucide-react";

interface Faq {
  id: number;
  category: string;
  question: string;
  answer: string;
  sort_order: number;
  is_active: boolean;
  created_at: string;
  updated_at: string;
}

interface ShowFaqPageProps {
  faq: Faq;
}

const ShowFaq = ({ faq }: ShowFaqPageProps) => {
  const categoryIcons = {
    general: HelpCircle,
    registration: Users,
    tournament: Trophy,
    schedule: Calendar,
    prizes: Trophy,
    support: Shield,
  };

  const categoryColors = {
    general: 'bg-blue-100 text-blue-800',
    registration: 'bg-green-100 text-green-800',
    tournament: 'bg-yellow-100 text-yellow-800',
    schedule: 'bg-purple-100 text-purple-800',
    prizes: 'bg-red-100 text-red-800',
    support: 'bg-indigo-100 text-indigo-800',
  };

  const categoryNames = {
    general: 'General Information',
    registration: 'Team Registration',
    tournament: 'Tournament Rules',
    schedule: 'Schedule & Fixtures',
    prizes: 'Prizes & Awards',
    support: 'Support & Contact',
  };

  const getCategoryIcon = (category: string) => {
    const IconComponent = categoryIcons[category as keyof typeof categoryIcons] || HelpCircle;
    return <IconComponent className="w-4 h-4" />;
  };

  return (
    <AppLayout>
      <Head title={`FAQ - ${faq.question}`} />
      
      <div className="space-y-6">
        {/* Header */}
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-4">
            <Link href={route('admin.faqs.index')}>
              <Button variant="ghost" size="sm">
                <ArrowLeft className="w-4 h-4 mr-2" />
                Back to FAQs
              </Button>
            </Link>
            <div>
              <h1 className="text-3xl font-bold text-gray-900">FAQ Details</h1>
              <p className="text-gray-600 mt-1">View FAQ information and content</p>
            </div>
          </div>
          
          <Link href={route('admin.faqs.edit', faq.id)}>
            <Button className="flex items-center space-x-2">
              <Edit className="w-4 h-4" />
              <span>Edit FAQ</span>
            </Button>
          </Link>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          {/* Main Content */}
          <div className="lg:col-span-2 space-y-6">
            <Card>
              <CardHeader>
                <div className="flex items-center space-x-3">
                  <Badge className={`${categoryColors[faq.category as keyof typeof categoryColors]} flex items-center space-x-1`}>
                    {getCategoryIcon(faq.category)}
                    <span>{categoryNames[faq.category as keyof typeof categoryNames]}</span>
                  </Badge>
                  <Badge variant={faq.is_active ? "default" : "secondary"}>
                    {faq.is_active ? "Active" : "Inactive"}
                  </Badge>
                </div>
              </CardHeader>
              <CardContent>
                <div className="space-y-6">
                  <div>
                    <h2 className="text-2xl font-bold text-gray-900 mb-4">{faq.question}</h2>
                    <div className="prose max-w-none">
                      <p className="text-gray-700 leading-relaxed whitespace-pre-wrap">
                        {faq.answer}
                      </p>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>

          {/* Sidebar */}
          <div className="space-y-6">
            {/* FAQ Information */}
            <Card>
              <CardHeader>
                <CardTitle>FAQ Information</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  <div className="flex justify-between items-center">
                    <span className="text-sm text-gray-600">ID</span>
                    <span className="font-medium">#{faq.id}</span>
                  </div>
                  
                  <div className="flex justify-between items-center">
                    <span className="text-sm text-gray-600">Category</span>
                    <div className="flex items-center space-x-1">
                      {getCategoryIcon(faq.category)}
                      <span className="font-medium text-sm">
                        {categoryNames[faq.category as keyof typeof categoryNames]}
                      </span>
                    </div>
                  </div>
                  
                  <div className="flex justify-between items-center">
                    <span className="text-sm text-gray-600">Sort Order</span>
                    <span className="font-medium">{faq.sort_order}</span>
                  </div>
                  
                  <div className="flex justify-between items-center">
                    <span className="text-sm text-gray-600">Status</span>
                    <Badge variant={faq.is_active ? "default" : "secondary"}>
                      {faq.is_active ? "Active" : "Inactive"}
                    </Badge>
                  </div>
                  
                  <div className="flex justify-between items-center">
                    <span className="text-sm text-gray-600">Created</span>
                    <span className="font-medium text-sm">
                      {new Date(faq.created_at).toLocaleDateString()}
                    </span>
                  </div>
                  
                  <div className="flex justify-between items-center">
                    <span className="text-sm text-gray-600">Last Updated</span>
                    <span className="font-medium text-sm">
                      {new Date(faq.updated_at).toLocaleDateString()}
                    </span>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Actions */}
            <Card>
              <CardHeader>
                <CardTitle>Actions</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  <Link href={route('admin.faqs.edit', faq.id)} className="block">
                    <Button className="w-full flex items-center justify-center space-x-2">
                      <Edit className="w-4 h-4" />
                      <span>Edit FAQ</span>
                    </Button>
                  </Link>
                  
                  <Link href={route('admin.faqs.index')} className="block">
                    <Button variant="outline" className="w-full">
                      Back to List
                    </Button>
                  </Link>
                </div>
              </CardContent>
            </Card>

            {/* Frontend Preview */}
            <Card>
              <CardHeader>
                <CardTitle>Frontend Preview</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  <div className="flex items-center space-x-2">
                    {getCategoryIcon(faq.category)}
                    <span className="text-sm font-medium text-gray-700">
                      {categoryNames[faq.category as keyof typeof categoryNames]}
                    </span>
                  </div>
                  
                  <div className="border rounded-lg p-4 bg-gray-50">
                    <h4 className="font-semibold text-gray-900 mb-2">
                      {faq.question}
                    </h4>
                    <p className="text-sm text-gray-600 line-clamp-3">
                      {faq.answer}
                    </p>
                  </div>
                  
                  <p className="text-xs text-gray-500">
                    This is how the FAQ will appear on the frontend website.
                  </p>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </AppLayout>
  );
};

export default ShowFaq;
