import { useState } from "react";
import { Head, Link, router } from "@inertiajs/react";
import AppLayout from "@/layouts/app-layout";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Checkbox } from "@/components/ui/checkbox";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger } from "@/components/ui/dropdown-menu";
import { Plus, Search, Filter, MoreHorizontal, Edit, Trash2, Eye, ToggleLeft, ToggleRight, HelpCircle, Users, Trophy, Calendar, Shield, Gift } from "lucide-react";
import { Dialog, DialogFooter, DialogHeader, DialogTitle, DialogContent, DialogClose, DialogDescription } from "@/components/ui/dialog";

interface Faq {
  id: number;
  category: string;
  question: string;
  answer: string;
  sort_order: number;
  is_active: boolean;
  created_at: string;
  updated_at: string;
}

interface FaqsPageProps {
  faqs: {
    data: Faq[];
    links: any[];
    meta: any;
  };
  filters: {
    search: string;
    category: string;
    status: string;
  };
  categories: Record<string, string>;
}

const FaqsIndex = ({ faqs, filters, categories }: FaqsPageProps) => {
  const [searchTerm, setSearchTerm] = useState(filters.search || '');
  const [selectedCategory, setSelectedCategory] = useState(filters.category || '');
  const [selectedStatus, setSelectedStatus] = useState(filters.status || '');
  const [selectedFaqs, setSelectedFaqs] = useState<number[]>([]);
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [faqToDelete, setFaqToDelete] = useState<Faq | null>(null);
  const [bulkActionDialogOpen, setBulkActionDialogOpen] = useState(false);
  const [bulkAction, setBulkAction] = useState<string>('');

  const categoryIcons = {
    general: HelpCircle,
    registration: Users,
    tournament: Trophy,
    schedule: Calendar,
    prizes: Trophy,
    support: Shield,
  };

  const categoryColors = {
    general: 'bg-blue-100 text-blue-800',
    registration: 'bg-green-100 text-green-800',
    tournament: 'bg-yellow-100 text-yellow-800',
    schedule: 'bg-purple-100 text-purple-800',
    prizes: 'bg-red-100 text-red-800',
    support: 'bg-indigo-100 text-indigo-800',
  };

  const handleSearch = () => {
    router.get(route('admin.faqs.index'), {
      search: searchTerm,
      category: selectedCategory,
      status: selectedStatus,
    }, {
      preserveState: true,
      replace: true,
    });
  };

  const handleClearFilters = () => {
    setSearchTerm('');
    setSelectedCategory('');
    setSelectedStatus('');
    router.get(route('admin.faqs.index'), {}, {
      preserveState: true,
      replace: true,
    });
  };

  const handleSelectAll = () => {
    if (selectedFaqs.length === faqs.data.length) {
      setSelectedFaqs([]);
    } else {
      setSelectedFaqs(faqs.data.map(faq => faq.id));
    }
  };

  const handleSelectFaq = (faqId: number) => {
    setSelectedFaqs(prev => 
      prev.includes(faqId) 
        ? prev.filter(id => id !== faqId)
        : [...prev, faqId]
    );
  };

  const handleDelete = (faq: Faq) => {
    setFaqToDelete(faq);
    setDeleteDialogOpen(true);
  };

  const confirmDelete = () => {
    if (faqToDelete) {
      router.delete(route('admin.faqs.destroy', faqToDelete.id), {
        onSuccess: () => {
          setDeleteDialogOpen(false);
          setFaqToDelete(null);
        },
      });
    }
  };

  const handleToggleStatus = (faq: Faq) => {
    router.patch(route('admin.faqs.toggle-status', faq.id), {}, {
      preserveState: true,
    });
  };

  const handleBulkAction = () => {
    if (selectedFaqs.length === 0 || !bulkAction) return;

    router.post(route('admin.faqs.bulk-action'), {
      action: bulkAction,
      faq_ids: selectedFaqs,
    }, {
      onSuccess: () => {
        setSelectedFaqs([]);
        setBulkActionDialogOpen(false);
        setBulkAction('');
      },
    });
  };

  const getCategoryIcon = (category: string) => {
    const IconComponent = categoryIcons[category as keyof typeof categoryIcons] || HelpCircle;
    return <IconComponent className="w-4 h-4" />;
  };

  return (
    <AppLayout>
      <Head title="FAQ Management" />
      
      <div className="space-y-6">
        {/* Header */}
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-3xl font-bold text-gray-900">FAQ Management</h1>
            <p className="text-gray-600 mt-1">Manage frequently asked questions for the frontend</p>
          </div>
          <Link href={route('admin.faqs.create')}>
            <Button className="flex items-center space-x-2">
              <Plus className="w-4 h-4" />
              <span>Add FAQ</span>
            </Button>
          </Link>
        </div>

        {/* Filters */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center space-x-2">
              <Filter className="w-5 h-5" />
              <span>Filters</span>
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Search</label>
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                  <Input
                    placeholder="Search questions or answers..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="pl-10"
                    onKeyPress={(e) => e.key === 'Enter' && handleSearch()}
                  />
                </div>
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Category</label>
                <Select value={selectedCategory} onValueChange={setSelectedCategory}>
                  <SelectTrigger>
                    <SelectValue placeholder="All categories" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All categories</SelectItem>
                    {Object.entries(categories).map(([key, value]) => (
                      <SelectItem key={key} value={key}>
                        {value}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Status</label>
                <Select value={selectedStatus} onValueChange={setSelectedStatus}>
                  <SelectTrigger>
                    <SelectValue placeholder="All statuses" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All statuses</SelectItem>
                    <SelectItem value="active">Active</SelectItem>
                    <SelectItem value="inactive">Inactive</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              
              <div className="flex items-end space-x-2">
                <Button onClick={handleSearch} className="flex-1">
                  Apply Filters
                </Button>
                <Button variant="outline" onClick={handleClearFilters}>
                  Clear
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Bulk Actions */}
        {selectedFaqs.length > 0 && (
          <Card>
            <CardContent className="pt-6">
              <div className="flex items-center justify-between">
                <div className="flex items-center space-x-4">
                  <span className="text-sm text-gray-600">
                    {selectedFaqs.length} FAQ{selectedFaqs.length !== 1 ? 's' : ''} selected
                  </span>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setBulkActionDialogOpen(true)}
                  >
                    Bulk Actions
                  </Button>
                </div>
                <Button
                  variant="ghost"
                  size="sm"
                  onClick={() => setSelectedFaqs([])}
                >
                  Clear Selection
                </Button>
              </div>
            </CardContent>
          </Card>
        )}

        {/* FAQ List */}
        <Card>
          <CardHeader>
            <div className="flex items-center justify-between">
              <CardTitle>FAQs ({faqs?.meta?.total})</CardTitle>
              <div className="flex items-center space-x-2">
                <Checkbox
                  checked={selectedFaqs.length === faqs?.data?.length && faqs?.data?.length > 0}
                  onCheckedChange={handleSelectAll}
                />
                <span className="text-sm text-gray-600">Select All</span>
              </div>
            </div>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {faqs.data.map((faq) => (
                <div key={faq.id} className="border rounded-lg p-4 hover:bg-gray-50 transition-colors">
                  <div className="flex items-start space-x-4">
                    <Checkbox
                      checked={selectedFaqs.includes(faq.id)}
                      onCheckedChange={() => handleSelectFaq(faq.id)}
                    />
                    
                    <div className="flex-1 min-w-0">
                      <div className="flex items-center space-x-3 mb-2">
                        <Badge className={`${categoryColors[faq.category as keyof typeof categoryColors]} flex items-center space-x-1`}>
                          {getCategoryIcon(faq.category)}
                          <span>{categories[faq.category]}</span>
                        </Badge>
                        <Badge variant={faq.is_active ? "default" : "secondary"}>
                          {faq.is_active ? "Active" : "Inactive"}
                        </Badge>
                        <span className="text-sm text-gray-500">Order: {faq.sort_order}</span>
                      </div>
                      
                      <h3 className="text-lg font-semibold text-gray-900 mb-2">{faq.question}</h3>
                      <p className="text-gray-600 text-sm line-clamp-2 mb-3">{faq.answer}</p>
                      
                      <div className="flex items-center space-x-4 text-xs text-gray-500">
                        <span>Created: {new Date(faq.created_at).toLocaleDateString()}</span>
                        <span>Updated: {new Date(faq.updated_at).toLocaleDateString()}</span>
                      </div>
                    </div>
                    
                    <div className="flex items-center space-x-2">
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => handleToggleStatus(faq)}
                        className="flex items-center space-x-1"
                      >
                        {faq.is_active ? (
                          <ToggleRight className="w-4 h-4 text-green-600" />
                        ) : (
                          <ToggleLeft className="w-4 h-4 text-gray-400" />
                        )}
                      </Button>
                      
                      <DropdownMenu>
                        <DropdownMenuTrigger asChild>
                          <Button variant="ghost" size="sm">
                            <MoreHorizontal className="w-4 h-4" />
                          </Button>
                        </DropdownMenuTrigger>
                        <DropdownMenuContent align="end">
                          <DropdownMenuItem asChild>
                            <Link href={route('admin.faqs.show', faq.id)} className="flex items-center space-x-2">
                              <Eye className="w-4 h-4" />
                              <span>View</span>
                            </Link>
                          </DropdownMenuItem>
                          <DropdownMenuItem asChild>
                            <Link href={route('admin.faqs.edit', faq.id)} className="flex items-center space-x-2">
                              <Edit className="w-4 h-4" />
                              <span>Edit</span>
                            </Link>
                          </DropdownMenuItem>
                          <DropdownMenuItem
                            onClick={() => handleDelete(faq)}
                            className="flex items-center space-x-2 text-red-600"
                          >
                            <Trash2 className="w-4 h-4" />
                            <span>Delete</span>
                          </DropdownMenuItem>
                        </DropdownMenuContent>
                      </DropdownMenu>
                    </div>
                  </div>
                </div>
              ))}
              
              {faqs.data.length === 0 && (
                <div className="text-center py-12">
                  <HelpCircle className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                  <h3 className="text-lg font-semibold text-gray-900 mb-2">No FAQs found</h3>
                  <p className="text-gray-600 mb-4">Get started by creating your first FAQ.</p>
                  <Link href={route('admin.faqs.create')}>
                    <Button>Create FAQ</Button>
                  </Link>
                </div>
              )}
            </div>
          </CardContent>
        </Card>

        {/* Pagination */}
        {faqs.links && faqs.links.length > 3 && (
          <div className="flex justify-center">
            <nav className="flex space-x-2">
              {faqs.links.map((link, index) => (
                <Link
                  key={index}
                  href={link.url || '#'}
                  className={`px-3 py-2 rounded-md text-sm ${
                    link.active
                      ? 'bg-blue-600 text-white'
                      : 'bg-white text-gray-700 hover:bg-gray-50 border'
                  } ${!link.url ? 'opacity-50 cursor-not-allowed' : ''}`}
                  dangerouslySetInnerHTML={{ __html: link.label }}
                />
              ))}
            </nav>
          </div>
        )}
      </div>

      {/* Delete Confirmation Dialog */}
      <Dialog open={deleteDialogOpen} onOpenChange={setDeleteDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Delete FAQ</DialogTitle>
            <DialogDescription>
              Are you sure you want to delete this FAQ? This action cannot be undone.
            </DialogDescription>
          </DialogHeader>
          <DialogFooter>
            <DialogClose>Cancel</DialogClose>
            <DialogClose onClick={confirmDelete} className="bg-red-600 hover:bg-red-700">
              Delete
            </DialogClose>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Bulk Action Dialog */}
      <Dialog open={bulkActionDialogOpen} onOpenChange={setBulkActionDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Bulk Action</DialogTitle>
            <DialogDescription>
              Choose an action to perform on {selectedFaqs.length} selected FAQ{selectedFaqs.length !== 1 ? 's' : ''}.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <Select value={bulkAction} onValueChange={setBulkAction}>
              <SelectTrigger>
                <SelectValue placeholder="Select action" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="activate">Activate</SelectItem>
                <SelectItem value="deactivate">Deactivate</SelectItem>
                <SelectItem value="delete">Delete</SelectItem>
              </SelectContent>
            </Select>
          </div>
          <DialogFooter>
        <DialogClose>Cancel</DialogClose>
            <DialogClose 
              onClick={handleBulkAction}
              disabled={!bulkAction}
              className={bulkAction === 'delete' ? 'bg-red-600 hover:bg-red-700' : ''}
            >
              {bulkAction === 'activate' ? 'Activate' : 
               bulkAction === 'deactivate' ? 'Deactivate' : 
               bulkAction === 'delete' ? 'Delete' : 'Confirm'}
            </DialogClose>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </AppLayout>
  );
};

export default FaqsIndex;
