import { useState } from "react";
import { Head, Link, useForm } from "@inertiajs/react";
import AppLayout from "@/layouts/app-layout";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { ArrowLeft, Save, HelpCircle, Users, Trophy, Calendar, Shield, Gift } from "lucide-react";

interface Faq {
  id: number;
  category: string;
  question: string;
  answer: string;
  sort_order: number;
  is_active: boolean;
  created_at: string;
  updated_at: string;
}

interface EditFaqPageProps {
  faq: Faq;
  categories: Record<string, string>;
}

const EditFaq = ({ faq, categories }: EditFaqPageProps) => {
  const { data, setData, put, processing, errors } = useForm({
    category: faq.category,
    question: faq.question,
    answer: faq.answer,
    sort_order: faq.sort_order,
    is_active: faq.is_active,
  });

  const categoryIcons = {
    general: HelpCircle,
    registration: Users,
    tournament: Trophy,
    schedule: Calendar,
    prizes: Trophy,
    support: Shield,
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    put(route('admin.faqs.update', faq.id));
  };

  const getCategoryIcon = (category: string) => {
    const IconComponent = categoryIcons[category as keyof typeof categoryIcons] || HelpCircle;
    return <IconComponent className="w-4 h-4" />;
  };

  return (
    <AppLayout>
      <Head title={`Edit FAQ - ${faq.question}`} />
      
      <div className="space-y-6">
        {/* Header */}
        <div className="flex items-center space-x-4">
          <Link href={route('admin.faqs.index')}>
            <Button variant="ghost" size="sm">
              <ArrowLeft className="w-4 h-4 mr-2" />
              Back to FAQs
            </Button>
          </Link>
          <div>
            <h1 className="text-3xl font-bold text-gray-900">Edit FAQ</h1>
            <p className="text-gray-600 mt-1">Update the frequently asked question</p>
          </div>
        </div>

        <form onSubmit={handleSubmit} className="space-y-6">
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
            {/* Main Form */}
            <div className="lg:col-span-2 space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle>FAQ Details</CardTitle>
                </CardHeader>
                <CardContent className="space-y-6">
                  {/* Category */}
                  <div className="space-y-2">
                    <Label htmlFor="category">Category</Label>
                    <Select value={data.category} onValueChange={(value) => setData('category', value)}>
                      <SelectTrigger>
                        <SelectValue placeholder="Select a category" />
                      </SelectTrigger>
                      <SelectContent>
                        {Object.entries(categories).map(([key, value]) => (
                          <SelectItem key={key} value={key}>
                            <div className="flex items-center space-x-2">
                              {getCategoryIcon(key)}
                              <span>{value}</span>
                            </div>
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    {errors.category && (
                      <p className="text-sm text-red-600">{errors.category}</p>
                    )}
                  </div>

                  {/* Question */}
                  <div className="space-y-2">
                    <Label htmlFor="question">Question</Label>
                    <Input
                      id="question"
                      value={data.question}
                      onChange={(e) => setData('question', e.target.value)}
                      placeholder="Enter the FAQ question..."
                      className={errors.question ? 'border-red-500' : ''}
                    />
                    {errors.question && (
                      <p className="text-sm text-red-600">{errors.question}</p>
                    )}
                  </div>

                  {/* Answer */}
                  <div className="space-y-2">
                    <Label htmlFor="answer">Answer</Label>
                    <Textarea
                      id="answer"
                      value={data.answer}
                      onChange={(e) => setData('answer', e.target.value)}
                      placeholder="Enter the detailed answer..."
                      rows={8}
                      className={errors.answer ? 'border-red-500' : ''}
                    />
                    {errors.answer && (
                      <p className="text-sm text-red-600">{errors.answer}</p>
                    )}
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* Sidebar */}
            <div className="space-y-6">
              {/* Settings */}
              <Card>
                <CardHeader>
                  <CardTitle>Settings</CardTitle>
                </CardHeader>
                <CardContent className="space-y-6">
                  {/* Sort Order */}
                  <div className="space-y-2">
                    <Label htmlFor="sort_order">Sort Order</Label>
                    <Input
                      id="sort_order"
                      type="number"
                      min="0"
                      value={data.sort_order}
                      onChange={(e) => setData('sort_order', parseInt(e.target.value) || 0)}
                      placeholder="0"
                    />
                    <p className="text-xs text-gray-500">
                      Lower numbers appear first. Leave as 0 for automatic ordering.
                    </p>
                    {errors.sort_order && (
                      <p className="text-sm text-red-600">{errors.sort_order}</p>
                    )}
                  </div>

                  {/* Active Status */}
                  <div className="flex items-center justify-between">
                    <div className="space-y-1">
                      <Label htmlFor="is_active">Active</Label>
                      <p className="text-xs text-gray-500">
                        Show this FAQ on the frontend
                      </p>
                    </div>
                    <Switch
                      id="is_active"
                      checked={data.is_active}
                      onCheckedChange={(checked) => setData('is_active', checked)}
                    />
                  </div>
                </CardContent>
              </Card>

              {/* FAQ Info */}
              <Card>
                <CardHeader>
                  <CardTitle>FAQ Information</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3 text-sm">
                    <div className="flex justify-between">
                      <span className="text-gray-600">ID:</span>
                      <span className="font-medium">#{faq.id}</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-gray-600">Created:</span>
                      <span className="font-medium">
                        {new Date(faq.created_at).toLocaleDateString()}
                      </span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-gray-600">Last Updated:</span>
                      <span className="font-medium">
                        {new Date(faq.updated_at).toLocaleDateString()}
                      </span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-gray-600">Status:</span>
                      <span className={`font-medium ${faq.is_active ? 'text-green-600' : 'text-gray-500'}`}>
                        {faq.is_active ? 'Active' : 'Inactive'}
                      </span>
                    </div>
                  </div>
                </CardContent>
              </Card>

              {/* Preview */}
              <Card>
                <CardHeader>
                  <CardTitle>Preview</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    <div className="flex items-center space-x-2">
                      {getCategoryIcon(data.category)}
                      <span className="text-sm font-medium text-gray-700">
                        {categories[data.category]}
                      </span>
                    </div>
                    
                    {data.question && (
                      <div>
                        <h4 className="font-semibold text-gray-900 mb-2">
                          {data.question}
                        </h4>
                        <p className="text-sm text-gray-600 line-clamp-3">
                          {data.answer || 'Answer will appear here...'}
                        </p>
                      </div>
                    )}
                    
                    {!data.question && (
                      <p className="text-sm text-gray-500 italic">
                        Enter a question to see preview
                      </p>
                    )}
                  </div>
                </CardContent>
              </Card>

              {/* Actions */}
              <Card>
                <CardContent className="pt-6">
                  <div className="space-y-3">
                    <Button
                      type="submit"
                      disabled={processing}
                      className="w-full flex items-center justify-center space-x-2"
                    >
                      <Save className="w-4 h-4" />
                      <span>{processing ? 'Updating...' : 'Update FAQ'}</span>
                    </Button>
                    
                    <Link href={route('admin.faqs.index')} className="block">
                      <Button variant="outline" className="w-full">
                        Cancel
                      </Button>
                    </Link>
                  </div>
                </CardContent>
              </Card>
            </div>
          </div>
        </form>
      </div>
    </AppLayout>
  );
};

export default EditFaq;
