import AppLayout from '@/layouts/app-layout';
import { Head, router } from '@inertiajs/react';
import { Card } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { useState } from 'react';
import { Trophy, Users, BarChart3, Settings, Play, ArrowRight, Eye } from 'lucide-react';
import EnhancedBracketVisualization from '@/components/ui/enhanced-bracket-visualization';

export default function AdminBracketShow({ bracket, tournament, matches, participants, stats }) {
  const [currentRound, setCurrentRound] = useState(1);
  const [isAdvanceModalOpen, setIsAdvanceModalOpen] = useState(false);
  const [advanceForm, setAdvanceForm] = useState({
    current_round: 1,
  });

  // Handle round advancement
  const handleAdvanceRound = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    router.post(route('admin.brackets.advance', bracket.id), advanceForm, {
      onSuccess: () => {
        setIsAdvanceModalOpen(false);
        // Refresh the page to show updated bracket
        router.reload();
      },
    });
  };

  // Get bracket type badge color
  const getBracketTypeColor = (type: string) => {
    switch (type) {
      case 'winners': return 'bg-green-100 text-green-800';
      case 'losers': return 'bg-red-100 text-red-800';
      case 'final': return 'bg-yellow-100 text-yellow-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  // Get tournament type badge color
  const getTournamentTypeColor = (type: string) => {
    switch (type) {
      case 'knockout': return 'bg-blue-100 text-blue-800';
      case 'double-elimination': return 'bg-purple-100 text-purple-800';
      case 'round-robin': return 'bg-orange-100 text-orange-800';
      case 'group-knockout': return 'bg-indigo-100 text-indigo-800';
      case 'swiss_system': return 'bg-pink-100 text-pink-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  // Get structure type display name
  const getStructureTypeDisplay = (type: string) => {
    switch (type) {
      case 'single_elimination': return 'Single Elimination';
      case 'double_elimination': return 'Double Elimination';
      case 'round_robin': return 'Round Robin';
      case 'group_knockout': return 'Group Stage + Knockout';
      case 'swiss_system': return 'Swiss System';
      default: return type?.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase()) || 'Unknown';
    }
  };

  return (
    <AppLayout breadcrumbs={[
      { title: 'Brackets', href: '/admin/brackets' },
      { title: bracket.name, href: `/admin/brackets/${bracket.id}` }
    ]}> 
      <Head title={`Bracket: ${bracket.name}`} />
      <div className="flex flex-col gap-4 p-4">
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-2xl font-bold flex items-center gap-2">
              <Trophy className="w-6 h-6" />
              {bracket.name}
            </h1>
            <div className="flex items-center gap-2 mt-2">
              <Badge className={getBracketTypeColor(bracket.type)}>
                {bracket.type.replace('-', ' ')}
              </Badge>
              <Badge className={getTournamentTypeColor(tournament.type)}>
                {tournament.type.replace('-', ' ')}
              </Badge>
              <Badge variant={bracket.is_active ? 'default' : 'secondary'}>
                {bracket.is_active ? 'Active' : 'Inactive'}
              </Badge>
            </div>
          </div>
          <div className="flex gap-2">
            <Dialog open={isAdvanceModalOpen} onOpenChange={setIsAdvanceModalOpen}>
              <DialogTrigger asChild>
                <Button variant="outline">
                  <ArrowRight className="w-4 h-4 mr-1" />
                  Advance Round
                </Button>
              </DialogTrigger>
              <DialogContent>
                <DialogHeader>
                  <DialogTitle>Advance to Next Round</DialogTitle>
                </DialogHeader>
                <form onSubmit={handleAdvanceRound} className="space-y-4">
                  <div>
                    <Label htmlFor="current_round">Current Round</Label>
                    <Select 
                      value={advanceForm.current_round.toString()} 
                      onValueChange={(value) => setAdvanceForm(prev => ({ ...prev, current_round: parseInt(value) }))}
                    >
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        {Array.from({ length: bracket.rounds || 1 }, (_, i) => (
                          <SelectItem key={i + 1} value={(i + 1).toString()}>
                            Round {i + 1}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>
                  <div className="flex justify-end gap-2">
                    <Button type="button" variant="outline" onClick={() => setIsAdvanceModalOpen(false)}>
                      Cancel
                    </Button>
                    <Button type="submit">
                      <Play className="w-4 h-4 mr-1" />
                      Advance Round
                    </Button>
                  </div>
                </form>
              </DialogContent>
            </Dialog>
            
            <Button variant="outline" onClick={() => router.get(route('admin.brackets.edit', bracket.id))}>
              <Settings className="w-4 h-4 mr-1" />
              Edit Bracket
            </Button>
            <Button variant="outline" onClick={() => router.get(route('admin.brackets.index'))}>
              <Eye className="w-4 h-4 mr-1" />
              Back to Brackets
            </Button>
          </div>
        </div>

        {/* Bracket Information */}
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          <Card className="p-4">
            <h3 className="text-lg font-semibold mb-3 flex items-center gap-2">
              <Trophy className="w-5 h-5" />
              Bracket Details
            </h3>
            <div className="space-y-2">
              <div className="flex justify-between">
                <span className="text-gray-600">Type:</span>
                <span className="font-medium">{getStructureTypeDisplay(bracket.structure?.type)}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-600">Rounds:</span>
                <span className="font-medium">{bracket.rounds || 0}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-600">Participants:</span>
                <span className="font-medium">{bracket.structure?.total_participants || 0}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-600">Total Matches:</span>
                <span className="font-medium">{bracket.structure?.total_matches || 0}</span>
              </div>
              {bracket.structure?.byes && (
                <div className="flex justify-between">
                  <span className="text-gray-600">Byes:</span>
                  <span className="font-medium">{bracket.structure.byes}</span>
                </div>
              )}
            </div>
          </Card>

          <Card className="p-4">
            <h3 className="text-lg font-semibold mb-3 flex items-center gap-2">
              <BarChart3 className="w-5 h-5" />
              Match Statistics
            </h3>
            <div className="space-y-2">
              <div className="flex justify-between">
                <span className="text-gray-600">Total Matches:</span>
                <span className="font-medium">{stats.total_matches}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-600">Completed:</span>
                <span className="font-medium text-green-600">{stats.completed_matches}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-600">In Progress:</span>
                <span className="font-medium text-blue-600">{stats.in_progress_matches}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-600">Scheduled:</span>
                <span className="font-medium text-gray-600">{stats.scheduled_matches}</span>
                </div>
                </div>
          </Card>

          <Card className="p-4">
            <h3 className="text-lg font-semibold mb-3 flex items-center gap-2">
              <Users className="w-5 h-5" />
              Tournament Info
            </h3>
            <div className="space-y-2">
              <div className="flex justify-between">
                <span className="text-gray-600">Tournament:</span>
                <span className="font-medium">{tournament.name}</span>
                </div>
              <div className="flex justify-between">
                <span className="text-gray-600">Format:</span>
                <span className="font-medium">{tournament.format}</span>
                </div>
              <div className="flex justify-between">
                <span className="text-gray-600">Type:</span>
                <span className="font-medium">{tournament.type}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-600">Status:</span>
                <Badge variant={tournament.status === 'active' ? 'default' : 'secondary'}>
                  {tournament.status}
                </Badge>
                </div>
              </div>
            </Card>
          </div>

        {/* Seeding Information */}
        {bracket.seeding && (
          <Card className="p-4">
            <h3 className="text-lg font-semibold mb-3">Seeding Information</h3>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <Label className="text-sm font-medium text-gray-600">Seeding Method</Label>
                <p className="font-medium">{bracket.seeding.method || 'Unknown'}</p>
              </div>
              <div>
                <Label className="text-sm font-medium text-gray-600">Participants Count</Label>
                <p className="font-medium">{bracket.seeding.participants?.length || 0}</p>
          </div>
        </div>
          </Card>
        )}

        {/* Enhanced Bracket Visualization */}
        <Card className="p-6">
          <EnhancedBracketVisualization
            bracket={bracket}
            matches={matches}
            players={tournament.format === 'individual' ? participants : []}
            teams={tournament.format === 'team' ? participants : []}
            onMatchUpdate={(matchId, data) => {
              // Handle match updates
              console.log('Match update:', matchId, data);
            }}
            onAdvanceRound={(round) => {
              setAdvanceForm({ current_round: round });
              setIsAdvanceModalOpen(true);
            }}
            showControls={true}
          />
        </Card>
      </div>
    </AppLayout>
  );
} 