import AppLayout from '@/layouts/app-layout';
import { Head, router } from '@inertiajs/react';
import { Card } from '@/components/ui/card';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Label } from '@/components/ui/label';
import { Badge } from '@/components/ui/badge';
import { useState } from 'react';
import { Eye, Pencil, Plus, Settings, Trophy, Users, BarChart3, Play } from 'lucide-react';
import { Trash } from 'lucide-react';

export default function AdminBrackets({ brackets, filters, tournaments }: { brackets: any, filters: any, tournaments: any }) {
  const [isCreateModalOpen, setIsCreateModalOpen] = useState(false);
  const [isGenerateModalOpen, setIsGenerateModalOpen] = useState(false);
  const [editingBracket, setEditingBracket] = useState(null);
  const [selectedBrackets, setSelectedBrackets] = useState([]);
  const [isDeleteModalOpen, setIsDeleteModalOpen] = useState(false);
  const [isMultipleDeleteModalOpen, setIsMultipleDeleteModalOpen] = useState(false);
  const [bracketToDelete, setBracketToDelete] = useState(null);
  const [localFilters, setLocalFilters] = useState({
    search: filters?.search || '',
    type: filters?.type || 'all',
    tournament_id: filters?.tournament_id || '',
  });

  // Create form state
  const [form, setForm] = useState({
    name: '',
    type: '',
    tournament_id: '',
  });
  const [formErrors, setFormErrors] = useState({});

  // Generate bracket form state
  const [generateForm, setGenerateForm] = useState({
    tournament_id: '',
    bracket_name: '',
    bracket_type: 'winners',
    seeding_method: 'balanced',
    participants: [],
  });
  const [generateFormErrors, setGenerateFormErrors] = useState({});
  const [availableParticipants, setAvailableParticipants] = useState([]);
  const [selectedTournament, setSelectedTournament] = useState(null);

  // Handle filter form submit
  const handleFilterSubmit = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    router.get(route('admin.brackets.index'), localFilters, {
      preserveState: true,
      replace: true,
    });
  };

  // Handle create form submit
  const handleCreateSubmit = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    setFormErrors({});
    
    router.post(route('admin.brackets.store'), form, {
      onSuccess: () => {
        setIsCreateModalOpen(false);
        setForm({ name: '', type: '', tournament_id: '' });
      },
      onError: (errors) => {
        setFormErrors(errors);
      },
    });
  };

  // Handle generate bracket form submit
  const handleGenerateSubmit = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    setGenerateFormErrors({});
    
    router.post(route('admin.brackets.generate', generateForm.tournament_id), generateForm, {
      onSuccess: () => {
        setIsGenerateModalOpen(false);
        setGenerateForm({ 
          tournament_id: '', 
          bracket_name: '', 
          bracket_type: 'winners', 
          seeding_method: 'balanced',
          participants: []
        });
        setAvailableParticipants([]);
        setSelectedTournament(null);
      },
      onError: (errors) => {
        setGenerateFormErrors(errors);
      },
    });
  };

  // Load participants for selected tournament
  const loadParticipants = async (tournamentId: string) => {
    if (!tournamentId) return;
    
    try {
      const response = await fetch(route('admin.brackets.participants', tournamentId));
      const data = await response.json();
      setAvailableParticipants(data.participants);
      setSelectedTournament(data.tournament);
    } catch (error) {
      console.error('Failed to load participants:', error);
    }
  };

  // Handle tournament selection for bracket generation
  const handleTournamentSelect = (tournamentId: string) => {
    setGenerateForm(prev => ({ ...prev, tournament_id: tournamentId }));
    loadParticipants(tournamentId);
  };

  // Handle participant selection
  const handleParticipantToggle = (participant: any) => {
    setGenerateForm(prev => {
      const isSelected = prev.participants.some(p => p.id === participant.id);
      if (isSelected) {
        return {
          ...prev,
          participants: prev.participants.filter(p => p.id !== participant.id)
        };
      } else {
        return {
          ...prev,
          participants: [...prev.participants, participant]
        };
      }
    });
  };

  // Handle bracket selection
  const handleBracketSelect = (bracketId: number, checked: boolean) => {
    if (checked) {
      setSelectedBrackets(prev => [...prev, bracketId]);
    } else {
      setSelectedBrackets(prev => prev.filter(id => id !== bracketId));
    }
  };

  // Handle select all brackets
  const handleSelectAll = (checked: boolean) => {
    if (checked) {
      setSelectedBrackets(brackets.data.map((bracket: any) => bracket.id));
    } else {
      setSelectedBrackets([]);
    }
  };

  // Handle bracket deletion
  const handleDelete = (bracket: any) => {
    setBracketToDelete(bracket);
    setIsDeleteModalOpen(true);
  };

  const confirmDelete = () => {
    if (bracketToDelete) {
      router.delete(route('admin.brackets.destroy', bracketToDelete.id), {
        onSuccess: () => {
          setIsDeleteModalOpen(false);
          setBracketToDelete(null);
        },
      });
    }
  };

  // Handle multiple bracket deletion
  const handleMultipleDelete = () => {
    if (selectedBrackets.length > 0) {
      setIsMultipleDeleteModalOpen(true);
    }
  };

  const confirmMultipleDelete = () => {
    router.delete(route('admin.brackets.destroyMultiple'), {
      data: { ids: selectedBrackets },
      onSuccess: () => {
        setIsMultipleDeleteModalOpen(false);
        setSelectedBrackets([]);
      },
    });
  };

  // Get bracket type badge color
  const getBracketTypeColor = (type: string) => {
    switch (type) {
      case 'winners': return 'bg-green-100 text-green-800';
      case 'losers': return 'bg-red-100 text-red-800';
      case 'final': return 'bg-yellow-100 text-yellow-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  // Get tournament type badge color
  const getTournamentTypeColor = (type: string) => {
    switch (type) {
      case 'knockout': return 'bg-blue-100 text-blue-800';
      case 'double-elimination': return 'bg-purple-100 text-purple-800';
      case 'round-robin': return 'bg-orange-100 text-orange-800';
      case 'group-knockout': return 'bg-indigo-100 text-indigo-800';
      case 'swiss_system': return 'bg-pink-100 text-pink-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  return (
    <AppLayout breadcrumbs={[{ title: 'Brackets', href: '/admin/brackets' }]}> 
      <Head title="Brackets" />
      <div className="flex flex-col gap-4 p-4">
        <div className="flex justify-between items-center">
          <h1 className="text-2xl font-bold flex items-center gap-2">
            <Trophy className="w-6 h-6" />
            Tournament Brackets
          </h1>
          <div className="flex gap-2">
            {selectedBrackets.length > 0 && (
              <Button variant="destructive" onClick={handleMultipleDelete}>
                <Trash className="w-4 h-4 mr-1" />
                Delete Selected ({selectedBrackets.length})
              </Button>
            )}
            <Dialog open={isGenerateModalOpen} onOpenChange={setIsGenerateModalOpen}>
              <DialogTrigger asChild>
                <Button variant="outline">
                  <Settings className="w-4 h-4 mr-1" />
                  Generate Bracket
                </Button>
              </DialogTrigger>
              <DialogContent className="max-w-2xl">
                <DialogHeader>
                  <DialogTitle>Generate Tournament Bracket</DialogTitle>
                </DialogHeader>
                <form onSubmit={handleGenerateSubmit} className="space-y-4">
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <Label htmlFor="tournament_id">Tournament *</Label>
                      <Select 
                        value={generateForm.tournament_id} 
                        onValueChange={handleTournamentSelect}
                      >
                        <SelectTrigger className={generateFormErrors.tournament_id ? 'border-red-500' : ''}>
                          <SelectValue placeholder="Select tournament" />
                        </SelectTrigger>
                        <SelectContent>
                          {tournaments.map((tournament: any) => (
                            <SelectItem key={tournament.id} value={tournament.id.toString()}>
                              {tournament.name} ({tournament.type})
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      {generateFormErrors.tournament_id && (
                        <p className="text-red-500 text-sm mt-1">{generateFormErrors.tournament_id}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="bracket_name">Bracket Name *</Label>
                      <Input
                        id="bracket_name"
                        value={generateForm.bracket_name}
                        onChange={(e) => setGenerateForm(prev => ({ ...prev, bracket_name: e.target.value }))}
                        className={generateFormErrors.bracket_name ? 'border-red-500' : ''}
                        placeholder="e.g., Main Bracket"
                      />
                      {generateFormErrors.bracket_name && (
                        <p className="text-red-500 text-sm mt-1">{generateFormErrors.bracket_name}</p>
                      )}
                    </div>
                  </div>

                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <Label htmlFor="bracket_type">Bracket Type *</Label>
                      <Select 
                        value={generateForm.bracket_type} 
                        onValueChange={(value) => setGenerateForm(prev => ({ ...prev, bracket_type: value }))}
                      >
                        <SelectTrigger>
                          <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="winners">Winners Bracket</SelectItem>
                          <SelectItem value="losers">Losers Bracket</SelectItem>
                          <SelectItem value="final">Final Bracket</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>

                    <div>
                      <Label htmlFor="seeding_method">Seeding Method *</Label>
                      <Select 
                        value={generateForm.seeding_method} 
                        onValueChange={(value) => setGenerateForm(prev => ({ ...prev, seeding_method: value }))}
                      >
                        <SelectTrigger>
                          <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="random">Random</SelectItem>
                          <SelectItem value="ranking">By Ranking</SelectItem>
                          <SelectItem value="rating">By Rating</SelectItem>
                          <SelectItem value="balanced">Balanced</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>
                  </div>

                  {selectedTournament && (
                    <div>
                      <Label>Select Participants ({generateForm.participants.length} selected)</Label>
                      <div className="max-h-60 overflow-y-auto border rounded-md p-2 space-y-2">
                        {availableParticipants.map((participant: any) => (
                          <div key={participant.id} className="flex items-center space-x-2">
                            <input
                              type="checkbox"
                              id={`participant-${participant.id}`}
                              checked={generateForm.participants.some(p => p.id === participant.id)}
                              onChange={() => handleParticipantToggle(participant)}
                              className="rounded"
                            />
                            <label htmlFor={`participant-${participant.id}`} className="flex-1 cursor-pointer">
                              <div className="flex justify-between items-center">
                                <span className="font-medium">{participant.name}</span>
                                <div className="flex gap-2 text-sm text-gray-500">
                                  <span>Rating: {participant.rating}</span>
                                  <span>Rank: {participant.ranking}</span>
                                </div>
                              </div>
                            </label>
                          </div>
                        ))}
                      </div>
                      {generateFormErrors.participants && (
                        <p className="text-red-500 text-sm mt-1">{generateFormErrors.participants}</p>
                      )}
                    </div>
                  )}

                  <div className="flex justify-end gap-2">
                    <Button type="button" variant="outline" onClick={() => setIsGenerateModalOpen(false)}>
                      Cancel
                    </Button>
                    <Button type="submit" disabled={generateForm.participants.length < 2}>
                      <Trophy className="w-4 h-4 mr-1" />
                      Generate Bracket
                    </Button>
                  </div>
                </form>
              </DialogContent>
            </Dialog>
            
            <Dialog open={isCreateModalOpen} onOpenChange={setIsCreateModalOpen}>
              <DialogTrigger asChild>
                <Button>
                  <Plus className="w-4 h-4 mr-1" />
                  Create Bracket
                </Button>
              </DialogTrigger>
              <DialogContent>
                <DialogHeader>
                  <DialogTitle>Create New Bracket</DialogTitle>
                </DialogHeader>
                <form onSubmit={handleCreateSubmit} className="space-y-4">
                  <div>
                    <Label htmlFor="name">Bracket Name *</Label>
                    <Input
                      id="name"
                      value={form.name}
                      onChange={(e) => setForm(prev => ({ ...prev, name: e.target.value }))}
                      className={formErrors.name ? 'border-red-500' : ''}
                      placeholder="Enter bracket name"
                    />
                    {formErrors.name && <p className="text-red-500 text-sm mt-1">{formErrors.name}</p>}
                  </div>

                  <div>
                    <Label htmlFor="type">Bracket Type *</Label>
                    <Select value={form.type} onValueChange={(value) => setForm(prev => ({ ...prev, type: value }))}>
                      <SelectTrigger className={formErrors.type ? 'border-red-500' : ''}>
                        <SelectValue placeholder="Select bracket type" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="winners">Winners Bracket</SelectItem>
                        <SelectItem value="losers">Losers Bracket</SelectItem>
                        <SelectItem value="final">Final Bracket</SelectItem>
                        <SelectItem value="quarter-finals">Quarter Finals</SelectItem>
                        <SelectItem value="semi-finals">Semi Finals</SelectItem>
                        <SelectItem value="third-place">Third Place</SelectItem>
                      </SelectContent>
                    </Select>
                    {formErrors.type && <p className="text-red-500 text-sm mt-1">{formErrors.type}</p>}
                  </div>

                  <div>
                    <Label htmlFor="tournament_id">Tournament *</Label>
                    <Select value={form.tournament_id} onValueChange={(value) => setForm(prev => ({ ...prev, tournament_id: value }))}>
                      <SelectTrigger className={formErrors.tournament_id ? 'border-red-500' : ''}>
                        <SelectValue placeholder="Select tournament" />
                      </SelectTrigger>
                      <SelectContent>
                        {tournaments.map((tournament: any) => (
                          <SelectItem key={tournament.id} value={tournament.id.toString()}>
                            {tournament.name}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    {formErrors.tournament_id && <p className="text-red-500 text-sm mt-1">{formErrors.tournament_id}</p>}
                  </div>

                  <div className="flex justify-end gap-2">
                    <Button type="button" variant="outline" onClick={() => setIsCreateModalOpen(false)}>
                      Cancel
                    </Button>
                    <Button type="submit">Create Bracket</Button>
                  </div>
                </form>
              </DialogContent>
            </Dialog>
          </div>
        </div>

        {/* Filters */}
        <Card className="p-4">
          <form onSubmit={handleFilterSubmit} className="flex gap-4 items-end">
            <div className="flex-1">
              <Label htmlFor="search">Search</Label>
              <Input
                id="search"
                value={localFilters.search}
                onChange={(e) => setLocalFilters(prev => ({ ...prev, search: e.target.value }))}
                placeholder="Search brackets..."
              />
            </div>
            <div>
              <Label htmlFor="type">Type</Label>
              <Select value={localFilters.type} onValueChange={(value) => setLocalFilters(prev => ({ ...prev, type: value }))}>
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Types</SelectItem>
                  <SelectItem value="winners">Winners</SelectItem>
                  <SelectItem value="losers">Losers</SelectItem>
                  <SelectItem value="final">Final</SelectItem>
                  <SelectItem value="quarter-finals">Quarter Finals</SelectItem>
                  <SelectItem value="semi-finals">Semi Finals</SelectItem>
                  <SelectItem value="third-place">Third Place</SelectItem>
                </SelectContent>
              </Select>
            </div>
            <div>
              <Label htmlFor="tournament_id">Tournament</Label>
              <Select value={localFilters.tournament_id} onValueChange={(value) => setLocalFilters(prev => ({ ...prev, tournament_id: value }))}>
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Tournaments</SelectItem>
                  {tournaments.map((tournament: any) => (
                    <SelectItem key={tournament.id} value={tournament.id.toString()}>
                      {tournament.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <Button type="submit">Filter</Button>
          </form>
        </Card>

        {/* Brackets Table */}
        <Card>
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead className="w-12">
                  <input
                    type="checkbox"
                    checked={selectedBrackets.length === brackets.data.length && brackets.data.length > 0}
                    onChange={(e) => handleSelectAll(e.target.checked)}
                    className="rounded"
                  />
                </TableHead>
                <TableHead>Name</TableHead>
                <TableHead>Type</TableHead>
                <TableHead>Tournament</TableHead>
                <TableHead>Rounds</TableHead>
                <TableHead>Matches</TableHead>
                <TableHead>Status</TableHead>
                <TableHead className="text-right">Actions</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {brackets.data.map((bracket: any) => (
                <TableRow key={bracket.id}>
                  <TableCell>
                    <input
                      type="checkbox"
                      checked={selectedBrackets.includes(bracket.id)}
                      onChange={(e) => handleBracketSelect(bracket.id, e.target.checked)}
                      className="rounded"
                    />
                  </TableCell>
                  <TableCell className="font-medium">{bracket.name}</TableCell>
                  <TableCell>
                    <Badge className={getBracketTypeColor(bracket.type)}>
                      {bracket.type.replace('-', ' ')}
                    </Badge>
                  </TableCell>
                  <TableCell>
                    <div className="space-y-1">
                      <div className="font-medium">{bracket.tournament?.name}</div>
                      <Badge className={getTournamentTypeColor(bracket.tournament?.type)}>
                        {bracket.tournament?.type?.replace('-', ' ')}
                      </Badge>
                    </div>
                  </TableCell>
                  <TableCell>{bracket.rounds || 0}</TableCell>
                  <TableCell>{bracket.game_matches?.length || 0}</TableCell>
                  <TableCell>
                    <Badge variant={bracket.is_active ? 'default' : 'secondary'}>
                      {bracket.is_active ? 'Active' : 'Inactive'}
                    </Badge>
                  </TableCell>
                  <TableCell className="text-right">
                    <div className="flex justify-end gap-2">
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => router.get(route('admin.brackets.show', bracket.id))}
                      >
                        <Eye className="w-4 h-4" />
                      </Button>
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => router.get(route('admin.brackets.edit', bracket.id))}
                      >
                        <Pencil className="w-4 h-4" />
                      </Button>
                      <Button
                        variant="destructive"
                        size="sm"
                        onClick={() => handleDelete(bracket)}
                      >
                        <Trash className="w-4 h-4" />
                      </Button>
                    </div>
                  </TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </Card>

        {/* Pagination */}
        {brackets.links && (
          <div className="flex justify-center">
            <div className="flex gap-2">
              {brackets.links.map((link: any, index: number) => (
                <Button
                  key={index}
                  variant={link.active ? 'default' : 'outline'}
                  size="sm"
                  onClick={() => link.url && router.get(link.url)}
                  disabled={!link.url}
                >
                  {link.label}
                </Button>
              ))}
            </div>
          </div>
        )}

        {/* Delete Confirmation Modal */}
        <Dialog open={isDeleteModalOpen} onOpenChange={setIsDeleteModalOpen}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Delete Bracket</DialogTitle>
            </DialogHeader>
            <p>Are you sure you want to delete "{bracketToDelete?.name}"? This action cannot be undone.</p>
            <div className="flex justify-end gap-2">
              <Button variant="outline" onClick={() => setIsDeleteModalOpen(false)}>
                Cancel
              </Button>
              <Button variant="destructive" onClick={confirmDelete}>
                Delete
              </Button>
            </div>
          </DialogContent>
        </Dialog>

        {/* Multiple Delete Confirmation Modal */}
        <Dialog open={isMultipleDeleteModalOpen} onOpenChange={setIsMultipleDeleteModalOpen}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Delete Multiple Brackets</DialogTitle>
            </DialogHeader>
            <p>Are you sure you want to delete {selectedBrackets.length} selected brackets? This action cannot be undone.</p>
            <div className="flex justify-end gap-2">
              <Button variant="outline" onClick={() => setIsMultipleDeleteModalOpen(false)}>
                Cancel
              </Button>
              <Button variant="destructive" onClick={confirmMultipleDelete}>
                Delete All
              </Button>
            </div>
          </DialogContent>
        </Dialog>
      </div>
    </AppLayout>
  );
}