import AppLayout from '@/layouts/app-layout';
import { Head, router } from '@inertiajs/react';
import { Card } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Switch } from '@/components/ui/switch';
import { useState } from 'react';
import { ArrowLeft, Save } from 'lucide-react';

export default function AdminBracketEdit({ bracket, tournaments }) {
  const [form, setForm] = useState({
    name: bracket.name || '',
    type: bracket.type || '',
    tournament_id: bracket.tournament_id?.toString() || '',
    rounds: bracket.rounds || 0,
    is_active: bracket.is_active ?? true,
  });
  const [formErrors, setFormErrors] = useState({});

  const handleSubmit = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    setFormErrors({});
    
    router.put(route('admin.brackets.update', bracket.id), form, {
      onSuccess: () => {
        router.get(route('admin.brackets.show', bracket.id));
      },
      onError: (errors) => {
        setFormErrors(errors);
      },
    });
  };

  return (
    <AppLayout breadcrumbs={[
      { title: 'Brackets', href: '/admin/brackets' },
      { title: bracket.name, href: `/admin/brackets/${bracket.id}` },
      { title: 'Edit', href: `/admin/brackets/${bracket.id}/edit` }
    ]}>
      <Head title={`Edit Bracket: ${bracket.name}`} />
      <div className="flex flex-col gap-4 p-4">
        <div className="flex justify-between items-center">
          <h1 className="text-2xl font-bold">Edit Bracket</h1>
          <div className="flex gap-2">
            <Button variant="outline" onClick={() => router.get(route('admin.brackets.show', bracket.id))}>
              <ArrowLeft className="w-4 h-4 mr-1" />
              Back to Bracket
            </Button>
          </div>
        </div>

        <Card className="p-6">
          <form onSubmit={handleSubmit} className="space-y-6">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div>
                <Label htmlFor="name">Bracket Name *</Label>
                <Input
                  id="name"
                  value={form.name}
                  onChange={(e) => setForm(prev => ({ ...prev, name: e.target.value }))}
                  className={formErrors.name ? 'border-red-500' : ''}
                  placeholder="Enter bracket name"
                />
                {formErrors.name && <p className="text-red-500 text-sm mt-1">{formErrors.name}</p>}
              </div>

              <div>
                <Label htmlFor="type">Bracket Type *</Label>
                <Select 
                  value={form.type} 
                  onValueChange={(value) => setForm(prev => ({ ...prev, type: value }))}
                >
                  <SelectTrigger className={formErrors.type ? 'border-red-500' : ''}>
                    <SelectValue placeholder="Select bracket type" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="winners">Winners Bracket</SelectItem>
                    <SelectItem value="losers">Losers Bracket</SelectItem>
                    <SelectItem value="final">Final Bracket</SelectItem>
                    <SelectItem value="quarter-finals">Quarter Finals</SelectItem>
                    <SelectItem value="semi-finals">Semi Finals</SelectItem>
                    <SelectItem value="third-place">Third Place</SelectItem>
                  </SelectContent>
                </Select>
                {formErrors.type && <p className="text-red-500 text-sm mt-1">{formErrors.type}</p>}
              </div>

              <div>
                <Label htmlFor="tournament_id">Tournament *</Label>
                <Select 
                  value={form.tournament_id} 
                  onValueChange={(value) => setForm(prev => ({ ...prev, tournament_id: value }))}
                >
                  <SelectTrigger className={formErrors.tournament_id ? 'border-red-500' : ''}>
                    <SelectValue placeholder="Select tournament" />
                  </SelectTrigger>
                  <SelectContent>
                    {tournaments.map((tournament) => (
                      <SelectItem key={tournament.id} value={tournament.id.toString()}>
                        {tournament.name} ({tournament.type})
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
                {formErrors.tournament_id && <p className="text-red-500 text-sm mt-1">{formErrors.tournament_id}</p>}
              </div>

              <div>
                <Label htmlFor="rounds">Number of Rounds</Label>
                <Input
                  id="rounds"
                  type="number"
                  min="0"
                  value={form.rounds}
                  onChange={(e) => setForm(prev => ({ ...prev, rounds: parseInt(e.target.value) || 0 }))}
                  className={formErrors.rounds ? 'border-red-500' : ''}
                  placeholder="0"
                />
                {formErrors.rounds && <p className="text-red-500 text-sm mt-1">{formErrors.rounds}</p>}
              </div>
            </div>

            <div className="flex items-center space-x-2">
              <Switch
                id="is_active"
                checked={form.is_active}
                onCheckedChange={(checked) => setForm(prev => ({ ...prev, is_active: checked }))}
              />
              <Label htmlFor="is_active">Active Bracket</Label>
            </div>

            <div className="flex justify-end gap-2">
              <Button type="button" variant="outline" onClick={() => router.get(route('admin.brackets.show', bracket.id))}>
                Cancel
              </Button>
              <Button type="submit">
                <Save className="w-4 h-4 mr-1" />
                Update Bracket
              </Button>
            </div>
          </form>
        </Card>
      </div>
    </AppLayout>
  );
}
