import { useForm, Link, usePage } from '@inertiajs/react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Switch } from '@/components/ui/switch';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { Badge } from '@/components/ui/badge';
import { 
  Globe, 
  Mail, 
  Clock, 
  Shield, 
  Search, 
  Image, 
  Database, 
  Settings as SettingsIcon,
  Terminal,
  Play,
  AlertTriangle,
  CheckCircle,
  Info,
  Send
} from 'lucide-react';
import AppLayout from '@/layouts/app-layout';
import MediaManager from '@/components/MediaManager';
import ConditionalSettings from '@/components/ConditionalSettings';
import { useState } from 'react';
import { toast } from 'react-hot-toast';
import axios from 'axios';

interface SettingsProps {
  settings: {
    site_name: string;
    site_description: string;
    contact_email: string;
    contact_phone: string;
    timezone: string;
    maintenance_mode: boolean;
    // SEO Settings
    meta_title: string;
    meta_description: string;
    meta_keywords: string;
    og_image: string;
    // Branding
    logo: string;
    favicon: string;
    // Environment
    app_debug: boolean;
    app_env: string;
    mail_from_address: string;
    mail_from_name: string;
    database_connection: string;
    // Queue Settings
    queue_connection: string;
    queue_driver: string;
    // Cache Settings
    cache_driver: string;
    session_driver: string;
    // Mail Settings
    mail_driver: string;
    mail_host: string;
    mail_port: string;
    mail_username: string;
    mail_password: string;
    mail_encryption: string;
    mail_timeout: number;
    mail_verify_peer: boolean;
    // Storage Settings
    filesystem_disk: string;
    storage_path: string;
  };
  validations: {
    database: { issues: string[]; recommendations: string[] };
    mail: { issues: string[]; recommendations: string[] };
    cache: { issues: string[]; recommendations: string[] };
    queue: { issues: string[]; recommendations: string[] };
    storage: { issues: string[]; recommendations: string[] };
  };
  hasCriticalIssues: boolean;
  session?: any;
}

interface ArtisanCommand {
  name: string;
  description: string;
  category: string;
  requiresConfirmation?: boolean;
  confirmationMessage?: string;
}

// Email Test Form Component
const EmailTestForm = () => {
  const [testEmail, setTestEmail] = useState('');
  const [testName, setTestName] = useState('');
  const [isTestingEmail, setIsTestingEmail] = useState(false);
  const [isTestingConnection, setIsTestingConnection] = useState(false);
  const [testResults, setTestResults] = useState<any>(null);

  const sendTestEmail = async () => {
    if (!testEmail) {
      toast.error('Please enter a test email address');
      return;
    }

    setIsTestingEmail(true);
    setTestResults(null);

    try {
      const response = await axios.post('/admin/settings/test-email', {
        test_email: testEmail,
        test_name: testName || 'Test User',
        use_current_settings: true
      });

      if (response.data.success) {
        toast.success('Test email sent successfully!');
        setTestResults(response.data);
      } else {
        toast.error(response.data.message || 'Failed to send test email');
        setTestResults(response.data);
      }
    } catch (error: any) {
      const errorMessage = error.response?.data?.message || 'Failed to send test email';
      toast.error(errorMessage);
      setTestResults({
        success: false,
        message: errorMessage,
        error_details: error.response?.data?.error_details
      });
    } finally {
      setIsTestingEmail(false);
    }
  };

  const testSmtpConnection = async () => {
    setIsTestingConnection(true);

    try {
      const response = await axios.get('/admin/settings/email-status');
      const settings = response.data.settings;

      if (!settings.mail_host || !settings.mail_port) {
        toast.error('SMTP settings are not configured');
        return;
      }

      const testResponse = await axios.post('/admin/settings/test-smtp', {
        mail_host: settings.mail_host,
        mail_port: settings.mail_port,
        mail_username: settings.mail_username,
        mail_password: settings.mail_password,
        mail_encryption: settings.mail_encryption,
        mail_timeout: settings.mail_timeout || 60
      });

      if (testResponse.data.success) {
        toast.success('SMTP connection successful!');
      } else {
        toast.error(testResponse.data.message || 'SMTP connection failed');
      }
    } catch (error: any) {
      const errorMessage = error.response?.data?.message || 'SMTP connection test failed';
      toast.error(errorMessage);
    } finally {
      setIsTestingConnection(false);
    }
  };

  return (
    <div className="space-y-4">
      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <div className="space-y-2">
          <Label htmlFor="test_email">Test Email Address</Label>
          <Input
            id="test_email"
            type="email"
            value={testEmail}
            onChange={(e) => setTestEmail(e.target.value)}
            placeholder="Enter email to send test to"
            required
          />
        </div>

        <div className="space-y-2">
          <Label htmlFor="test_name">Recipient Name (Optional)</Label>
          <Input
            id="test_name"
            value={testName}
            onChange={(e) => setTestName(e.target.value)}
            placeholder="Test User"
          />
        </div>
      </div>

      <div className="flex gap-2">
        <Button 
          onClick={sendTestEmail} 
          disabled={isTestingEmail || !testEmail}
          className="bg-green-600 hover:bg-green-700"
        >
          {isTestingEmail ? (
            <>
              <div className="mr-2 h-4 w-4 animate-spin rounded-full border-2 border-white border-t-transparent"></div>
              Sending...
            </>
          ) : (
            <>
              <Send className="mr-2 h-4 w-4" />
              Send Test Email
            </>
          )}
        </Button>

        <Button 
          variant="outline"
          onClick={testSmtpConnection} 
          disabled={isTestingConnection}
        >
          {isTestingConnection ? (
            <>
              <div className="mr-2 h-4 w-4 animate-spin rounded-full border-2 border-gray-600 border-t-transparent"></div>
              Testing...
            </>
          ) : (
            'Test SMTP Connection'
          )}
        </Button>
      </div>

      {testResults && (
        <Alert className={testResults.success ? 'border-green-200 bg-green-50' : 'border-red-200 bg-red-50'}>
          <div className="flex items-start">
            {testResults.success ? (
              <CheckCircle className="h-4 w-4 text-green-600 mt-0.5" />
            ) : (
              <AlertTriangle className="h-4 w-4 text-red-600 mt-0.5" />
            )}
            <div className="ml-2 flex-1">
              <AlertDescription className={testResults.success ? 'text-green-800' : 'text-red-800'}>
                <div className="font-medium mb-1">
                  {testResults.success ? 'Email Test Successful!' : 'Email Test Failed'}
                </div>
                <div className="text-sm">
                  {testResults.message}
                </div>
                {testResults.details && (
                  <div className="mt-2 text-xs">
                    <div>Sent to: {testResults.details.to}</div>
                    <div>Subject: {testResults.details.subject}</div>
                    <div>Mailer: {testResults.details.mailer}</div>
                    <div>Time: {testResults.details.sent_at}</div>
                  </div>
                )}
                {testResults.error_details && (
                  <div className="mt-2 text-xs">
                    <div>Error: {testResults.error_details.exception}</div>
                    <div>File: {testResults.error_details.file}:{testResults.error_details.line}</div>
                  </div>
                )}
              </AlertDescription>
            </div>
          </div>
        </Alert>
      )}
    </div>
  );
};

const Settings = ({ settings, validations, hasCriticalIssues, session }: SettingsProps) => {
  const [selectedCommand, setSelectedCommand] = useState<string>('');
  const [commandOutput, setCommandOutput] = useState<string>('');
  const [isRunningCommand, setIsRunningCommand] = useState(false);

    const { data, setData, put, patch, processing, errors } = useForm({
    // General Settings
    site_name: settings.site_name,
    site_description: settings.site_description,
    contact_email: settings.contact_email,
    contact_phone: settings.contact_phone,
    timezone: settings.timezone,
    maintenance_mode: settings.maintenance_mode,
    // SEO Settings
    meta_title: settings.meta_title || '',
    meta_description: settings.meta_description || '',
    meta_keywords: settings.meta_keywords || '',
    og_image: settings.og_image || '',
    // Branding
    logo: settings.logo || '',
    favicon: settings.favicon || '',
    // Environment
    app_debug: settings.app_debug || false,
    app_env: settings.app_env || 'production',
    mail_from_address: settings.mail_from_address || '',
    mail_from_name: settings.mail_from_name || '',
    database_connection: settings.database_connection || 'mysql',
    // Queue Settings
    queue_connection: settings.queue_connection || 'sync',
    queue_driver: settings.queue_driver || 'sync',
    // Cache Settings
    cache_driver: settings.cache_driver || 'file',
    session_driver: settings.session_driver || 'file',
    // Mail Settings
    mail_driver: settings.mail_driver || 'smtp',
    mail_host: settings.mail_host || '',
    mail_port: settings.mail_port || '',
    mail_username: settings.mail_username || '',
    mail_password: settings.mail_password || '',
    mail_encryption: settings.mail_encryption === '' ? 'none' : settings.mail_encryption || 'tls',
    mail_timeout: settings.mail_timeout || 60,
    mail_verify_peer: settings.mail_verify_peer || true,
    // Storage Settings
    filesystem_disk: settings.filesystem_disk || 'local',
    storage_path: settings.storage_path || '',
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    setData('mail_encryption', data.mail_encryption === 'none' ? '' : data.mail_encryption);
    put('/admin/settings');
  };

  const runArtisanCommand = async (command: string) => {
    setIsRunningCommand(true);
    setCommandOutput('Running command...');
    
    try {
      const response = await fetch('/admin/settings/run-command', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
        },
        body: JSON.stringify({ command }),
      });
      
      const result = await response.json();
      
      if (result.success) {
        setCommandOutput(result.output);
        toast.success('Command executed successfully!');
      } else {
        setCommandOutput(result.error || 'Command failed');
        toast.error('Command failed to execute');
      }
    } catch (error) {
      setCommandOutput('Error executing command');
      toast.error('Error executing command');
    } finally {
      setIsRunningCommand(false);
    }
  };

  const artisanCommands: ArtisanCommand[] = [
    // Database Commands
    {
      name: 'migrate',
      description: 'Run database migrations',
      category: 'Database',
      requiresConfirmation: true,
      confirmationMessage: 'This will run all pending migrations. Continue?'
    },
    {
      name: 'migrate:fresh',
      description: 'Drop all tables and re-run migrations',
      category: 'Database',
      requiresConfirmation: true,
      confirmationMessage: 'This will drop all tables and re-run migrations. This action cannot be undone!'
    },
    {
      name: 'migrate:rollback',
      description: 'Rollback the last batch of migrations',
      category: 'Database',
      requiresConfirmation: true,
      confirmationMessage: 'This will rollback the last batch of migrations. Continue?'
    },
    {
      name: 'db:seed',
      description: 'Seed the database with sample data',
      category: 'Database'
    },
    // Cache Commands
    {
      name: 'cache:clear',
      description: 'Clear application cache',
      category: 'Cache'
    },
    {
      name: 'config:clear',
      description: 'Clear configuration cache',
      category: 'Cache'
    },
    {
      name: 'config:cache',
      description: 'Cache configuration files',
      category: 'Cache'
    },
    {
      name: 'route:clear',
      description: 'Clear route cache',
      category: 'Cache'
    },
    {
      name: 'view:clear',
      description: 'Clear compiled views',
      category: 'Cache'
    },
    // Queue Commands
    {
      name: 'queue:work',
      description: 'Start queue worker',
      category: 'Queue'
    },
    {
      name: 'queue:restart',
      description: 'Restart queue workers',
      category: 'Queue'
    },
    {
      name: 'queue:failed',
      description: 'List failed queue jobs',
      category: 'Queue'
    },
    // Maintenance Commands
    {
      name: 'down',
      description: 'Put application in maintenance mode',
      category: 'Maintenance'
    },
    {
      name: 'up',
      description: 'Bring application out of maintenance mode',
      category: 'Maintenance'
    },
    // Storage Commands
    {
      name: 'storage:link',
      description: 'Create symbolic link for storage (legacy)',
      category: 'Storage'
    },
    {
      name: 'storage:setup',
      description: 'Setup public storage directory',
      category: 'Storage'
    },
    {
      name: 'storage:sync',
      description: 'Sync files from storage/app/public to public/storage',
      category: 'Storage'
    },
    {
      name: 'storage:cleanup',
      description: 'Clean up old files in public storage',
      category: 'Storage'
    },
    {
      name: 'storage:clear',
      description: 'Clear storage cache',
      category: 'Storage'
    },
  ];

  const timezoneOptions = [
    { value: 'UTC', label: 'UTC (Coordinated Universal Time)' },
    { value: 'America/New_York', label: 'Eastern Time (ET)' },
    { value: 'America/Chicago', label: 'Central Time (CT)' },
    { value: 'America/Denver', label: 'Mountain Time (MT)' },
    { value: 'America/Los_Angeles', label: 'Pacific Time (PT)' },
    { value: 'Europe/London', label: 'London (GMT/BST)' },
    { value: 'Europe/Paris', label: 'Paris (CET/CEST)' },
    { value: 'Asia/Tokyo', label: 'Tokyo (JST)' },
    { value: 'Asia/Shanghai', label: 'Shanghai (CST)' },
    { value: 'Australia/Sydney', label: 'Sydney (AEST/AEDT)' },
  ];

  const databaseOptions = [
    { value: 'mysql', label: 'MySQL' },
    { value: 'pgsql', label: 'PostgreSQL' },
    { value: 'sqlite', label: 'SQLite' },
  ];

  const queueOptions = [
    { value: 'sync', label: 'Synchronous (Sync)' },
    { value: 'database', label: 'Database' },
    { value: 'redis', label: 'Redis' },
    { value: 'beanstalkd', label: 'Beanstalkd' },
    { value: 'sqs', label: 'Amazon SQS' },
  ];

  const cacheOptions = [
    { value: 'file', label: 'File' },
    { value: 'database', label: 'Database' },
    { value: 'redis', label: 'Redis' },
    { value: 'memcached', label: 'Memcached' },
  ];

  const mailOptions = [
    { value: 'smtp', label: 'SMTP' },
    { value: 'mailgun', label: 'Mailgun' },
    { value: 'ses', label: 'Amazon SES' },
    { value: 'postmark', label: 'Postmark' },
    { value: 'resend', label: 'Resend' },
    { value: 'sendmail', label: 'Sendmail' },
    { value: 'log', label: 'Log Driver' },
  ];

  const mailEncryptionOptions = [
    { value: 'tls', label: 'TLS' },
    { value: 'ssl', label: 'SSL' },
    { value: 'starttls', label: 'STARTTLS' },
    { value: 'none', label: 'None' },
  ];

  const filesystemOptions = [
    { value: 'local', label: 'Local' },
    { value: 's3', label: 'Amazon S3' },
    { value: 'azure', label: 'Azure Blob Storage' },
    { value: 'digitalocean', label: 'DigitalOcean Spaces' },
  ];

  const groupedCommands = artisanCommands.reduce((acc, command) => {
    if (!acc[command.category]) {
      acc[command.category] = [];
    }
    acc[command.category].push(command);
    return acc;
  }, {} as Record<string, ArtisanCommand[]>);

  const renderValidationIssues = (validation: { issues: string[]; recommendations: string[] }, title: string) => {
    if (validation.issues.length === 0 && validation.recommendations.length === 0) {
      return null;
    }

    return (
      <ConditionalSettings
        condition={true}
        title={title}
        description={`Configuration issues and recommendations for ${title.toLowerCase()}`}
        type={validation.issues.length > 0 ? 'warning' : 'info'}
        actionRequired={validation.issues.length > 0}
        actionText={validation.issues.length > 0 ? 'Please address the issues below to ensure proper functionality.' : ''}
      >
        {validation.issues.length > 0 && (
          <div className="space-y-2">
            <h4 className="font-medium text-destructive">Issues:</h4>
            <ul className="list-disc list-inside space-y-1 text-sm">
              {validation.issues.map((issue, index) => (
                <li key={index} className="text-destructive">{issue}</li>
              ))}
            </ul>
          </div>
        )}
        
        {validation.recommendations.length > 0 && (
          <div className="space-y-2">
            <h4 className="font-medium text-blue-600">Recommendations:</h4>
            <ul className="list-disc list-inside space-y-1 text-sm">
              {validation.recommendations.map((recommendation, index) => (
                <li key={index} className="text-blue-600">{recommendation}</li>
              ))}
            </ul>
          </div>
        )}
      </ConditionalSettings>
    );
  };

  return (
    <AppLayout session={session}>
      <div className="space-y-6 p-4">
        {/* Header */}
        {/* Validation Issues Summary */}
        {hasCriticalIssues && (
          <Alert variant="destructive">
            <AlertTriangle className="h-4 w-4" />
            <AlertDescription>
              There are critical configuration issues that need to be addressed. Please review the validation results below.
            </AlertDescription>
          </Alert>
        )}

        <form onSubmit={handleSubmit} className="space-y-6">
          <Tabs defaultValue="general" className="w-full">
            <TabsList className="grid w-full grid-cols-8">
              <TabsTrigger value="general" className="flex items-center gap-2">
                <Globe className="h-4 w-4" />
                General
              </TabsTrigger>
              <TabsTrigger value="seo" className="flex items-center gap-2">
                <Search className="h-4 w-4" />
                SEO
              </TabsTrigger>
              <TabsTrigger value="branding" className="flex items-center gap-2">
                <Image className="h-4 w-4" />
                Branding
              </TabsTrigger>
              <TabsTrigger value="environment" className="flex items-center gap-2">
                <Database className="h-4 w-4" />
                Environment
              </TabsTrigger>
              <TabsTrigger value="mail" className="flex items-center gap-2">
                <Mail className="h-4 w-4" />
                Mail
              </TabsTrigger>
              <TabsTrigger value="emails" className="flex items-center gap-2">
                <Mail className="h-4 w-4" />
                Emails
              </TabsTrigger>
              <TabsTrigger value="storage" className="flex items-center gap-2">
                <Database className="h-4 w-4" />
                Storage
              </TabsTrigger>
              <TabsTrigger value="artisan" className="flex items-center gap-2">
                <Terminal className="h-4 w-4" />
                Artisan
              </TabsTrigger>
            </TabsList>

            {/* General Settings */}
            <TabsContent value="general" className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Globe className="h-5 w-5" />
                    General Settings
                  </CardTitle>
                  <CardDescription>
                    Basic application configuration
                  </CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="space-y-2">
                    <Label htmlFor="site_name">Site Name</Label>
                    <Input
                      id="site_name"
                      value={data.site_name}
                      onChange={(e) => setData('site_name', e.target.value)}
                      placeholder="Enter site name"
                    />
                    {errors.site_name && (
                      <p className="text-sm text-red-600">{errors.site_name}</p>
                    )}
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="site_description">Site Description</Label>
                    <Textarea
                      id="site_description"
                      value={data.site_description}
                      onChange={(e) => setData('site_description', e.target.value)}
                      placeholder="Enter site description"
                      rows={3}
                    />
                    {errors.site_description && (
                      <p className="text-sm text-red-600">{errors.site_description}</p>
                    )}
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="contact_email">Contact Email</Label>
                    <Input
                      id="contact_email"
                      type="email"
                      value={data.contact_email}
                      onChange={(e) => setData('contact_email', e.target.value)}
                      placeholder="Enter contact email"
                    />
                    {errors.contact_email && (
                      <p className="text-sm text-red-600">{errors.contact_email}</p>
                    )}
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="contact_email">Contact Phone</Label>
                    <Input
                      id="contact_phone"
                      type="tel"
                      value={data.contact_phone}
                      onChange={(e) => setData('contact_phone', e.target.value)}
                      placeholder="Enter contact phone"
                    />
                    {errors.contact_phone && (
                      <p className="text-sm text-red-600">{errors.contact_phone}</p>
                    )}
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="timezone">Timezone</Label>
                    <Select value={data.timezone} onValueChange={(value) => setData('timezone', value)}>
                      <SelectTrigger>
                        <SelectValue placeholder="Select timezone" />
                      </SelectTrigger>
                      <SelectContent>
                        {timezoneOptions.map((option) => (
                          <SelectItem key={option.value} value={option.value}>
                            {option.label}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    {errors.timezone && (
                      <p className="text-sm text-red-600">{errors.timezone}</p>
                    )}
                  </div>

                  <div className="flex items-center justify-between">
                    <div className="space-y-0.5">
                      <Label>Maintenance Mode</Label>
                      <p className="text-sm text-muted-foreground">
                        Enable maintenance mode to restrict access
                      </p>
                    </div>
                    <Switch
                      checked={data.maintenance_mode}
                      onCheckedChange={(checked) => setData('maintenance_mode', checked)}
                    />
                  </div>
                </CardContent>
              </Card>

              {/* Validation Issues */}
              {renderValidationIssues(validations.storage, 'Storage Configuration')}
            </TabsContent>

            {/* SEO Settings */}
            <TabsContent value="seo" className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Search className="h-5 w-5" />
                    SEO Settings
                  </CardTitle>
                  <CardDescription>
                    Search engine optimization settings
                  </CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="space-y-2">
                    <Label htmlFor="meta_title">Meta Title</Label>
                    <Input
                      id="meta_title"
                      value={data.meta_title}
                      onChange={(e) => setData('meta_title', e.target.value)}
                      placeholder="Enter meta title"
                    />
                    {errors.meta_title && (
                      <p className="text-sm text-red-600">{errors.meta_title}</p>
                    )}
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="meta_description">Meta Description</Label>
                    <Textarea
                      id="meta_description"
                      value={data.meta_description}
                      onChange={(e) => setData('meta_description', e.target.value)}
                      placeholder="Enter meta description"
                      rows={3}
                    />
                    {errors.meta_description && (
                      <p className="text-sm text-red-600">{errors.meta_description}</p>
                    )}
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="meta_keywords">Meta Keywords</Label>
                    <Input
                      id="meta_keywords"
                      value={data.meta_keywords}
                      onChange={(e) => setData('meta_keywords', e.target.value)}
                      placeholder="Enter meta keywords (comma separated)"
                    />
                    {errors.meta_keywords && (
                      <p className="text-sm text-red-600">{errors.meta_keywords}</p>
                    )}
                  </div>

                  <div className="space-y-2">
                    <Label>Open Graph Image</Label>
                    <div className="flex items-center gap-4">
                      {data.og_image && (
                        <img 
                          src={data.og_image} 
                          alt="OG Image" 
                          className="w-16 h-16 object-cover rounded border"
                        />
                      )}
                      <MediaManager
                        trigger={
                          <Button type="button" variant="outline">
                            {data.og_image ? 'Change Image' : 'Select Image'}
                          </Button>
                        }
                        onSelect={(files) => {
                          const file = Array.isArray(files) ? files[0] : files;
                          if (file) {
                            setData('og_image', file.url);
                          }
                        }}
                      />
                    </div>
                    {errors.og_image && (
                      <p className="text-sm text-red-600">{errors.og_image}</p>
                    )}
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            {/* Branding Settings */}
            <TabsContent value="branding" className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Image className="h-5 w-5" />
                    Branding Settings
                  </CardTitle>
                  <CardDescription>
                    Logo and favicon configuration
                  </CardDescription>
                </CardHeader>
                <CardContent className="space-y-6">
                  <div className="space-y-4">
                    <Label>Site Logo</Label>
                    <div className="flex items-center gap-4">
                      {data.logo && (
                        <img 
                          src={data.logo} 
                          alt="Site Logo" 
                          className="w-20 h-20 object-contain rounded border"
                        />
                      )}
                      <div className="space-y-2">
                        <MediaManager
                          trigger={
                            <Button type="button" variant="outline">
                              {data.logo ? 'Change Logo' : 'Select Logo'}
                            </Button>
                          }
                          onSelect={(files) => {
                            const file = Array.isArray(files) ? files[0] : files;
                            if (file) {
                              setData('logo', file.url);
                            }
                          }}
                        />
                        {data.logo && (
                          <Button
                            type="button"
                            variant="destructive"
                            size="sm"
                            onClick={() => setData('logo', '')}
                          >
                            Remove Logo
                          </Button>
                        )}
                      </div>
                    </div>
                    {errors.logo && (
                      <p className="text-sm text-red-600">{errors.logo}</p>
                    )}
                  </div>

                  <div className="space-y-4">
                    <Label>Favicon</Label>
                    <div className="flex items-center gap-4">
                      {data.favicon && (
                        <img 
                          src={data.favicon} 
                          alt="Favicon" 
                          className="w-8 h-8 object-contain rounded border"
                        />
                      )}
                      <div className="space-y-2">
                        <MediaManager
                          trigger={
                            <Button type="button" variant="outline">
                              {data.favicon ? 'Change Favicon' : 'Select Favicon'}
                            </Button>
                          }
                          onSelect={(files) => {
                            const file = Array.isArray(files) ? files[0] : files;
                            if (file) {
                              setData('favicon', file.url);
                            }
                          }}
                        />
                        {data.favicon && (
                          <Button
                            type="button"
                            variant="destructive"
                            size="sm"
                            onClick={() => setData('favicon', '')}
                          >
                            Remove Favicon
                          </Button>
                        )}
                      </div>
                    </div>
                    {errors.favicon && (
                      <p className="text-sm text-red-600">{errors.favicon}</p>
                    )}
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            {/* Environment Settings */}
            <TabsContent value="environment" className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Database className="h-5 w-5" />
                    Environment Settings
                  </CardTitle>
                  <CardDescription>
                    Application environment configuration
                  </CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="space-y-2">
                    <Label htmlFor="app_env">Application Environment</Label>
                    <Select value={data.app_env} onValueChange={(value) => setData('app_env', value)}>
                      <SelectTrigger>
                        <SelectValue placeholder="Select environment" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="production">Production</SelectItem>
                        <SelectItem value="staging">Staging</SelectItem>
                        <SelectItem value="local">Local</SelectItem>
                      </SelectContent>
                    </Select>
                    {errors.app_env && (
                      <p className="text-sm text-red-600">{errors.app_env}</p>
                    )}
                  </div>

                  <div className="flex items-center justify-between">
                    <div className="space-y-0.5">
                      <Label>Debug Mode</Label>
                      <p className="text-sm text-muted-foreground">
                        Enable debug mode for development
                      </p>
                    </div>
                    <Switch
                      checked={data.app_debug}
                      onCheckedChange={(checked) => setData('app_debug', checked)}
                    />
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="database_connection">Database Connection</Label>
                    <Select value={data.database_connection} onValueChange={(value) => setData('database_connection', value)}>
                      <SelectTrigger>
                        <SelectValue placeholder="Select database" />
                      </SelectTrigger>
                      <SelectContent>
                        {databaseOptions.map((option) => (
                          <SelectItem key={option.value} value={option.value}>
                            {option.label}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    {errors.database_connection && (
                      <p className="text-sm text-red-600">{errors.database_connection}</p>
                    )}
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="queue_connection">Queue Connection</Label>
                    <Select value={data.queue_connection} onValueChange={(value) => setData('queue_connection', value)}>
                      <SelectTrigger>
                        <SelectValue placeholder="Select queue driver" />
                      </SelectTrigger>
                      <SelectContent>
                        {queueOptions.map((option) => (
                          <SelectItem key={option.value} value={option.value}>
                            {option.label}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    {errors.queue_connection && (
                      <p className="text-sm text-red-600">{errors.queue_connection}</p>
                    )}
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="cache_driver">Cache Driver</Label>
                    <Select value={data.cache_driver} onValueChange={(value) => setData('cache_driver', value)}>
                      <SelectTrigger>
                        <SelectValue placeholder="Select cache driver" />
                      </SelectTrigger>
                      <SelectContent>
                        {cacheOptions.map((option) => (
                          <SelectItem key={option.value} value={option.value}>
                            {option.label}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    {errors.cache_driver && (
                      <p className="text-sm text-red-600">{errors.cache_driver}</p>
                    )}
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="session_driver">Session Driver</Label>
                    <Select value={data.session_driver} onValueChange={(value) => setData('session_driver', value)}>
                      <SelectTrigger>
                        <SelectValue placeholder="Select session driver" />
                      </SelectTrigger>
                      <SelectContent>
                        {cacheOptions.map((option) => (
                          <SelectItem key={option.value} value={option.value}>
                            {option.label}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    {errors.session_driver && (
                      <p className="text-sm text-red-600">{errors.session_driver}</p>
                    )}
                  </div>
                </CardContent>
              </Card>

              {/* Validation Issues */}
              {renderValidationIssues(validations.database, 'Database Configuration')}
              {renderValidationIssues(validations.cache, 'Cache Configuration')}
              {renderValidationIssues(validations.queue, 'Queue Configuration')}
            </TabsContent>

            {/* Mail Settings */}
            <TabsContent value="mail" className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Mail className="h-5 w-5" />
                    Mail Configuration
                  </CardTitle>
                  <CardDescription>
                    Email server configuration
                  </CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="space-y-2">
                    <Label htmlFor="mail_driver">Mail Driver</Label>
                    <Select value={data.mail_driver} onValueChange={(value) => setData('mail_driver', value)}>
                      <SelectTrigger>
                        <SelectValue placeholder="Select mail driver" />
                      </SelectTrigger>
                      <SelectContent>
                        {mailOptions.map((option) => (
                          <SelectItem key={option.value} value={option.value}>
                            {option.label}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    {errors.mail_driver && (
                      <p className="text-sm text-red-600">{errors.mail_driver}</p>
                    )}
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="mail_from_address">Mail From Address</Label>
                    <Input
                      id="mail_from_address"
                      type="email"
                      value={data.mail_from_address}
                      onChange={(e) => setData('mail_from_address', e.target.value)}
                      placeholder="Enter mail from address"
                    />
                    {errors.mail_from_address && (
                      <p className="text-sm text-red-600">{errors.mail_from_address}</p>
                    )}
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="mail_from_name">Mail From Name</Label>
                    <Input
                      id="mail_from_name"
                      value={data.mail_from_name}
                      onChange={(e) => setData('mail_from_name', e.target.value)}
                      placeholder="Enter mail from name"
                    />
                    {errors.mail_from_name && (
                      <p className="text-sm text-red-600">{errors.mail_from_name}</p>
                    )}
                  </div>

                  {data.mail_driver === 'smtp' && (
                    <>
                      <div className="space-y-2">
                        <Label htmlFor="mail_host">SMTP Host</Label>
                        <Input
                          id="mail_host"
                          value={data.mail_host}
                          onChange={(e) => setData('mail_host', e.target.value)}
                          placeholder="Enter SMTP host"
                        />
                        {errors.mail_host && (
                          <p className="text-sm text-red-600">{errors.mail_host}</p>
                        )}
                      </div>

                      <div className="space-y-2">
                        <Label htmlFor="mail_port">SMTP Port</Label>
                        <Input
                          id="mail_port"
                          value={data.mail_port}
                          onChange={(e) => setData('mail_port', e.target.value)}
                          placeholder="Enter SMTP port"
                        />
                        {errors.mail_port && (
                          <p className="text-sm text-red-600">{errors.mail_port}</p>
                        )}
                      </div>

                      <div className="space-y-2">
                        <Label htmlFor="mail_username">SMTP Username</Label>
                        <Input
                          id="mail_username"
                          value={data.mail_username}
                          onChange={(e) => setData('mail_username', e.target.value)}
                          placeholder="Enter SMTP username"
                        />
                        {errors.mail_username && (
                          <p className="text-sm text-red-600">{errors.mail_username}</p>
                        )}
                      </div>

                      <div className="space-y-2">
                        <Label htmlFor="mail_password">SMTP Password</Label>
                        <Input
                          id="mail_password"
                          type="password"
                          value={data.mail_password}
                          onChange={(e) => setData('mail_password', e.target.value)}
                          placeholder="Enter SMTP password"
                        />
                        {errors.mail_password && (
                          <p className="text-sm text-red-600">{errors.mail_password}</p>
                        )}
                      </div>

                      <div className="space-y-2">
                        <Label htmlFor="mail_encryption">Encryption</Label>
                        <Select value={data.mail_encryption} onValueChange={(value) => setData('mail_encryption', value)}>
                          <SelectTrigger>
                            <SelectValue placeholder="Select encryption" />
                          </SelectTrigger>
                          <SelectContent>
                            {mailEncryptionOptions.map((option) => (
                              <SelectItem key={option.value} value={option.value}>
                                {option.label}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                        {errors.mail_encryption && (
                          <p className="text-sm text-red-600">{errors.mail_encryption}</p>
                        )}
                      </div>

                      <div className="grid grid-cols-2 gap-4">
                        <div className="space-y-2">
                          <Label htmlFor="mail_timeout">Timeout (seconds)</Label>
                          <Input
                            id="mail_timeout"
                            type="number"
                            min="10"
                            max="300"
                            value={data.mail_timeout}
                            onChange={(e) => setData('mail_timeout', parseInt(e.target.value) || 60)}
                            placeholder="60"
                          />
                          {errors.mail_timeout && (
                            <p className="text-sm text-red-600">{errors.mail_timeout}</p>
                          )}
                        </div>

                        <div className="space-y-2">
                          <Label htmlFor="mail_verify_peer">Verify SSL Peer</Label>
                          <div className="flex items-center space-x-2">
                            <Switch
                              id="mail_verify_peer"
                              checked={data.mail_verify_peer}
                              onCheckedChange={(checked) => setData('mail_verify_peer' as keyof typeof data, checked)}
                            />
                            <Label htmlFor="mail_verify_peer" className="text-sm">
                              Verify SSL certificates
                            </Label>
                          </div>
                        </div>
                      </div>
                    </>
                  )}
                </CardContent>
              </Card>

              {/* Email Testing Card */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Send className="h-5 w-5" />
                    Email Testing
                  </CardTitle>
                  <CardDescription>
                    Test your email configuration by sending a test email
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <EmailTestForm />
                </CardContent>
              </Card>

              {/* Validation Issues */}
              {renderValidationIssues(validations.mail, 'Mail Configuration')}
            </TabsContent>

            {/* Email Management */}
            <TabsContent value="emails" className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Mail className="h-5 w-5" />
                    Email Account Management
                  </CardTitle>
                  <CardDescription>
                    Manage email credentials and user access for system emails
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="text-center py-8">
                    <Mail className="h-12 w-12 mx-auto mb-4 text-muted-foreground" />
                    <h3 className="text-lg font-medium mb-2">Email Account Management</h3>
                    <p className="text-muted-foreground mb-4">
                      Create and manage email accounts for system use. Grant access to specific users and provide setup instructions.
                    </p>
                    <Button asChild>
                      <Link href={route('admin.settings.emails.index')}>
                        <Mail className="h-4 w-4 mr-2" />
                        Manage Email Accounts
                      </Link>
                    </Button>
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            {/* Storage Settings */}
            <TabsContent value="storage" className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Database className="h-5 w-5" />
                    Storage Settings
                  </CardTitle>
                  <CardDescription>
                    Configure application file storage
                  </CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="space-y-2">
                    <Label htmlFor="filesystem_disk">Filesystem Disk</Label>
                    <Select value={data.filesystem_disk} onValueChange={(value) => setData('filesystem_disk', value)}>
                      <SelectTrigger>
                        <SelectValue placeholder="Select filesystem disk" />
                      </SelectTrigger>
                      <SelectContent>
                        {filesystemOptions.map((option) => (
                          <SelectItem key={option.value} value={option.value}>
                            {option.label}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    {errors.filesystem_disk && (
                      <p className="text-sm text-red-600">{errors.filesystem_disk}</p>
                    )}
                  </div>

                  {data.filesystem_disk === 's3' && (
                    <>
                      <div className="space-y-2">
                        <Label htmlFor="storage_path">S3 Storage Path</Label>
                        <Input
                          id="storage_path"
                          value={data.storage_path}
                          onChange={(e) => setData('storage_path', e.target.value)}
                          placeholder="Enter S3 storage path (e.g., 's3://bucket-name/path')"
                        />
                        {errors.storage_path && (
                          <p className="text-sm text-red-600">{errors.storage_path}</p>
                        )}
                      </div>
                    </>
                  )}

                  <div className="space-y-4">
                    <h4 className="font-medium">Storage Information</h4>
                    <div className="grid gap-4 md:grid-cols-2">
                      <div className="space-y-2">
                        <Label>Current Storage Path</Label>
                        <p className="text-sm text-muted-foreground">{data.storage_path || 'public/storage'}</p>
                      </div>
                      <div className="space-y-2">
                        <Label>Filesystem Driver</Label>
                        <p className="text-sm text-muted-foreground">{data.filesystem_disk}</p>
                      </div>
                    </div>
                  </div>

                  <div className="space-y-4">
                    <h4 className="font-medium">Storage Actions</h4>
                    <div className="grid gap-2 md:grid-cols-2">
                      <Button
                        type="button"
                        variant="outline"
                        size="sm"
                        onClick={() => runArtisanCommand('storage:setup')}
                        disabled={isRunningCommand}
                      >
                        Setup Storage Directories
                      </Button>
                      <Button
                        type="button"
                        variant="outline"
                        size="sm"
                        onClick={() => runArtisanCommand('storage:sync')}
                        disabled={isRunningCommand}
                      >
                        Sync Storage Files
                      </Button>
                      <Button
                        type="button"
                        variant="outline"
                        size="sm"
                        onClick={() => runArtisanCommand('storage:cleanup')}
                        disabled={isRunningCommand}
                      >
                        Cleanup Storage
                      </Button>
                      <Button
                        type="button"
                        variant="outline"
                        size="sm"
                        onClick={() => runArtisanCommand('storage:clear')}
                        disabled={isRunningCommand}
                      >
                        Clear Storage Cache
                      </Button>
                    </div>
                  </div>
                </CardContent>
              </Card>

              {/* Validation Issues */}
              {renderValidationIssues(validations.storage, 'Storage Configuration')}
            </TabsContent>

            {/* Artisan Commands */}
            <TabsContent value="artisan" className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Terminal className="h-5 w-5" />
                    Artisan Commands
                  </CardTitle>
                  <CardDescription>
                    Execute Laravel Artisan commands
                  </CardDescription>
                </CardHeader>
                <CardContent className="space-y-6">
                  {Object.entries(groupedCommands).map(([category, commands]) => (
                    <div key={category} className="space-y-4">
                      <h3 className="text-lg font-semibold">{category}</h3>
                      <div className="grid gap-3 md:grid-cols-2">
                        {commands.map((command) => (
                          <div key={command.name} className="flex items-center justify-between p-3 border rounded-lg">
                            <div className="space-y-1">
                              <p className="font-medium">{command.name}</p>
                              <p className="text-sm text-muted-foreground">{command.description}</p>
                            </div>
                            <Button
                              size="sm"
                              onClick={() => {
                                if (command.requiresConfirmation) {
                                  if (confirm(command.confirmationMessage)) {
                                    runArtisanCommand(command.name);
                                  }
                                } else {
                                  runArtisanCommand(command.name);
                                }
                              }}
                              disabled={isRunningCommand}
                            >
                              <Play className="h-4 w-4" />
                            </Button>
                          </div>
                        ))}
                      </div>
                    </div>
                  ))}

                  {commandOutput && (
                    <Alert>
                      <Terminal className="h-4 w-4" />
                      <AlertDescription className="font-mono text-sm">
                        <pre className="whitespace-pre-wrap">{commandOutput}</pre>
                      </AlertDescription>
                    </Alert>
                  )}
                </CardContent>
              </Card>
            </TabsContent>
          </Tabs>

          {/* Save Button */}
          <div className="flex justify-end">
            <Button type="submit" disabled={processing}>
              {processing ? 'Saving...' : 'Save Settings'}
            </Button>
          </div>
        </form>
      </div>
    </AppLayout>
  );
};

export default Settings;