import React from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';

interface JobRow { id: number; queue: string; attempts: number; reserved_at: number | null; available_at: number; created_at: number; }
interface FailedJobRow { id: number; uuid: string; connection: string; queue: string; exception: string; failed_at: string; }
interface BatchRow { id: string; name: string; total_jobs: number; pending_jobs: number; failed_jobs: number; created_at: number; finished_at: number | null; }
interface NotificationRow { id: string; type: string; notifiable_type: string; notifiable_id: number; data: any; read_at: string | null; created_at: string; }

const TimeCell: React.FC<{ value: number | string | null }> = ({ value }) => {
  if (value === null || value === undefined) return <span>-</span>;
  const v = typeof value === 'number' ? new Date(value * 1000) : new Date(value);
  return <span>{isNaN(v.getTime()) ? '-' : v.toLocaleString()}</span>;
};

const QueueMonitor: React.FC<{ pendingJobs: JobRow[]; failedJobs: FailedJobRow[]; jobBatches: BatchRow[]; notifications: NotificationRow[]; }> = ({ pendingJobs, failedJobs, jobBatches, notifications }) => {
  return (
    <div className="p-4 space-y-6">
     

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <Card>
          <CardHeader>
            <CardTitle>Pending Jobs</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="overflow-x-auto">
              <table className="min-w-full text-sm">
                <thead>
                  <tr className="text-left border-b">
                    <th className="py-2 pr-4">ID</th>
                    <th className="py-2 pr-4">Queue</th>
                    <th className="py-2 pr-4">Attempts</th>
                    <th className="py-2 pr-4">Available</th>
                    <th className="py-2 pr-4">Created</th>
                  </tr>
                </thead>
                <tbody>
                  {pendingJobs.map(j => (
                    <tr key={j.id} className="border-b last:border-0">
                      <td className="py-2 pr-4">{j.id}</td>
                      <td className="py-2 pr-4"><Badge variant="secondary">{j.queue}</Badge></td>
                      <td className="py-2 pr-4">{j.attempts}</td>
                      <td className="py-2 pr-4"><TimeCell value={j.available_at} /></td>
                      <td className="py-2 pr-4"><TimeCell value={j.created_at} /></td>
                    </tr>
                  ))}
                  {pendingJobs.length === 0 && (
                    <tr><td className="py-4 text-gray-500" colSpan={5}>No pending jobs</td></tr>
                  )}
                </tbody>
              </table>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Job Batches</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="overflow-x-auto">
              <table className="min-w-full text-sm">
                <thead>
                  <tr className="text-left border-b">
                    <th className="py-2 pr-4">ID</th>
                    <th className="py-2 pr-4">Name</th>
                    <th className="py-2 pr-4">Total</th>
                    <th className="py-2 pr-4">Pending</th>
                    <th className="py-2 pr-4">Failed</th>
                    <th className="py-2 pr-4">Created</th>
                    <th className="py-2 pr-4">Finished</th>
                  </tr>
                </thead>
                <tbody>
                  {jobBatches.map(b => (
                    <tr key={b.id} className="border-b last:border-0">
                      <td className="py-2 pr-4">{b.id}</td>
                      <td className="py-2 pr-4">{b.name}</td>
                      <td className="py-2 pr-4">{b.total_jobs}</td>
                      <td className="py-2 pr-4">{b.pending_jobs}</td>
                      <td className="py-2 pr-4">{b.failed_jobs}</td>
                      <td className="py-2 pr-4"><TimeCell value={b.created_at} /></td>
                      <td className="py-2 pr-4"><TimeCell value={b.finished_at} /></td>
                    </tr>
                  ))}
                  {jobBatches.length === 0 && (
                    <tr><td className="py-4 text-gray-500" colSpan={7}>No batches</td></tr>
                  )}
                </tbody>
              </table>
            </div>
          </CardContent>
        </Card>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <Card>
          <CardHeader>
            <CardTitle>Failed Jobs</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="overflow-x-auto">
              <table className="min-w-full text-sm">
                <thead>
                  <tr className="text-left border-b">
                    <th className="py-2 pr-4">ID</th>
                    <th className="py-2 pr-4">Queue</th>
                    <th className="py-2 pr-4">Failed At</th>
                    <th className="py-2 pr-4">Exception</th>
                  </tr>
                </thead>
                <tbody>
                  {failedJobs.map(f => (
                    <tr key={f.id} className="border-b last:border-0">
                      <td className="py-2 pr-4">{f.id}</td>
                      <td className="py-2 pr-4"><Badge variant="destructive">{f.queue}</Badge></td>
                      <td className="py-2 pr-4">{new Date(f.failed_at).toLocaleString()}</td>
                      <td className="py-2 pr-4 max-w-md truncate" title={f.exception}>{f.exception}</td>
                    </tr>
                  ))}
                  {failedJobs.length === 0 && (
                    <tr><td className="py-4 text-gray-500" colSpan={4}>No failed jobs</td></tr>
                  )}
                </tbody>
              </table>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Notifications Sent (Database)</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="overflow-x-auto">
              <table className="min-w-full text-sm">
                <thead>
                  <tr className="text-left border-b">
                    <th className="py-2 pr-4">ID</th>
                    <th className="py-2 pr-4">Type</th>
                    <th className="py-2 pr-4">Notifiable</th>
                    <th className="py-2 pr-4">Created</th>
                    <th className="py-2 pr-4">Message</th>
                  </tr>
                </thead>
                <tbody>
                  {notifications.map(n => (
                    <tr key={n.id} className="border-b last:border-0">
                      <td className="py-2 pr-4">{n.id}</td>
                      <td className="py-2 pr-4">{n.type.split('\\').pop()}</td>
                      <td className="py-2 pr-4">{n.notifiable_type.split('\\').pop()} #{n.notifiable_id}</td>
                      <td className="py-2 pr-4">{new Date(n.created_at).toLocaleString()}</td>
                      <td className="py-2 pr-4 max-w-md truncate" title={n?.data?.message || ''}>{n?.data?.message || '-'}</td>
                    </tr>
                  ))}
                  {notifications.length === 0 && (
                    <tr><td className="py-4 text-gray-500" colSpan={5}>No notifications</td></tr>
                  )}
                </tbody>
              </table>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
};

export default QueueMonitor;
