import { Badge } from '@/components/ui/badge';  
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Trophy, Users, Gamepad, User, BarChart3, Calendar, Award, Clock, Target, TrendingUp } from 'lucide-react';
import { usePermissions } from '@/hooks/use-permissions';
import { PermissionGate, CanView, CanCreate } from '@/components/auth';
import AppLayout from '@/layouts/app-layout';
import { AdminDashboard, OrganizerDashboard, RefereeDashboard, PlayerDashboard, TeamManagerDashboard } from '@/components/dashboard';
import { WelcomeFlow } from '@/components/onboarding/WelcomeFlow';
import { useOnboarding } from '@/hooks/use-onboarding';

interface Team {
  id: number;
  name: string;
  description?: string;
  logo?: string;
  website?: string;
  created_at: string;
}

interface DashboardProps {
  stats: {
    total_tournaments: number;
    active_tournaments: number;
    total_players?: number;
    total_teams?: number;
    total_matches?: number;
    completed_matches?: number;
    total_users?: number;
    recent_matches: any[];
  };
  recentActivity: {
    recent_tournaments?: any[];
    recent_players?: any[];
    recent_teams?: any[];
  };
  teams?: Team[];
  userPermissions: string[];
  userRoles: string[];
  user: {
    id: number;
    name: string;
    email: string;
    roles: string[];
    permissions: string[];
  } | null;
}

export default function Dashboard({ stats, recentActivity, teams, userPermissions, userRoles, user }: DashboardProps) {
  const { hasPermission, isAdmin, isOrganizer, isReferee, isPlayer, isTeamManager } = usePermissions();
  const { showOnboarding, completeOnboarding } = useOnboarding();

  // Determine primary role for role-specific content
  const getPrimaryRole = () => {
    if (isAdmin()) return 'admin';
    if (isOrganizer()) return 'organizer';
    if (isReferee()) return 'referee';
    if (isPlayer()) return 'player';
    if (isTeamManager()) return 'team_manager';
    return 'user';
  };

  const primaryRole = getPrimaryRole();

  // Role-specific welcome messages
  const getWelcomeMessage = () => {
    switch (primaryRole) {
      case 'admin':
        return {
          title: 'System Administration',
          subtitle: 'Manage tournaments, users, and system settings',
          icon: <Trophy className="w-8 h-8 text-green-600" />
        };
      case 'organizer':
        return {
          title: 'Tournament Management',
          subtitle: 'Create and manage tournaments, players, and teams',
          icon: <Award className="w-8 h-8 text-blue-600" />
        };
      case 'referee':
        return {
          title: 'Match Management',
          subtitle: 'Oversee matches, scores, and tournament progress',
          icon: <Target className="w-8 h-8 text-orange-600" />
        };
      case 'player':
        return {
          title: 'Player Dashboard',
          subtitle: 'View your tournaments, matches, and performance',
          icon: <User className="w-8 h-8 text-purple-600" />
        };
      case 'team_manager':
        return {
          title: 'Team Manager Dashboard',
          subtitle: 'View your team tournaments, matches, and performance',
          icon: <User className="w-8 h-8 text-purple-600" />
        };
      default:
        return {
          title: 'Welcome',
          subtitle: 'Access your tournament information',
          icon: <Trophy className="w-8 h-8 text-gray-600" />
        };
    }
  };

  const welcome = getWelcomeMessage();

  // Role-specific quick actions
  const getQuickActions = () => {
    switch (primaryRole) {
      case 'admin':
        return [
          { title: 'Manage Users', href: '/admin/users', icon: Users, permission: 'users.view' },
          { title: 'System Settings', href: '/admin/settings', icon: BarChart3, permission: 'settings.view' },
          { title: 'Role Management', href: '/admin/roles', icon: Users, permission: 'roles.view' },
          { title: 'Analytics', href: '/admin/analytics', icon: TrendingUp, permission: 'analytics.view' }
        ];
      case 'organizer':
        return [
          { title: 'Create Tournament', href: '/admin/tournaments/create', icon: Trophy, permission: 'tournaments.create' },
          { title: 'Add Players', href: '/admin/players/create', icon: User, permission: 'players.create' },
          { title: 'Create Teams', href: '/admin/teams/create', icon: Users, permission: 'teams.create' },
          { title: 'View Analytics', href: '/admin/analytics', icon: TrendingUp, permission: 'analytics.view' }
        ];
      case 'referee':
        return [
          { title: 'Schedule Match', href: '/admin/matches/create', icon: Gamepad, permission: 'matches.create' },
          { title: 'View Matches', href: '/admin/matches', icon: Clock, permission: 'matches.view' },
          { title: 'Tournament Overview', href: '/admin/tournaments', icon: Trophy, permission: 'tournaments.view' }
        ];
      case 'player':
        return [
          { title: 'My Tournaments', href: '/player/tournaments', icon: Trophy },
          { title: 'My Matches', href: '/player/matches', icon: Gamepad },
          { title: 'My Profile', href: '/player/profile', icon: User }
        ];
      case 'team_manager':
        return [
          { title: 'My Team', href: '/team/profile', icon: User },
          { title: 'My Matches', href: '/team/matches', icon: Gamepad },
          { title: 'My Profile', href: '/team/profile', icon: User }
        ];
      default:
        return [];
    }
  };

  const quickActions = getQuickActions();

  // Role-specific statistics
  const getRoleStats = () => {
    const baseStats = [
      { title: 'Active Tournaments', value: stats.active_tournaments, icon: Trophy, color: 'text-green-600' }
    ];

    switch (primaryRole) {
      case 'admin':
        return [
          ...baseStats,
          { title: 'Total Users', value: stats.total_users || 0, icon: Users, color: 'text-blue-600' },
          { title: 'Total Tournaments', value: stats.total_tournaments, icon: Trophy, color: 'text-purple-600' },
          { title: 'System Health', value: 'Good', icon: BarChart3, color: 'text-green-600' }
        ];
      case 'organizer':
        return [
          ...baseStats,
          { title: 'Total Players', value: stats.total_players || 0, icon: User, color: 'text-blue-600' },
          { title: 'Total Teams', value: stats.total_teams || 0, icon: Users, color: 'text-orange-600' },
          { title: 'Total Matches', value: stats.total_matches || 0, icon: Gamepad, color: 'text-purple-600' }
        ];
      case 'referee':
        return [
          ...baseStats,
          { title: 'Pending Matches', value: stats.total_matches ? stats.total_matches - (stats.completed_matches || 0) : 0, icon: Clock, color: 'text-orange-600' },
          { title: 'Completed Matches', value: stats.completed_matches || 0, icon: Award, color: 'text-green-600' },
          { title: 'Total Matches', value: stats.total_matches || 0, icon: Gamepad, color: 'text-purple-600' }
        ];
      case 'player':
        return [
          { title: 'My Tournaments', value: '3', icon: Trophy, color: 'text-blue-600' },
          { title: 'My Matches', value: '12', icon: Gamepad, color: 'text-green-600' },
          { title: 'Wins', value: '8', icon: Award, color: 'text-yellow-600' },
          { title: 'Win Rate', value: '67%', icon: TrendingUp, color: 'text-purple-600' }
        ];
      case 'team_manager':
        return [
          { title: 'My Team', value: '3', icon: User, color: 'text-blue-600' },
          { title: 'My Matches', value: '12', icon: Gamepad, color: 'text-green-600' },
          { title: 'Wins', value: '8', icon: Award, color: 'text-yellow-600' },
          { title: 'Win Rate', value: '67%', icon: TrendingUp, color: 'text-purple-600' }
        ];
      default:
        return baseStats;
    }
  };

  const roleStats = getRoleStats();

  return (
    <>
      <AppLayout>
        {/* Render role-specific dashboard components */}
        {primaryRole === 'admin' && (
          <AdminDashboard stats={stats} recentActivity={recentActivity} />
        )}
        {primaryRole === 'organizer' && (
          <OrganizerDashboard stats={stats} recentActivity={recentActivity} />
        )}
        {primaryRole === 'referee' && (
          <RefereeDashboard stats={stats} recentActivity={recentActivity} />
        )}
        {primaryRole === 'team_manager' && (
          <TeamManagerDashboard stats={stats} recentActivity={recentActivity} teams={teams} />
        )}
        {primaryRole === 'player' && (
          <PlayerDashboard stats={stats} />
        )}
        
          {primaryRole === 'user' && (
          <div className="space-y-6 p-4">
           
            {/* Basic statistics for all users */}
            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Active Tournaments</CardTitle>
                  <Trophy className="h-4 w-4 text-green-600" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{stats.active_tournaments}</div>
                </CardContent>
              </Card>
            </div>

            {/* Recent Matches - Show to all users */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Gamepad className="h-5 w-5" />
                  Recent Matches
                </CardTitle>
                <CardDescription>
                  Latest match activity
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  {stats.recent_matches && stats.recent_matches.length > 0 ? (
                    stats.recent_matches.map((match, index) => (
                      <div key={index} className="flex items-center justify-between">
                        <div>
                          <p className="text-sm font-medium">
                            {match.team1?.name || 'Team 1'} vs {match.team2?.name || 'Team 2'}
                          </p>
                          <p className="text-xs text-muted-foreground">
                            {match.tournament?.name || 'Unknown Tournament'}
                          </p>
                        </div>
                        <div className="flex items-center gap-2">
                          <Badge variant={match.status === 'completed' ? 'default' : 'secondary'}>
                            {match.status}
                          </Badge>
                          {match.status === 'completed' && (
                            <span className="text-sm font-medium">
                              {match.score_team1} - {match.score_team2}
                            </span>
                          )}
                        </div>
                      </div>
                    ))
                  ) : (
                    <p className="text-sm text-muted-foreground">No recent matches</p>
                  )}
                </div>
              </CardContent>
            </Card>
          </div>
        )}
      </AppLayout>

      {/* Show onboarding flow for new users */}
      {showOnboarding && (
        <WelcomeFlow onComplete={completeOnboarding} />
      )}
    </>
  );
}