import React from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { BarChart3, Users, Trophy, Gamepad, Target, TrendingUp, Award, User } from 'lucide-react';
import { usePermissions } from '@/hooks/use-permissions';
import AppLayout from '@/layouts/app-layout';

interface AnalyticsProps {
  analytics: any;
  userRoles: string[];
  userPermissions: string[];
  user: {
    id: number;
    name: string;
    email: string;
  };
}

export default function Analytics({ analytics, userRoles, userPermissions, user }: AnalyticsProps) {
  const { isAdmin, isOrganizer, isReferee, isPlayer } = usePermissions();

  const getPrimaryRole = () => {
    if (isAdmin()) return 'admin';
    if (isOrganizer()) return 'organizer';
    if (isReferee()) return 'referee';
    if (isPlayer()) return 'player';
    return 'user';
  };

  const primaryRole = getPrimaryRole();

  return (
    <AppLayout>
      <div className="space-y-6 p-4">
        {/* Header */}
        <div className="flex items-center gap-4 p-6 bg-gradient-to-r from-white to-gray-50 rounded-lg border">
          <div className="p-3 bg-blue-50 rounded-full">
            <BarChart3 className="w-8 h-8 text-blue-600" />
          </div>
          
        </div>

        {/* Role-specific Analytics */}
        {primaryRole === 'admin' && analytics.system && (
          <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-4">
            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Total Users</CardTitle>
                <Users className="h-4 w-4 text-blue-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{analytics.system.total_users}</div>
                <p className="text-xs text-muted-foreground">Registered users</p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Total Tournaments</CardTitle>
                <Trophy className="h-4 w-4 text-green-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{analytics.system.total_tournaments}</div>
                <p className="text-xs text-muted-foreground">All tournaments</p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Active Tournaments</CardTitle>
                <TrendingUp className="h-4 w-4 text-orange-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{analytics.system.active_tournaments}</div>
                <p className="text-xs text-muted-foreground">Currently running</p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Total Matches</CardTitle>
                <Gamepad className="h-4 w-4 text-purple-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{analytics.system.total_matches}</div>
                <p className="text-xs text-muted-foreground">All matches</p>
              </CardContent>
            </Card>
          </div>
        )}

        {primaryRole === 'organizer' && analytics.tournaments && (
          <div className="grid gap-6 md:grid-cols-3">
            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">My Tournaments</CardTitle>
                <Trophy className="h-4 w-4 text-blue-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{analytics.tournaments.total_tournaments}</div>
                <p className="text-xs text-muted-foreground">Total created</p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Active Tournaments</CardTitle>
                <TrendingUp className="h-4 w-4 text-green-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{analytics.tournaments.active_tournaments}</div>
                <p className="text-xs text-muted-foreground">Currently running</p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Total Players</CardTitle>
                <Users className="h-4 w-4 text-purple-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{analytics.tournaments.total_players}</div>
                <p className="text-xs text-muted-foreground">Participating</p>
              </CardContent>
            </Card>
          </div>
        )}

        {primaryRole === 'referee' && analytics.matches && (
          <div className="grid gap-6 md:grid-cols-3">
            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Matches Officiated</CardTitle>
                <Target className="h-4 w-4 text-orange-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{analytics.matches.total_matches_officiated}</div>
                <p className="text-xs text-muted-foreground">Total assigned</p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Completed Matches</CardTitle>
                <Award className="h-4 w-4 text-green-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{analytics.matches.completed_matches}</div>
                <p className="text-xs text-muted-foreground">Successfully finished</p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Pending Matches</CardTitle>
                <Gamepad className="h-4 w-4 text-blue-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{analytics.matches.pending_matches}</div>
                <p className="text-xs text-muted-foreground">Awaiting completion</p>
              </CardContent>
            </Card>
          </div>
        )}

        {primaryRole === 'player' && analytics.personal && (
          <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-4">
            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Total Matches</CardTitle>
                <Gamepad className="h-4 w-4 text-purple-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{analytics.personal.total_matches}</div>
                <p className="text-xs text-muted-foreground">Matches played</p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Wins</CardTitle>
                <Award className="h-4 w-4 text-green-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{analytics.personal.wins}</div>
                <p className="text-xs text-muted-foreground">Victories</p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Win Rate</CardTitle>
                <TrendingUp className="h-4 w-4 text-blue-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{analytics.personal.win_rate}%</div>
                <p className="text-xs text-muted-foreground">Success rate</p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Tournaments</CardTitle>
                <Trophy className="h-4 w-4 text-orange-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{analytics.personal.tournaments_participated}</div>
                <p className="text-xs text-muted-foreground">Participated in</p>
              </CardContent>
            </Card>
          </div>
        )}

        {/* General Analytics */}
        <Card>
          <CardHeader>
            <CardTitle>General Statistics</CardTitle>
            <CardDescription>Overall tournament system statistics</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="grid gap-4 md:grid-cols-3">
              <div className="flex items-center gap-3">
                <Trophy className="h-8 w-8 text-green-600" />
                <div>
                  <p className="text-sm font-medium">Total Tournaments</p>
                  <p className="text-2xl font-bold">{analytics.general.total_tournaments}</p>
                </div>
              </div>

              <div className="flex items-center gap-3">
                <TrendingUp className="h-8 w-8 text-blue-600" />
                <div>
                  <p className="text-sm font-medium">Active Tournaments</p>
                  <p className="text-2xl font-bold">{analytics.general.active_tournaments}</p>
                </div>
              </div>

              <div className="flex items-center gap-3">
                <Gamepad className="h-8 w-8 text-purple-600" />
                <div>
                  <p className="text-sm font-medium">Total Matches</p>
                  <p className="text-2xl font-bold">{analytics.general.total_matches}</p>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
    </AppLayout>
  );
}