import React, { useState } from 'react';
import { Head, useForm, router } from '@inertiajs/react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Trophy, Calendar, Users, DollarSign, Plus, ExternalLink, Clock, CheckCircle, XCircle, Link } from 'lucide-react';
import AppLayout from '@/layouts/app-layout';
import TeamSelector from '@/components/TeamManager/TeamSelector';

interface Tournament {
  id: number;
  name: string;
  description: string;
  status: string;
  start_date: string;
  end_date: string;
  registration_end: string;
  entry_fee: number;
  currency: string;
  max_teams: number;
  current_teams: number;
  game_type: string;
  location: string;
  matches_count?: number;
}

interface Team {
  id: number;
  name: string;
  tournaments: Array<{
    id: number;
    name: string;
    status: string;
    start_date: string;
    end_date: string;
    entry_fee: number;
    currency: string;
    registered_at: string;
    payment_status: string;
    verification_status: string;
    approval_status: string;
  }>;
}

interface TournamentsProps {
  team: Team;
  teams: Team[];
  tournaments: Team['tournaments'];
  availableTournaments: Tournament[];
}

export default function TeamTournaments({ team, teams, tournaments, availableTournaments }: TournamentsProps) {
  const [isRegisterOpen, setIsRegisterOpen] = useState(false);
  const [selectedTournament, setSelectedTournament] = useState<Tournament | null>(null);

  const { data, setData, post, processing, errors } = useForm({
    tournament_id: '',
  });

  const handleRegister = (tournament: Tournament) => {
    setSelectedTournament(tournament);
    setData('tournament_id', tournament.id.toString());
    setIsRegisterOpen(true);
  };

  const handleSubmitRegistration = (e: React.FormEvent) => {
    e.preventDefault();
    post(route('team.tournaments.register'), {
      onSuccess: () => {
        setIsRegisterOpen(false);
        setSelectedTournament(null);
      },
    });
  };

  const getStatusColor = (status: string) => {
    switch (status?.toLowerCase()) {
      case 'active':
        return 'bg-green-100 text-green-800';
      case 'upcoming':
        return 'bg-blue-100 text-blue-800';
      case 'completed':
        return 'bg-gray-100 text-gray-800';
      case 'cancelled':
        return 'bg-red-100 text-red-800';
      case 'pending_payment':
        return 'bg-yellow-100 text-yellow-800';
      case 'registered':
        return 'bg-green-100 text-green-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const getPaymentStatusColor = (status: string) => {
    switch (status?.toLowerCase()) {
      case 'paid':
      case 'completed':
        return 'bg-green-100 text-green-800';
      case 'pending':
        return 'bg-yellow-100 text-yellow-800';
      case 'failed':
        return 'bg-red-100 text-red-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const getVerificationStatusColor = (status: string) => {
    switch (status?.toLowerCase()) {
      case 'verified':
      case 'approved':
        return 'bg-green-100 text-green-800';
      case 'pending':
        return 'bg-yellow-100 text-yellow-800';
      case 'rejected':
        return 'bg-red-100 text-red-800';
      case 'not_verified':
      case 'unverified':
        return 'bg-gray-100 text-gray-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const getApprovalStatusColor = (status: string) => {
    switch (status?.toLowerCase()) {
      case 'first_approved':
        return 'bg-blue-100 text-blue-800';
      case 'second_approved':
      case 'approved':
        return 'bg-green-100 text-green-800';
      case 'pending':
        return 'bg-yellow-100 text-yellow-800';
      case 'rejected':
        return 'bg-red-100 text-red-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const formatCurrency = (amount: number, currency: string) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: currency || 'KES',
    }).format(amount);
  };

  const isRegistrationOpen = (tournament: Tournament) => {
    return new Date(tournament.registration_end) > new Date();
  };

  return (
    <AppLayout>
      <Head title="Team Tournaments" />
      
      <div className="space-y-6 p-6">
        {/* Team Selector */}
        <TeamSelector 
          teams={teams} 
          selectedTeamId={team.id} 
          currentPath="/team/tournaments" 
        />

        {/* Header */}
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-gray-900">Team Tournaments</h1>
            <p className="text-gray-600">Manage your team's tournament participation</p>
          </div>
        </div>

        {/* Current Tournaments */}
        <div>
          <h2 className="text-xl font-semibold text-gray-900 mb-4">Current Tournaments</h2>
          {tournaments.length > 0 ? (
            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
              {tournaments.map((tournament) => {
                const isPaid = tournament.payment_status?.toLowerCase() === 'paid' || tournament.payment_status?.toLowerCase() === 'completed';
                const isUnpaid = !isPaid && tournament.payment_status?.toLowerCase() !== 'pending';
                
                return (
                  <Card 
                    key={tournament.id} 
                    className={`hover:shadow-md transition-shadow ${
                      isUnpaid ? 'bg-red-50 border-red-200' : ''
                    }`}
                  >
                    <CardHeader>
                      <div className="flex items-center justify-between">
                        <CardTitle className="flex items-center gap-2">
                          <Trophy className="w-5 h-5" />
                          {tournament.name}
                        </CardTitle>
                        <Badge className={getStatusColor(tournament.status)}>
                          {tournament.status.replace('_', ' ')}
                        </Badge>
                      </div>
                      <CardDescription>
                        {formatCurrency(tournament.entry_fee, tournament.currency)}
                      </CardDescription>
                    </CardHeader>
                    <CardContent>
                      <div className="space-y-3">
                        <div className="flex items-center justify-between text-sm">
                          <span className="text-gray-600">Start Date</span>
                          <span className="font-medium">
                            {new Date(tournament.start_date).toLocaleDateString()}
                          </span>
                        </div>
                        <div className="flex items-center justify-between text-sm">
                          <span className="text-gray-600">End Date</span>
                          <span className="font-medium">
                            {new Date(tournament.end_date).toLocaleDateString()}
                          </span>
                        </div>
                        <div className="flex items-center justify-between text-sm">
                          <span className="text-gray-600">Registered</span>
                          <span className="font-medium">
                            {new Date(tournament.registered_at).toLocaleDateString()}
                          </span>
                        </div>
                        <div className="flex items-center justify-between text-sm">
                          <span className="text-gray-600">Payment Status</span>
                          <Badge className={getPaymentStatusColor(tournament.payment_status)}>
                            {tournament.payment_status?.replace('_', ' ') || 'Not Paid'}
                          </Badge>
                        </div>
                        <div className="flex items-center justify-between text-sm">
                          <span className="text-gray-600">Verification</span>
                          <Badge className={getVerificationStatusColor(tournament.verification_status)}>
                            {tournament.verification_status?.replace('_', ' ') || 'Not Verified'}
                          </Badge>
                        </div>
                        <div className="flex items-center justify-between text-sm">
                          <span className="text-gray-600">Approval</span>
                          <Badge className={getApprovalStatusColor(tournament.approval_status)}>
                            {tournament.approval_status?.replace('_', ' ') || 'Pending'}
                          </Badge>
                        </div>
                      </div>
                      
                      {/* Payment Warning for Unpaid Tournaments */}
                      {isUnpaid && (
                        <div className="mt-4 p-3 bg-red-100 border border-red-200 rounded-lg">
                          <div className="flex items-center gap-2 mb-1">
                            <XCircle className="w-4 h-4 text-red-600" />
                            <span className="font-medium text-red-800 text-sm">Payment Required</span>
                          </div>
                          <p className="text-xs text-red-700">
                            Verification will not be processed until payment is completed. 
                            Please complete payment <button onClick={() => router.visit(route('team.payments'))} className="text-blue-600">here</button> for your application to be processed. <br />
                            <br />
                            If you have already paid, please wait for the payment to be processed.
                            <br />
                            If you have any questions, please contact us at support@footballhub.co.ke
                          </p>
                        </div>
                      )}
                    </CardContent>
                  </Card>
                );
              })}
            </div>
          ) : (
            <Card>
              <CardContent className="text-center py-12">
                <Trophy className="w-12 h-12 text-gray-300 mx-auto mb-4" />
                <h3 className="text-lg font-semibold text-gray-900 mb-2">No Tournaments Yet</h3>
                <p className="text-gray-500">Your team hasn't registered for any tournaments yet</p>
              </CardContent>
            </Card>
          )}
        </div>

        {/* Available Tournaments */}
        <div>
          <h2 className="text-xl font-semibold text-gray-900 mb-4">Available Tournaments</h2>
          {availableTournaments.length > 0 ? (
            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
              {availableTournaments.map((tournament) => (
                <Card key={tournament.id} className="hover:shadow-md transition-shadow">
                  <CardHeader>
                    <div className="flex items-center justify-between">
                      <CardTitle className="flex items-center gap-2">
                        <Trophy className="w-5 h-5" />
                        {tournament.name}
                      </CardTitle>
                      <Badge className={getStatusColor(tournament.status)}>
                        {tournament.status}
                      </Badge>
                    </div>
                    <CardDescription>
                      {tournament.description}
                    </CardDescription>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-3">
                      <div className="flex items-center justify-between text-sm">
                        <span className="text-gray-600">Entry Fee</span>
                        <span className="font-medium">
                          {formatCurrency(tournament.entry_fee, tournament.currency)}
                        </span>
                      </div>
                      <div className="flex items-center justify-between text-sm">
                        <span className="text-gray-600">Start Date</span>
                        <span className="font-medium">
                          {new Date(tournament.start_date).toLocaleDateString()}
                        </span>
                      </div>
                      <div className="flex items-center justify-between text-sm">
                        <span className="text-gray-600">Registration Ends</span>
                        <span className="font-medium">
                          {new Date(tournament.registration_end).toLocaleDateString()}
                        </span>
                      </div>
                      <div className="flex items-center justify-between text-sm">
                        <span className="text-gray-600">Teams</span>
                        <span className="font-medium">
                          {tournament.current_teams} / {tournament.max_teams}
                        </span>
                      </div>
                      <div className="flex items-center justify-between text-sm">
                        <span className="text-gray-600">Location</span>
                        <span className="font-medium">{tournament.location}</span>
                      </div>
                      <div className="flex items-center justify-between text-sm">
                        <span className="text-gray-600">Game Type</span>
                        <span className="font-medium">{tournament.game_type}</span>
                      </div>
                    </div>
                    <div className="mt-4">
                      <Button
                        onClick={() => handleRegister(tournament)}
                        disabled={!isRegistrationOpen(tournament)}
                        className="w-full"
                      >
                        {isRegistrationOpen(tournament) ? (
                          <>
                            <Plus className="w-4 h-4 mr-2" />
                            Register Team
                          </>
                        ) : (
                          <>
                            <XCircle className="w-4 h-4 mr-2" />
                            Registration Closed
                          </>
                        )}
                      </Button>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          ) : (
            <Card>
              <CardContent className="text-center py-12">
                <Trophy className="w-12 h-12 text-gray-300 mx-auto mb-4" />
                <h3 className="text-lg font-semibold text-gray-900 mb-2">No Available Tournaments</h3>
                <p className="text-gray-500">There are no tournaments available for registration at the moment</p>
              </CardContent>
            </Card>
          )}
        </div>

        {/* Registration Dialog */}
        <Dialog open={isRegisterOpen} onOpenChange={setIsRegisterOpen}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Register for Tournament</DialogTitle>
              <DialogDescription>
                Register {team.name} for {selectedTournament?.name}
              </DialogDescription>
            </DialogHeader>
            {selectedTournament && (
              <div className="space-y-4">
                <div className="bg-gray-50 p-4 rounded-lg">
                  <h3 className="font-semibold mb-2">Tournament Details</h3>
                  <div className="space-y-2 text-sm">
                    <div className="flex justify-between">
                      <span>Entry Fee:</span>
                      <span className="font-medium">
                        {formatCurrency(selectedTournament.entry_fee, selectedTournament.currency)}
                      </span>
                    </div>
                    <div className="flex justify-between">
                      <span>Start Date:</span>
                      <span className="font-medium">
                        {new Date(selectedTournament.start_date).toLocaleDateString()}
                      </span>
                    </div>
                    <div className="flex justify-between">
                      <span>End Date:</span>
                      <span className="font-medium">
                        {new Date(selectedTournament.end_date).toLocaleDateString()}
                      </span>
                    </div>
                    <div className="flex justify-between">
                      <span>Location:</span>
                      <span className="font-medium">{selectedTournament.location}</span>
                    </div>
                  </div>
                </div>

                <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
                  <div className="flex items-center gap-2 mb-2">
                    <Clock className="w-4 h-4 text-yellow-600" />
                    <span className="font-medium text-yellow-800">Important Notice</span>
                  </div>
                  <p className="text-sm text-yellow-700">
                    After registration, you'll need to complete payment to confirm your team's participation.
                    Payment must be completed before the tournament starts.
                  </p>
                </div>

                <form onSubmit={handleSubmitRegistration}>
                  <DialogFooter>
                    <Button type="button" variant="outline" onClick={() => setIsRegisterOpen(false)}>
                      Cancel
                    </Button>
                    <Button type="submit" disabled={processing}>
                      {processing ? 'Registering...' : 'Confirm Registration'}
                    </Button>
                  </DialogFooter>
                </form>
              </div>
            )}
          </DialogContent>
        </Dialog>
      </div>
    </AppLayout>
  );
}
