import React, { useState, useEffect } from 'react';
import { Head, Link, usePage } from '@inertiajs/react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Download, Printer, ArrowLeft, Eye, Trophy, Users, Calendar } from 'lucide-react';
import { toast } from 'sonner';
import AppLayout from '@/layouts/app-layout';

interface Team {
    id: number;
    name: string;
    description?: string;
    logo?: string;
    created_at: string;
    players: Array<{
        id: number;
        user: {
            name: string;
            email: string;
        };
        display_name?: string;
        football_position?: string;
        jersey_number?: number;
    }>;
    tournaments: Array<{
        id: number;
        name: string;
        status: string;
        start_date: string;
        end_date: string;
    }>;
}

interface TeamStats {
    total_players: number;
    verified_players: number;
    active_tournaments: number;
    completed_tournaments: number;
    total_matches: number;
    team_win_rate: number;
}

interface Props {
    team: Team;
    teams: Team[];
    user: {
        id: number;
        name: string;
        email: string;
    };
    teamStats: TeamStats;
}

export default function TeamCertificate({ team, teams, user, teamStats }: Props) {
    const [isLoading, setIsLoading] = useState(false);
    const [certificateUrl, setCertificateUrl] = useState<string | null>(null);

    const handleViewCertificate = async () => {
        setIsLoading(true);
        try {
            const response = await fetch(route('team.certificate.pdf'));
            if (response.ok) {
                // Get the PDF blob
                const blob = await response.blob();
                const url = window.URL.createObjectURL(blob);
                setCertificateUrl(url);
                // Open PDF in new tab
                window.open(url, '_blank');
            } else {
                toast.error('Failed to load certificate');
            }
        } catch (error) {
            toast.error('Error loading certificate');
        } finally {
            setIsLoading(false);
        }
    };

    const handleDownloadCertificate = async () => {
        setIsLoading(true);
        try {
            const response = await fetch(route('team.certificate.pdf'));
            if (response.ok) {
                // Get the PDF blob
                const blob = await response.blob();
                const url = window.URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = `team-certificate-${team.name.replace(/\s+/g, '-').toLowerCase()}.pdf`;
                document.body.appendChild(a);
                a.click();
                window.URL.revokeObjectURL(url);
                document.body.removeChild(a);
                toast.success('Team certificate downloaded successfully');
            } else {
                toast.error('Failed to download certificate');
            }
        } catch (error) {
            toast.error('Error downloading certificate');
        } finally {
            setIsLoading(false);
        }
    };

    const handlePrintCertificate = () => {
        if (certificateUrl) {
            const printWindow = window.open(certificateUrl, '_blank');
            if (printWindow) {
                printWindow.onload = () => {
                    printWindow.print();
                };
            }
        } else {
            toast.error('Please view certificate first');
        }
    };

    const activeTournaments = team.tournaments.filter(t => t.status === 'active');

    return (
        <AppLayout breadcrumbs={[{ title: 'Team Certificate', href: '/team/certificate' }]}>
            <Head title={`Team Certificate - ${team.name}`} />
    
                <div className="">
                    {/* Header */}
                    <div className="mb-8">
                        <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-4">
                                <Link
                                    href="/team/profile"
                                    className="inline-flex items-center px-4 py-2 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50"
                                >
                                    <ArrowLeft className="h-4 w-4 mr-2" />
                                    Back to Team Profile
                                </Link>
                                <div>
                                    <h1 className="text-3xl font-bold text-gray-900">
                                        Team Certificate
                                    </h1>
                                    <p className="text-gray-600">
                                        Generate and manage team certificate for {team.name}
                                    </p>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Team Info Card */}
                    <Card className="mb-8">
                        <CardHeader>
                            <CardTitle className="flex items-center justify-between">
                                <span className="flex items-center">
                                    <Trophy className="h-5 w-5 mr-2" />
                                    Team Information
                                </span>
                                <Badge variant="default">
                                    {team.players.length} Players
                                </Badge>
                            </CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                                <div className="flex items-center space-x-4">
                                    <div className="flex-shrink-0">
                                        {team.logo ? (
                                            <img
                                                className="h-16 w-16 rounded-full object-cover"
                                                src={team.logo}
                                                alt={team.name}
                                            />
                                        ) : (
                                            <div className="h-16 w-16 rounded-full bg-gray-300 flex items-center justify-center">
                                                <span className="text-gray-600 font-medium text-lg">
                                                    {team.name.charAt(0).toUpperCase()}
                                                </span>
                                            </div>
                                        )}
                                    </div>
                                    <div>
                                        <h3 className="text-lg font-medium text-gray-900">{team.name}</h3>
                                        <p className="text-gray-600">Team Manager: {user.name}</p>
                                    </div>
                                </div>
                                
                                <div className="space-y-2">
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Total Players:</span>
                                        <p className="text-gray-900">{teamStats.total_players}</p>
                                    </div>
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Verified Players:</span>
                                        <p className="text-gray-900">{teamStats.verified_players}</p>
                                    </div>
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Active Tournaments:</span>
                                        <p className="text-gray-900">{teamStats.active_tournaments}</p>
                                    </div>
                                </div>
                                
                                <div className="space-y-2">
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Team Created:</span>
                                        <p className="text-gray-900">{new Date(team.created_at).toLocaleDateString()}</p>
                                    </div>
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Total Matches:</span>
                                        <p className="text-gray-900">{teamStats.total_matches}</p>
                                    </div>
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Team Win Rate:</span>
                                        <p className="text-gray-900">{teamStats.team_win_rate.toFixed(1)}%</p>
                                    </div>
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Team ID:</span>
                                        <p className="text-gray-900 font-mono text-sm">TEAM-{team.id.toString().padStart(6, '0')}</p>
                                    </div>
                                </div>
                            </div>

                            {team.description && (
                                <div className="mt-6">
                                    <h4 className="text-sm font-medium text-gray-500 mb-2">Description</h4>
                                    <p className="text-gray-700">{team.description}</p>
                                </div>
                            )}
                        </CardContent>
                    </Card>

                    {/* Team Statistics */}
                    <Card className="mb-8">
                        <CardHeader>
                            <CardTitle className="flex items-center">
                                <Trophy className="h-5 w-5 mr-2" />
                                Team Statistics
                            </CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                                <div className="text-center p-4 bg-blue-50 rounded-lg">
                                    <div className="text-2xl font-bold text-blue-600">{teamStats.total_players}</div>
                                    <div className="text-sm text-blue-800">Total Players</div>
                                </div>
                                <div className="text-center p-4 bg-green-50 rounded-lg">
                                    <div className="text-2xl font-bold text-green-600">{teamStats.verified_players}</div>
                                    <div className="text-sm text-green-800">Verified Players</div>
                                </div>
                                <div className="text-center p-4 bg-purple-50 rounded-lg">
                                    <div className="text-2xl font-bold text-purple-600">{teamStats.total_matches}</div>
                                    <div className="text-sm text-purple-800">Total Matches</div>
                                </div>
                                <div className="text-center p-4 bg-orange-50 rounded-lg">
                                    <div className="text-2xl font-bold text-orange-600">{teamStats.team_win_rate.toFixed(1)}%</div>
                                    <div className="text-sm text-orange-800">Win Rate</div>
                                </div>
                            </div>
                        </CardContent>
                    </Card>

                    {/* Team Players */}
                    <Card className="mb-8">
                        <CardHeader>
                            <CardTitle className="flex items-center">
                                <Users className="h-5 w-5 mr-2" />
                                Team Players
                            </CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                                {team.players.map((player) => {
                                    const playerName = player.display_name || player.user.name;
                                    return (
                                        <div key={player.id} className="flex items-center justify-between p-3 border rounded-lg">
                                            <div className="flex items-center space-x-3 min-w-0">
                                                <div className="flex-shrink-0">
                                                    <div className="h-10 w-10 rounded-full bg-gray-300 flex items-center justify-center">
                                                        <span className="text-gray-600 font-medium text-sm">
                                                            {playerName.charAt(0).toUpperCase()}
                                                        </span>
                                                    </div>
                                                </div>
                                                <div className="flex-1 min-w-0">
                                                    <h4 className="text-sm font-medium text-gray-900 truncate">
                                                        {playerName}
                                                    </h4>
                                                    <p className="text-xs text-gray-600">
                                                        {player.football_position} • {player.user.email}
                                                    </p>
                                                    {player.jersey_number && (
                                                        <Badge variant="outline" className="text-xs mt-1">
                                                            #{player.jersey_number}
                                                        </Badge>
                                                    )}
                                                </div>
                                            </div>
                                            <div className="flex-shrink-0">
                                                <Button
                                                    variant="outline"
                                                    size="sm"
                                                    onClick={() => window.open(route('team.players.certificate.pdf', player.id), '_blank')}
                                                >
                                                    <Eye className="h-3.5 w-3.5 mr-1" />
                                                    View Certificate
                                                </Button>
                                            </div>
                                        </div>
                                    );
                                })}
                            </div>
                        </CardContent>
                    </Card>

                    {/* Active Tournaments */}
                    {activeTournaments.length > 0 && (
                        <Card className="mb-8">
                            <CardHeader>
                                <CardTitle className="flex items-center">
                                    <Calendar className="h-5 w-5 mr-2" />
                                    Active Tournaments
                                </CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="space-y-3">
                                    {activeTournaments.map((tournament) => (
                                        <div key={tournament.id} className="flex items-center justify-between p-3 border rounded-lg">
                                            <div>
                                                <h4 className="text-sm font-medium text-gray-900">{tournament.name}</h4>
                                                <p className="text-xs text-gray-600">
                                                    {new Date(tournament.start_date).toLocaleDateString()} - {new Date(tournament.end_date).toLocaleDateString()}
                                                </p>
                                            </div>
                                            <Badge variant="default">{tournament.status}</Badge>
                                        </div>
                                    ))}
                                </div>
                            </CardContent>
                        </Card>
                    )}

                    {/* Certificate Actions */}
                    <Card>
                        <CardHeader>
                            <CardTitle>Certificate Actions</CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                                <Button
                                    onClick={handleViewCertificate}
                                    disabled={isLoading}
                                    className="w-full"
                                    variant="outline"
                                >
                                    <Eye className="h-4 w-4 mr-2" />
                                    {isLoading ? 'Loading...' : 'View Certificate'}
                                </Button>                                
                                <Button
                                    onClick={handleDownloadCertificate}
                                    disabled={isLoading}
                                    className="w-full"
                                    variant="outline"
                                >
                                    <Download className="h-4 w-4 mr-2" />
                                    {isLoading ? 'Downloading...' : 'Download Certificate'}
                                </Button>                                
                                <Button
                                    onClick={handlePrintCertificate}
                                    disabled={!certificateUrl || isLoading}
                                    className="w-full"
                                    variant="outline"
                                >
                                    <Printer className="h-4 w-4 mr-2" />
                                    Print Certificate
                                </Button>
                            </div>
                        </CardContent>
                    </Card>

                    {/* Certificate Preview */}
                    {certificateUrl && (
                        <Card className="mt-8">
                            <CardHeader>
                                <CardTitle>Certificate Preview</CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="border rounded-lg p-4 bg-white">
                                    <iframe
                                        src={certificateUrl}
                                        className="w-full h-96 border-0"
                                        title="Team Certificate Preview"
                                    />
                                </div>
                            </CardContent>
                        </Card>
                    )}
                </div>
            
        </AppLayout>
    );
}
