import React, { useState } from 'react';
import { Head, useForm, router } from '@inertiajs/react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Badge } from '@/components/ui/badge';
import { UsersRound, Trophy, Calendar, Globe, Camera, Save, Edit, Award, IdCard, FileText, Users, CheckCircle, Clock, XCircle } from 'lucide-react';
import AppLayout from '@/layouts/app-layout';
import TeamSelector from '@/components/TeamManager/TeamSelector';
import { toast } from 'sonner';
import { CategoryBadge } from '@/components/ui/category-badge';

interface Team {
  id: number;
  name: string;
  description?: string;
  logo?: string;
  website?: string;
  category?: 'men' | 'women' | 'youth_male' | 'youth_female' | 'children';
  social_media?: {
    twitter?: string;
    facebook?: string;
    instagram?: string;
  };
  created_at: string;
  players: Array<{
    id: number;
    user: {
      name: string;
      email: string;
    };
    position: string;
    jersey_number?: number;
    status: string;
  }>;
  tournaments: Array<{
    id: number;
    name: string;
    status: string;
    start_date: string;
    end_date: string;
  }>;
}

interface TournamentRegistration {
  id: number;
  name: string;
  status: string;
  start_date: string;
  end_date: string;
  entry_fee: number;
  currency: string;
  category: string;
  category_name: string;
  category_fee: number;
  payment_status: string;
  approval_status: string;
  registered_at: string;
}

interface CategoryConfig {
  name: string;
  age_min?: number;
  age_max?: number;
  gender?: string;
  icon: string;
  color: string;
  description: string;
}

interface ProfileProps {
  team: Team;
  teams: Team[];
  user: {
    id: number;
    name: string;
    email: string;
  };
  tournamentsByCategory?: Record<string, TournamentRegistration[]>;
  categoryConfig?: CategoryConfig;
}

export default function TeamProfile({ team, teams, user, tournamentsByCategory = {}, categoryConfig }: ProfileProps) {
  const [isEditing, setIsEditing] = useState(false);
  
  const { data, setData, post, processing, errors } = useForm({
    name: team.name,
    description: team.description || '',
    website: team.website || '',
    social_media: {
      twitter: team.social_media?.twitter || '',
      facebook: team.social_media?.facebook || '',
      instagram: team.social_media?.instagram || '',
    },
    logo: null as File | null,
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    
    // Log form data for debugging
    console.log('Submitting team profile data:', data);
    
    // Use POST with forceFormData for proper file upload handling
    // Inertia will add _method=PUT automatically when using post on a PUT route
    post(route('team.profile.update'), {
      forceFormData: true,
      preserveScroll: true,
      onSuccess: (response) => {
        setIsEditing(false);
        toast.success('Team profile updated successfully!');
      },
      onError: (errors) => {
        console.log('Validation errors:', errors);
        
        // Format errors for display
        const errorEntries = Object.entries(errors);
        let errorMessage = '';
        errorEntries.slice(0, 3).forEach(([field, messages]) => {
          const fieldName = field.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
          const errorMsg = Array.isArray(messages) ? messages[0] : messages;
          errorMessage += `• ${errorMsg}\n`;
        });
        
        if (errorEntries.length > 3) {
          errorMessage += `\n... and ${errorEntries.length - 3} more error(s)`;
        }
        
        toast.error(`Validation Failed (${errorEntries.length} error${errorEntries.length > 1 ? 's' : ''})`, {
          description: errorMessage.trim(),
          duration: 5000,
        });
      },
    });
  };

  const getStatusColor = (status: string) => {
    switch (status?.toLowerCase()) {
      case 'active':
        return 'bg-green-100 text-green-800';
      case 'pending':
        return 'bg-yellow-100 text-yellow-800';
      case 'inactive':
        return 'bg-red-100 text-red-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  return (
    <AppLayout>
      <Head title="Team Profile" />
      
      <div className="space-y-6 p-6">
        {/* Team Selector */}
        <TeamSelector 
          teams={teams} 
          selectedTeamId={team.id} 
          currentPath="/team/profile" 
        />

        {/* Header */}
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-gray-900">Team Profile</h1>
            <p className="text-gray-600">Manage your team information and settings</p>
          </div>
          <Button
            onClick={() => setIsEditing(!isEditing)}
            variant={isEditing ? "outline" : "default"}
          >
            {isEditing ? <Edit className="w-4 h-4 mr-2" /> : <Edit className="w-4 h-4 mr-2" />}
            {isEditing ? 'Cancel' : 'Edit Profile'}
          </Button>
        </div>

        <div className="grid gap-6 lg:grid-cols-3">
          {/* Team Information */}
          <div className="lg:col-span-2">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <UsersRound className="w-5 h-5" />
                  Team Information
                </CardTitle>
                <CardDescription>
                  Basic information about your team
                </CardDescription>
              </CardHeader>
              <CardContent>
                {isEditing ? (
                  <form onSubmit={handleSubmit} className="space-y-4">
                    <div className="grid gap-4 md:grid-cols-2">
                      <div>
                        <Label htmlFor="name">Team Name *</Label>
                        <Input
                          id="name"
                          value={data.name}
                          onChange={(e) => setData('name', e.target.value)}
                          className={errors.name ? 'border-red-500' : ''}
                        />
                        {errors.name && <p className="text-red-500 text-sm mt-1">{errors.name}</p>}
                      </div>
                      <div>
                        <Label htmlFor="website">Website</Label>
                        <Input
                          id="website"
                          type="url"
                          value={data.website}
                          onChange={(e) => setData('website', e.target.value)}
                          placeholder="https://example.com"
                        />
                      </div>
                    </div>

                    <div>
                      <Label htmlFor="description">Description</Label>
                      <Textarea
                        id="description"
                        value={data.description}
                        onChange={(e) => setData('description', e.target.value)}
                        rows={4}
                        placeholder="Tell us about your team..."
                      />
                    </div>

                    <div>
                      <Label htmlFor="logo">Team Logo</Label>
                      <Input
                        id="logo"
                        type="file"
                        accept="image/*"
                        onChange={(e) => setData('logo', e.target.files?.[0] || null)}
                      />
                    </div>

                    <div className="grid gap-4 md:grid-cols-3">
                      <div>
                        <Label htmlFor="twitter">Twitter</Label>
                        <Input
                          id="twitter"
                          value={data.social_media.twitter}
                          onChange={(e) => setData('social_media', { ...data.social_media, twitter: e.target.value })}
                          placeholder="@teamhandle"
                        />
                      </div>
                      <div>
                        <Label htmlFor="facebook">Facebook</Label>
                        <Input
                          id="facebook"
                          value={data.social_media.facebook}
                          onChange={(e) => setData('social_media', { ...data.social_media, facebook: e.target.value })}
                          placeholder="facebook.com/team"
                        />
                      </div>
                      <div>
                        <Label htmlFor="instagram">Instagram</Label>
                        <Input
                          id="instagram"
                          value={data.social_media.instagram}
                          onChange={(e) => setData('social_media', { ...data.social_media, instagram: e.target.value })}
                          placeholder="@teamhandle"
                        />
                      </div>
                    </div>

                    <div className="flex gap-2">
                      <Button type="submit" disabled={processing}>
                        <Save className="w-4 h-4 mr-2" />
                        {processing ? 'Saving...' : 'Save Changes'}
                      </Button>
                      <Button type="button" variant="outline" onClick={() => setIsEditing(false)}>
                        Cancel
                      </Button>
                    </div>
                  </form>
                ) : (
                  <div className="space-y-4">
                    <div className="flex items-center gap-4">
                      {team.logo && (
                        <img
                          src={team.logo.startsWith('http') ? team.logo : `/storage/${team.logo}`}
                          alt={team.name}
                          className="w-16 h-16 rounded-full object-cover"
                        />
                      )}
                      <div className="flex-1">
                        <div className="flex items-center gap-2">
                          <h2 className="text-2xl font-bold">{team.name}</h2>
                          {team.category && (
                            <CategoryBadge category={team.category} />
                          )}
                        </div>
                        <p className="text-gray-600">Team Manager: {user.name}</p>
                        {categoryConfig && (
                          <p className="text-sm text-gray-500 mt-1">
                            {categoryConfig.description}
                          </p>
                        )}
                      </div>
                    </div>

                    {team.description && (
                      <div>
                        <h3 className="font-semibold text-gray-900 mb-2">Description</h3>
                        <p className="text-gray-700">{team.description}</p>
                      </div>
                    )}

                    {team.website && (
                      <div>
                        <h3 className="font-semibold text-gray-900 mb-2">Website</h3>
                        <a
                          href={team.website}
                          target="_blank"
                          rel="noopener noreferrer"
                          className="text-blue-600 hover:text-blue-800 flex items-center gap-1"
                        >
                          <Globe className="w-4 h-4" />
                          {team.website}
                        </a>
                      </div>
                    )}

                    {(team.social_media?.twitter || team.social_media?.facebook || team.social_media?.instagram) && (
                      <div>
                        <h3 className="font-semibold text-gray-900 mb-2">Social Media</h3>
                        <div className="flex gap-4">
                          {team.social_media?.twitter && (
                            <a
                              href={`https://twitter.com/${team.social_media.twitter.replace('@', '')}`}
                              target="_blank"
                              rel="noopener noreferrer"
                              className="text-blue-400 hover:text-blue-600"
                            >
                              Twitter
                            </a>
                          )}
                          {team.social_media?.facebook && (
                            <a
                              href={team.social_media.facebook}
                              target="_blank"
                              rel="noopener noreferrer"
                              className="text-blue-600 hover:text-blue-800"
                            >
                              Facebook
                            </a>
                          )}
                          {team.social_media?.instagram && (
                            <a
                              href={`https://instagram.com/${team.social_media.instagram.replace('@', '')}`}
                              target="_blank"
                              rel="noopener noreferrer"
                              className="text-pink-600 hover:text-pink-800"
                            >
                              Instagram
                            </a>
                          )}
                        </div>
                      </div>
                    )}
                  </div>
                )}
              </CardContent>
            </Card>
          </div>

          {/* Team Stats */}
          <div className="space-y-6">
            {/* Quick Stats */}
            <Card>
              <CardHeader>
                <CardTitle>Team Statistics</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-center justify-between">
                  <span className="text-sm font-medium">Total Players</span>
                  <span className="text-2xl font-bold">{team.players.length}</span>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-sm font-medium">Active Tournaments</span>
                  <span className="text-2xl font-bold">
                    {team.tournaments.filter(t => t.status === 'active').length}
                  </span>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-sm font-medium">Team Created</span>
                  <span className="text-sm text-gray-600">
                    {new Date(team.created_at).toLocaleDateString()}
                  </span>
                </div>
              </CardContent>
            </Card>

            {/* Tournament Registrations by Category */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Trophy className="w-5 h-5" />
                  Tournament Registrations
                </CardTitle>
                <CardDescription>
                  Organized by team category
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {Object.entries(tournamentsByCategory).map(([category, tournaments]) => (
                    <div key={category} className="space-y-2">
                      <div className="flex items-center gap-2">
                        <CategoryBadge category={category as any} size="sm" />
                        <span className="text-xs text-gray-500">
                          {tournaments.length} tournament{tournaments.length !== 1 ? 's' : ''}
                        </span>
                      </div>
                      <div className="space-y-2 pl-4 border-l-2 border-gray-200">
                        {tournaments.slice(0, 3).map((tournament) => (
                          <div key={tournament.id} className="space-y-1">
                            <div className="flex items-center justify-between">
                              <p className="text-sm font-medium">{tournament.name}</p>
                              <Badge className={getStatusColor(tournament.status)} variant="outline">
                                {tournament.status}
                              </Badge>
                            </div>
                            <div className="flex items-center gap-3 text-xs text-gray-500">
                              <span className="flex items-center gap-1">
                                <Calendar className="w-3 h-3" />
                                {new Date(tournament.start_date).toLocaleDateString()}
                              </span>
                              <span className="font-medium text-gray-700">
                                {tournament.currency} {tournament.category_fee}
                              </span>
                              {tournament.payment_status === 'completed' ? (
                                <span className="flex items-center gap-1 text-green-600">
                                  <CheckCircle className="w-3 h-3" />
                                  Paid
                                </span>
                              ) : (
                                <span className="flex items-center gap-1 text-orange-600">
                                  <Clock className="w-3 h-3" />
                                  Pending
                                </span>
                              )}
                            </div>
                          </div>
                        ))}
                        {tournaments.length > 3 && (
                          <p className="text-xs text-gray-500 italic">
                            +{tournaments.length - 3} more
                          </p>
                        )}
                      </div>
                    </div>
                  ))}
                  {Object.keys(tournamentsByCategory).length === 0 && (
                    <p className="text-sm text-gray-500">No tournament registrations yet</p>
                  )}
                </div>
              </CardContent>
            </Card>

            {/* Certificates & ID Cards */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <FileText className="w-5 h-5" />
                  Certificates & ID Cards
                </CardTitle>
                <CardDescription>
                  Generate and manage team documents
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-3">
                <Button
                  variant="outline"
                  className="w-full justify-start"
                  onClick={() => router.visit(route('team.certificate.view'))}
                >
                  <Trophy className="w-4 h-4 mr-2" />
                  Team Certificate
                </Button>
                
                <Button
                  variant="outline"
                  className="w-full justify-start"
                  onClick={() => router.visit(route('team.players'))}
                >
                  <IdCard className="w-4 h-4 mr-2" />
                  Player ID Cards
                </Button>
                
                <div className="space-y-2">
                  <p className="text-sm font-medium text-gray-700">Individual Player Certificates</p>
                  <div className="space-y-1">
                    {team.players.slice(0, 3).map((player) => (
                      <Button
                        key={player.id}
                        variant="ghost"
                        size="sm"
                        className="w-full justify-start text-sm"
                        onClick={() => router.visit(route('team.players.certificate.view', player.id))}
                      >
                        <Award className="w-3 h-3 mr-2" />
                        {player.user.name}
                      </Button>
                    ))}
                    {team.players.length > 3 && (
                      <p className="text-xs text-gray-500 text-center">
                        +{team.players.length - 3} more players
                      </p>
                    )}
                    {team.players.length === 0 && (
                      <p className="text-xs text-gray-500 text-center">No players yet</p>
                    )}
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </AppLayout>
  );
}
