import React from 'react';
import { Head, router } from '@inertiajs/react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Users, Trash2, Edit, User, Mail, Phone, Shirt, Eye, UserPlus, Award, IdCard, Download, Copy, CheckCircle, FileText, Camera, CheckCircle2, XCircle, Clock } from 'lucide-react';
import AppLayout from '@/layouts/app-layout';
import TeamSelector from '@/components/TeamManager/TeamSelector';

interface Player {
  id: number;
  user: {
    id: number;
    name: string;
    email: string;
    phone?: string;
  };
  display_name?: string;
  avatar?: string;
  position: string;
  football_position?: string;
  jersey_number?: number;
  status: string;
  verification_status?: string;
  digital_id?: string;
  id_number?: string;
  birth_certificate_number?: string;
  passport_photo?: string;
  birth_cert_photo?: string;
  approval_status?: string;
  achievements?: Array<{
    name: string;
    earned_at: string;
    data?: any;
  }>;
  tournament_matches_played?: number;
  tournament_matches_won?: number;
  tournament_win_rate?: number;
  global_rating?: number;
  global_rank?: number;
  player_type?: string;
  experience_level?: string;
  created_at: string;
}

interface Team {
  id: number;
  name: string;
  players: Player[];
}

interface PlayersProps {
  team: Team;
  teams: Team[];
  players: Player[];
}

export default function TeamPlayers({ team, teams, players }: PlayersProps) {
  const handleRemovePlayer = (playerId: number) => {
    if (confirm('Are you sure you want to remove this player from the team?')) {
      router.delete(route('team.players.remove', playerId));
    }
  };

  const handleCopyDigitalId = (digitalId: string) => {
    navigator.clipboard.writeText(digitalId);
    // You could add a toast notification here
  };

  const handleDownloadCertificate = (playerId: number) => {
    // This would trigger a certificate download
    router.get(route('team.players.certificate', playerId));
  };

  const getStatusColor = (status: string) => {
    switch (status?.toLowerCase()) {
      case 'active':
        return 'bg-green-100 text-green-800';
      case 'inactive':
        return 'bg-red-100 text-red-800';
      case 'suspended':
        return 'bg-yellow-100 text-yellow-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const getPositionColor = (position: string) => {
    switch (position?.toLowerCase()) {
      case 'goalkeeper':
        return 'bg-blue-100 text-blue-800';
      case 'defender':
        return 'bg-green-100 text-green-800';
      case 'midfielder':
        return 'bg-yellow-100 text-yellow-800';
      case 'forward':
      case 'striker':
        return 'bg-red-100 text-red-800';
      case 'captain':
        return 'bg-purple-100 text-purple-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const getApprovalStatusColor = (status: string) => {
    switch (status?.toLowerCase()) {
      case 'approved':
        return 'bg-green-100 text-green-800';
      case 'rejected':
        return 'bg-red-100 text-red-800';
      case 'pending':
        return 'bg-yellow-100 text-yellow-800';
      case 'under_review':
        return 'bg-blue-100 text-blue-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const getApprovalStatusIcon = (status: string) => {
    switch (status?.toLowerCase()) {
      case 'approved':
        return <CheckCircle2 className="w-4 h-4 text-green-600" />;
      case 'rejected':
        return <XCircle className="w-4 h-4 text-red-600" />;
      case 'pending':
        return <Clock className="w-4 h-4 text-yellow-600" />;
      case 'under_review':
        return <Eye className="w-4 h-4 text-blue-600" />;
      default:
        return <Clock className="w-4 h-4 text-gray-600" />;
    }
  };

  return (
    <AppLayout>
      <Head title="Team Players" />
      
      <div className="space-y-6 p-6">
        {/* Team Selector */}
        <TeamSelector 
          teams={teams} 
          selectedTeamId={team.id} 
          currentPath="/team/players" 
        />

        {/* Header */}
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-gray-900">Team Players</h1>
            <p className="text-gray-600">Manage your team roster and player information</p>
          </div>
          <div className="flex gap-2">
            <Button
              variant="outline"
              onClick={() => router.visit(route('team.players.id-cards'))}
            >
              <IdCard className="w-4 h-4 mr-2" />
              ID Cards
            </Button>
            <Button
              variant="outline"
              onClick={() => router.visit(route('team.players.create'))}
            >
              <UserPlus className="w-4 h-4 mr-2" />
              Create Player
            </Button>
          </div>
        </div>

        {/* Players Grid */}
        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
          {players.map((player) => (
            <Card key={player.id} className="hover:shadow-md transition-shadow">
              <CardHeader className="pb-3">
                <div className="flex items-center justify-between">
                  <div className="flex items-center gap-3">
                    <div className="relative w-12 h-12 rounded-full overflow-hidden flex-shrink-0 ring-2 ring-gray-200 hover:ring-blue-300 transition-all duration-200">
                      {player.avatar ? (
                        <img
                          src={player.avatar.startsWith('http') ? player.avatar : `../${player.avatar}`}
                          alt={player.display_name || player.user.name}
                          className="w-full h-full object-cover hover:scale-105 transition-transform duration-200"
                          onError={(e) => {
                            // Fallback to default icon if image fails to load
                            const target = e.target as HTMLImageElement;
                            target.style.display = 'none';
                            const parent = target.parentElement;
                            if (parent) {
                              parent.innerHTML = '<div class="w-full h-full bg-gray-100 flex items-center justify-center"><svg class="w-6 h-6 text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"></path></svg></div>';
                            }
                          }}
                        />
                      ) : (
                        <div className="w-full h-full bg-gradient-to-br from-gray-100 to-gray-200 flex items-center justify-center">
                          <User className="w-6 h-6 text-gray-600" />
                        </div>
                      )}
                      
                      {/* Approval Status Indicator */}
                      {player.approval_status && (
                        <div 
                          className="absolute -bottom-1 -right-1 w-4 h-4 rounded-full border-2 border-white flex items-center justify-center cursor-help"
                          title={`Document Status: ${player.approval_status.charAt(0).toUpperCase() + player.approval_status.slice(1)}`}
                        >
                          {player.approval_status === 'approved' && (
                            <div className="w-2.5 h-2.5 bg-green-500 rounded-full"></div>
                          )}
                          {player.approval_status === 'rejected' && (
                            <div className="w-2.5 h-2.5 bg-red-500 rounded-full"></div>
                          )}
                          {player.approval_status === 'pending' && (
                            <div className="w-2.5 h-2.5 bg-yellow-500 rounded-full"></div>
                          )}
                          {player.approval_status === 'under_review' && (
                            <div className="w-2.5 h-2.5 bg-blue-500 rounded-full"></div>
                          )}
                        </div>
                      )}
                    </div>
                    <div className="min-w-0 flex-1">
                      <h3 className="font-semibold text-gray-900 truncate">{player.user.name}</h3>
                      <p className="text-sm text-gray-500 truncate">{player.display_name}</p>
                      <p className="text-sm text-gray-500 truncate">{player.user.email}</p>
                    </div>
                  </div>
                  <div className="flex gap-1">
                    <Button
                      size="sm"
                      variant="outline"
                      onClick={() => router.visit(route('team.players.profile', player.id))}
                      title="View Profile"
                    >
                      <Eye className="w-3 h-3" />
                    </Button>
                    <Button
                      size="sm"
                      variant="outline"
                      onClick={() => router.visit(route('team.players.certificate.view', player.id))}
                      title="View Certificate"
                    >
                      <Award className="w-3 h-3" />
                    </Button>
                    <Button
                      size="sm"
                      variant="outline"
                      onClick={() => router.visit(route('team.players.profile', player.id))}
                      title="Edit Player"
                    >
                      <Edit className="w-3 h-3" />
                    </Button>
                    <Button
                      size="sm"
                      variant="outline"
                      onClick={() => handleRemovePlayer(player.id)}
                      className="text-red-600 hover:text-red-700"
                      title="Remove Player"
                    >
                      <Trash2 className="w-3 h-3" />
                    </Button>
                  </div>
                </div>
              </CardHeader>
              <CardContent className="pt-0">
                <div className="space-y-3">
                  {/* Basic Info */}
                  <div className="space-y-2">
                    <div className="flex items-center justify-between">
                      <span className="text-sm text-gray-600">Position</span>
                      <Badge className={getPositionColor(player.football_position || player.position)}>
                        {player.football_position || player.position}
                      </Badge>
                    </div>
                    
                    {player.jersey_number && (
                      <div className="flex items-center justify-between">
                        <span className="text-sm text-gray-600">Jersey #</span>
                        <span className="text-sm font-medium flex items-center gap-1">
                          <Shirt className="w-3 h-3" />
                          {player.jersey_number}
                        </span>
                      </div>
                    )}

                    <div className="flex items-center justify-between">
                      <span className="text-sm text-gray-600">Status</span>
                      <Badge className={getStatusColor(player.status)}>
                        {player.status}
                      </Badge>
                    </div>

                    {player.verification_status && (
                      <div className="flex items-center justify-between">
                        <span className="text-sm text-gray-600">Verification</span>
                        <Badge className={getStatusColor(player.verification_status)}>
                          {player.verification_status}
                        </Badge>
                      </div>
                    )}

                    {/* Approval Status */}
                    {player.approval_status && (
                      <div className="flex items-center justify-between">
                        <span className="text-sm text-gray-600">Approval</span>
                        <div className="flex items-center gap-2">
                          {getApprovalStatusIcon(player.approval_status)}
                          <Badge className={getApprovalStatusColor(player.approval_status)}>
                            {player.approval_status}
                          </Badge>
                        </div>
                      </div>
                    )}

                    {/* ID Information */}
                    {(player.id_number || player.birth_certificate_number) && (
                      <div className="pt-2 border-t">
                        <h4 className="text-sm font-medium text-gray-700 mb-2 flex items-center gap-1">
                          <FileText className="w-4 h-4" />
                          Identification
                        </h4>
                        <div className="space-y-2">
                          {player.id_number && (
                            <div className="flex items-center justify-between">
                              <span className="text-xs text-gray-600">ID Number</span>
                              <span className="text-xs font-mono bg-gray-100 px-2 py-1 rounded">
                                {player.id_number}
                              </span>
                            </div>
                          )}
                          {player.birth_certificate_number && (
                            <div className="flex items-center justify-between">
                              <span className="text-xs text-gray-600">Birth Cert #</span>
                              <span className="text-xs font-mono bg-gray-100 px-2 py-1 rounded">
                                {player.birth_certificate_number}
                              </span>
                            </div>
                          )}
                        </div>
                      </div>
                    )}

                    {/* Document Photos */}
                    {(player.passport_photo || player.birth_cert_photo) && (
                      <div className="pt-2 border-t">
                        <h4 className="text-sm font-medium text-gray-700 mb-2 flex items-center gap-1">
                          <Camera className="w-4 h-4" />
                          Documents
                        </h4>
                        <div className="grid grid-cols-2 gap-2">
                          {player.passport_photo && (
                            <div className="text-center">
                              <div className="w-16 h-16 mx-auto bg-gray-100 rounded-lg flex items-center justify-center mb-1">
                                <Camera className="w-6 h-6 text-gray-400" />
                              </div>
                              <div className="text-xs text-gray-600">Passport</div>
                            </div>
                          )}
                          {player.birth_cert_photo && (
                            <div className="text-center">
                              <div className="w-16 h-16 mx-auto bg-gray-100 rounded-lg flex items-center justify-center mb-1">
                                <FileText className="w-6 h-6 text-gray-400" />
                              </div>
                              <div className="text-xs text-gray-600">Birth Cert</div>
                            </div>
                          )}
                        </div>
                      </div>
                    )}
                  </div>

                  {/* Digital ID Section */}
                  {player.digital_id && (
                    <div className="pt-2 border-t">
                      <div className="flex items-center justify-between mb-2">
                        <span className="text-sm font-medium text-gray-700 flex items-center gap-1">
                          <IdCard className="w-4 h-4" />
                          Digital ID
                        </span>
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={() => handleCopyDigitalId(player.digital_id!)}
                          className="h-6 px-2 text-xs"
                        >
                          <Copy className="w-3 h-3" />
                        </Button>
                      </div>
                      <div className="bg-gray-50 p-2 rounded text-xs font-mono text-gray-600 break-all">
                        {player.digital_id}
                      </div>
                    </div>
                  )}

                  

                  {/* Achievements & Certificates */}
                  {/* <div className="pt-2 border-t">
                    <div className="flex items-center justify-between mb-2">
                      <span className="text-sm font-medium text-gray-700 flex items-center gap-1">
                        <Award className="w-4 h-4" />
                        Achievements
                      </span>
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => handleDownloadCertificate(player.id)}
                        className="h-6 px-2 text-xs"
                        title="Download Certificate of Participation"
                      >
                        <Download className="w-3 h-3" />
                      </Button>
                    </div>
                    <div className="space-y-1">
                      {player.achievements && player.achievements.length > 0 ? (
                        player.achievements.slice(0, 2).map((achievement, index) => (
                          <div key={index} className="flex items-center gap-2 text-xs">
                            <CheckCircle className="w-3 h-3 text-green-500" />
                            <span className="text-gray-600">{achievement.name}</span>
                          </div>
                        ))
                      ) : (
                        <div className="text-xs text-gray-500">No achievements yet</div>
                      )}
                      {player.achievements && player.achievements.length > 2 && (
                        <div className="text-xs text-gray-500">
                          +{player.achievements.length - 2} more achievements
                        </div>
                      )}
                    </div>
                  </div> */}

                  {/* Player Type & Experience */}
                  {(player.player_type || player.experience_level) && (
                    <div className="pt-2 border-t">
                      <div className="flex items-center gap-2 text-xs">
                        {player.player_type && (
                          <Badge variant="outline" className="text-xs">
                            {player.player_type}
                          </Badge>
                        )}
                        {player.experience_level && (
                          <Badge variant="outline" className="text-xs">
                            {player.experience_level}
                          </Badge>
                        )}
                      </div>
                    </div>
                  )}

                  {/* Contact Info */}
                  {player.user.phone && (
                    <div className="flex items-center justify-between text-xs">
                      <span className="text-gray-600">Phone</span>
                      <span className="font-medium flex items-center gap-1">
                        <Phone className="w-3 h-3" />
                        {player.user.phone}
                      </span>
                    </div>
                  )}

                  <div className="flex items-center justify-between text-xs text-gray-500">
                    <span>Joined</span>
                    <span>{new Date(player.created_at).toLocaleDateString()}</span>
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>

        {players.length === 0 && (
          <Card>
            <CardContent className="text-center py-12">
              <Users className="w-12 h-12 text-gray-300 mx-auto mb-4" />
              <h3 className="text-lg font-semibold text-gray-900 mb-2">No Players Yet</h3>
              <p className="text-gray-500 mb-4">Start building your team by creating players</p>
              <Button onClick={() => router.visit(route('team.players.create'))}>
                <UserPlus className="w-4 h-4 mr-2" />
                Create First Player
              </Button>
            </CardContent>
          </Card>
        )}
      </div>
    </AppLayout>
  );
}
