import React, { useState } from 'react';
import { Head, useForm, router } from '@inertiajs/react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { 
  User, 
  Mail, 
  Phone, 
  Calendar, 
  MapPin, 
  Shirt, 
  Trophy, 
  Target, 
  Globe, 
  Camera,
  Edit,
  Save,
  X,
  Upload,
  Award,
  BarChart3,
  Gamepad2,
  Users,
  Shield,
  Star,
  FileText,
  IdCard,
  CheckCircle2,
  XCircle,
  Clock
} from 'lucide-react';
import AppLayout from '@/layouts/app-layout';
import TeamSelector from '@/components/TeamManager/TeamSelector';
import { toast } from 'sonner';
import { useErrorHandler } from '@/hooks/use-error-handler';

interface Player {
  id: number;
  user: {
    id: number;
    name: string;
    email: string;
    phone?: string;
  };
  display_name?: string;
  bio?: string;
  avatar?: string;
  country?: string;
  city?: string;
  date_of_birth?: string;
  gender?: string;
  phone?: string;
  discord_username?: string;
  steam_id?: string;
  twitch_username?: string;
  youtube_channel?: string;
  twitter_handle?: string;
  instagram_handle?: string;
  website?: string;
  player_type?: string;
  experience_level?: string;
  primary_game?: string;
  game_preferences?: string[];
  football_position?: string;
  jersey_number?: number;
  is_team_captain?: boolean;
  global_rating?: number;
  global_rank?: number;
  tournament_rating?: number;
  tournament_rank?: number;
  status?: string;
  verification_status?: string;
  verified_at?: string;
  tournament_wins?: number;
  tournament_losses?: number;
  tournament_draws?: number;
  tournament_win_rate?: number;
  tournament_matches_played?: number;
  tournament_matches_won?: number;
  total_earnings?: number;
  tournaments_won?: number;
  achievements?: any[];
  is_public_profile?: boolean;
  allow_messages?: boolean;
  show_contact_info?: boolean;
  id_number?: string;
  birth_certificate_number?: string;
  passport_photo?: string;
  birth_cert_photo?: string;
  approval_status?: string;
  created_at: string;
}

interface Team {
  id: number;
  name: string;
}

interface PlayerProfileProps {
  team: Team;
  teams: Team[];
  player: Player;
}

const POSITIONS = [
  'Goalkeeper',
  'Defender',
  'Midfielder',
  'Forward',
  'Striker',
  'Winger',
  'Captain',
  'Vice-Captain'
];

const PLAYER_TYPES = [
  'Amateur',
  'Semi-Professional',
  'Professional',
  'Legend'
];

const EXPERIENCE_LEVELS = [
  'Beginner',
  'Intermediate',
  'Advanced',
  'Expert',
  'Master'
];

const GENDERS = [
  { value: 'male', label: 'Male' },
  { value: 'female', label: 'Female' },
  { value: 'other', label: 'Other' },
  { value: 'prefer_not_to_say', label: 'Prefer not to say' }
];

const GAMES = [
  'FIFA',
  'PES',
  'Football Manager',
  'Other'
];

export default function PlayerProfile({ team, teams, player }: PlayerProfileProps) {
  const [isEditing, setIsEditing] = useState(false);
  const [validationErrors, setValidationErrors] = useState<Record<string, string[]>>({});
  const { handleFormError } = useErrorHandler();
  const { data, setData, post, processing, errors, reset } = useForm({
    display_name: player.display_name || '',
    bio: player.bio || '',
    country: player.country || '',
    city: player.city || '',
    date_of_birth: player.date_of_birth || '',
    gender: player.gender || 'male',
    phone: player.phone || '',
    discord_username: player.discord_username || '',
    steam_id: player.steam_id || '',
    twitch_username: player.twitch_username || '',
    youtube_channel: player.youtube_channel || '',
    twitter_handle: player.twitter_handle || '',
    instagram_handle: player.instagram_handle || '',
    website: player.website || '',
    player_type: player.player_type || '',
    experience_level: player.experience_level || '',
    primary_game: player.primary_game || '',
    game_preferences: player.game_preferences || [],
    football_position: player.football_position || '',
    jersey_number: player.jersey_number || '',
    is_team_captain: player.is_team_captain || false,
    is_public_profile: player.is_public_profile || false,
    allow_messages: player.allow_messages || false,
    show_contact_info: player.show_contact_info || false,
    
    // Avatar field
    avatar: null as File | null,
    
    // Identification fields
    id_number: player.id_number || '',
    birth_certificate_number: player.birth_certificate_number || '',
    passport_photo: null as File | null,
    birth_cert_photo: null as File | null,
    approval_status: player.approval_status || 'pending',
  });

  const handleSave = async (e: React.FormEvent) => {
    e.preventDefault();
    
    // Clear previous validation errors
    setValidationErrors({});
    
    // Create FormData for file uploads
    const formData = new FormData();
    
    // Add all form data
    Object.keys(data).forEach(key => {
      if (key === 'avatar' || key === 'passport_photo' || key === 'birth_cert_photo') {
        // Handle file uploads - only append if it's a File object
        if (data[key as keyof typeof data] instanceof File) {
          formData.append(key, data[key as keyof typeof data] as File);
        }
      } else if (key === 'is_team_captain' || key === 'is_public_profile' || key === 'allow_messages' || key === 'show_contact_info') {
        // Handle boolean fields - always send true/false
        formData.append(key, data[key as keyof typeof data] ? '1' : '0');
      } else {
        // Handle regular form fields
        const value = data[key as keyof typeof data];
        if (value !== null && value !== undefined && value !== '') {
          formData.append(key, String(value));
        }
      }
    });

    // Debug: Log what's being sent
    console.log('Form data being sent:');
    for (let [key, value] of formData.entries()) {
      console.log(key, value);
    }

    try {
      const response = await fetch(route('team.players.update-profile', player.id), {
        method: 'POST',
        headers: {
          'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
          'X-Requested-With': 'XMLHttpRequest',
        },
        body: formData,
      });

      const result = await response.json();

      if (response.ok) {
        setIsEditing(false);
        setValidationErrors({});
        // Show success message with toast
        console.log('Success response:', result);
        toast.success(result.message || 'Player profile updated successfully!');
        // Redirect after 2 seconds
        setTimeout(() => {
          router.visit(route('team.players'));
        }, 2000);
      } else {
        // Show error messages with toast
        console.log('Error response:', result);
        
        if (result.error) {
          // Single error message
          toast.error(result.error, {
            duration: 5000,
          });
        } else if (result.errors) {
          // Store validation errors for display
          setValidationErrors(result.errors);
          
          // Handle validation errors - build formatted error list
          const errorEntries = Object.entries(result.errors);
          const errorCount = errorEntries.length;
          
          // Create error message string
          let errorMessage = '';
          errorEntries.slice(0, 3).forEach(([field, messages]) => {
            const fieldName = field.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
            const errorMsg = Array.isArray(messages) ? messages[0] : messages;
            errorMessage += `• ${errorMsg}\n`;
          });
          
          if (errorCount > 3) {
            errorMessage += `\n... and ${errorCount - 3} more error(s)`;
          }
          
          toast.error(`Validation Failed (${errorCount} error${errorCount > 1 ? 's' : ''})`, {
            description: errorMessage.trim(),
            duration: 8000,
          });
          
          // Scroll to first error
          const firstErrorField = Object.keys(result.errors)[0];
          const element = document.getElementById(firstErrorField);
          if (element) {
            element.scrollIntoView({ behavior: 'smooth', block: 'center' });
            element.focus();
          }
        } else if (result.message) {
          // Generic message from server
          toast.error(result.message, {
            duration: 5000,
          });
        } else {
          // Fallback error
          toast.error('Failed to update player profile. Please check the form for errors.', {
            duration: 5000,
          });
        }
      }
    } catch (error) {
      console.error('Error updating profile:', error);
      toast.error('An unexpected error occurred. Please try again.');
    }
  };

  const handleCancel = () => {
    reset();
    setIsEditing(false);
  };

  const getStatusColor = (status: string) => {
    switch (status?.toLowerCase()) {
      case 'active':
        return 'bg-green-100 text-green-800';
      case 'inactive':
        return 'bg-red-100 text-red-800';
      case 'suspended':
        return 'bg-yellow-100 text-yellow-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const getVerificationColor = (status: string) => {
    switch (status?.toLowerCase()) {
      case 'verified':
        return 'bg-green-100 text-green-800';
      case 'pending':
        return 'bg-yellow-100 text-yellow-800';
      case 'rejected':
        return 'bg-red-100 text-red-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const getPlayerTypeColor = (type: string) => {
    switch (type?.toLowerCase()) {
      case 'legend':
        return 'bg-purple-100 text-purple-800';
      case 'professional':
        return 'bg-blue-100 text-blue-800';
      case 'semi-professional':
        return 'bg-green-100 text-green-800';
      case 'amateur':
        return 'bg-gray-100 text-gray-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  return (
    <AppLayout>
      <Head title={`${player.user.name} - Player Profile`} />
      
      <div className="space-y-6 p-6">
        {/* Team Selector */}
        <TeamSelector 
          teams={teams} 
          selectedTeamId={team.id} 
          currentPath={`/team/players/${player.id}`} 
        />

        {/* Header */}
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-4">
            <div className="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center">
              {player.avatar ? (
                <img src={player.avatar} alt={player.user.name} className="w-16 h-16 rounded-full object-cover" />
              ) : (
                <User className="w-8 h-8 text-gray-600" />
              )}
            </div>
            <div>
              <h1 className="text-3xl font-bold text-gray-900">
                {player.user.name}
                <p className="text-sm text-gray-500">{player.display_name}</p>
              </h1>
              <p className="text-gray-600">{player.user.email}</p>
              <div className="flex items-center gap-2 mt-2">
                <Badge className={getStatusColor(player.status || '')}>
                  {player.status || 'Unknown'}
                </Badge>
                <Badge className={getVerificationColor(player.verification_status || '')}>
                  {player.verification_status || 'Unverified'}
                </Badge>
                {player.player_type && (
                  <Badge className={getPlayerTypeColor(player.player_type)}>
                    {player.player_type}
                  </Badge>
                )}
              </div>
            </div>
          </div>
          <div className="flex gap-2">
            <Button
              variant="outline"
              onClick={() => router.visit(route('team.players'))}
            >
              <X className="w-4 h-4 mr-2" />
              Back to Players
            </Button>
            <Button
              onClick={() => setIsEditing(!isEditing)}
              variant={isEditing ? "outline" : "default"}
            >
              {isEditing ? (
                <>
                  <X className="w-4 h-4 mr-2" />
                  Cancel Edit
                </>
              ) : (
                <>
                  <Edit className="w-4 h-4 mr-2" />
                  Edit Profile
                </>
              )}
            </Button>
          </div>
        </div>

        {/* Profile Content */}
        <div className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <User className="w-5 h-5" />
                  Personal Information
                </CardTitle>
                <CardDescription>
                  Basic personal details and team information
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                {isEditing ? (
                  <form onSubmit={handleSave} className="space-y-4" encType="multipart/form-data">
                    {/* Avatar Upload Section */}
                    <div className="flex items-center gap-6">
                      <div className="flex-shrink-0">
                        <div className="w-20 h-20 rounded-full overflow-hidden ring-2 ring-gray-200">
                          {data.avatar ? (
                            <img
                              src={URL.createObjectURL(data.avatar)}
                              alt="Preview"
                              className="w-full h-full object-cover"
                            />
                          ) : player.avatar ? (
                            <img
                              src={player.avatar.startsWith('http') ? player.avatar : player.avatar}
                              alt={player.display_name || player.user.name}
                              className="w-full h-full object-cover"
                            />
                          ) : (
                            <div className="w-full h-full bg-gray-100 flex items-center justify-center">
                              <User className="w-8 h-8 text-gray-600" />
                            </div>
                          )}
                        </div>
                      </div>
                      <div className="flex-1">
                        <Label htmlFor="avatar" className="text-sm font-medium text-gray-700">
                          Profile Photo
                        </Label>
                        <div className="mt-2">
                          <input
                            id="avatar"
                            name="avatar"
                            type="file"
                            accept="image/*"
                            className="block w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-sm file:font-semibold file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100"
                            onChange={(e) => {
                              const file = e.target.files?.[0];
                              if (file) {
                                setData('avatar', file);
                              }
                            }}
                          />
                          <p className="text-xs text-gray-500 mt-1">PNG, JPG, GIF up to 10MB</p>
                        </div>
                        {errors.avatar && (
                          <p className="text-red-500 text-sm mt-1">{errors.avatar}</p>
                        )}
                      </div>
                    </div>

                    <div className="grid gap-4 md:grid-cols-2">
                      <div>
                        <Label htmlFor="display_name">
                          Display Name <span className="text-red-500">*</span>
                        </Label>
                        <Input
                          id="display_name"
                          value={data.display_name}
                          onChange={(e) => setData('display_name', e.target.value)}
                          placeholder="Enter display name"
                          className={validationErrors.display_name ? 'border-red-500 focus:ring-red-500' : ''}
                          required
                        />
                        {validationErrors.display_name && (
                          <p className="text-red-500 text-sm mt-1">{validationErrors.display_name[0]}</p>
                        )}
                      </div>

                      <div>
                        <Label htmlFor="phone">
                          Phone Number <span className="text-red-500">*</span>
                        </Label>
                        <Input
                          id="phone"
                          value={data.phone}
                          onChange={(e) => setData('phone', e.target.value)}
                          placeholder="Enter phone number"
                          className={!data.phone || validationErrors.phone ? 'border-red-500 focus:ring-red-500' : ''}
                          required
                        />
                        {validationErrors.phone && (
                          <p className="text-red-500 text-sm mt-1">{validationErrors.phone[0]}</p>
                        )}
                      </div>

                      <div>
                        <Label htmlFor="date_of_birth">
                          Date of Birth <span className="text-red-500">*</span>
                        </Label>
                        <Input
                          id="date_of_birth"
                          type="date"
                          value={data.date_of_birth}
                          onChange={(e) => setData('date_of_birth', e.target.value)}
                          className={!data.date_of_birth || validationErrors.date_of_birth ? 'border-red-500 focus:ring-red-500' : ''}
                          required
                        />
                        {validationErrors.date_of_birth && (
                          <p className="text-red-500 text-sm mt-1">{validationErrors.date_of_birth[0]}</p>
                        )}
                      </div>

                      <div>
                        {/* Male default selected value */}
                        <Label htmlFor="gender">
                          Gender <span className="text-red-500">*</span>
                        </Label>
                        <Select
                          value={data.gender || 'male'}
                          onValueChange={(value) => setData('gender', value)}
                          required
                        >
                          <SelectTrigger className={!data.gender || validationErrors.gender ? 'border-red-500 focus:ring-red-500' : ''}>
                            <SelectValue placeholder="Select gender" />
                          </SelectTrigger>
                          <SelectContent>
                            {GENDERS.map((gender) => (
                              <SelectItem key={gender.value} value={gender.value}>
                                {gender.label}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                        {validationErrors.gender && (
                          <p className="text-red-500 text-sm mt-1">{validationErrors.gender[0]}</p>
                        )}
                      </div>

                      <div>
                        <Label htmlFor="country">Country</Label>
                        <Input
                          id="country"
                          value={data.country}
                          onChange={(e) => setData('country', e.target.value)}
                          placeholder="Enter country"
                        />
                        {errors.country && (
                          <p className="text-red-500 text-sm mt-1">{errors.country}</p>
                        )}
                      </div>

                      <div>
                        <Label htmlFor="city">City</Label>
                        <Input
                          id="city"
                          value={data.city}
                          onChange={(e) => setData('city', e.target.value)}
                          placeholder="Enter city"
                        />
                        {errors.city && (
                          <p className="text-red-500 text-sm mt-1">{errors.city}</p>
                        )}
                      </div>
                    </div>

                    <div>
                      <Label htmlFor="bio">Bio</Label>
                      <textarea
                        id="bio"
                        className="w-full p-3 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                        rows={4}
                        value={data.bio}
                        onChange={(e) => setData('bio', e.target.value)}
                        placeholder="Tell us about yourself..."
                      />
                      {errors.bio && (
                        <p className="text-red-500 text-sm mt-1">{errors.bio}</p>
                      )}
                    </div>

                    {/* Identification Section */}
                    <div className="space-y-4">
                      <h4 className="text-lg font-medium text-gray-900 border-b pb-2">Identification Documents</h4>
                      
                    <div className="bg-red-50 border border-red-200 rounded-lg p-4 mb-4">
                      <p className="text-sm text-red-800 mb-2">
                        <strong className="text-red-900">Required for Approval:</strong> ID Number, Passport Photo, and Birth Certificate/ID Photo are mandatory for team approval.
                      </p>
                    </div>

                    <div className="grid gap-4 md:grid-cols-2">
                      <div>
                        <Label htmlFor="id_number">
                          ID Number / National ID <span className="text-red-500">*</span>
                        </Label>
                        <Input
                          id="id_number"
                          value={data.id_number}
                          onChange={(e) => setData('id_number', e.target.value)}
                          placeholder="Enter ID number"
                          className={!data.id_number || validationErrors.id_number ? 'border-red-500 focus:ring-red-500' : ''}
                          required
                        />
                        {validationErrors.id_number && (
                          <p className="text-red-500 text-sm mt-1">{validationErrors.id_number[0]}</p>
                        )}
                      </div>

                      <div>
                        <Label htmlFor="birth_certificate_number">
                          Birth Certificate Number {!data.id_number && <span className="text-red-500">*</span>}
                        </Label>
                        <Input
                          id="birth_certificate_number"
                          value={data.birth_certificate_number}
                          onChange={(e) => setData('birth_certificate_number', e.target.value)}
                          placeholder="Enter birth certificate number"
                          className={validationErrors.birth_certificate_number ? 'border-red-500 focus:ring-red-500' : ''}
                        />
                        {validationErrors.birth_certificate_number && (
                          <p className="text-red-500 text-sm mt-1">{validationErrors.birth_certificate_number[0]}</p>
                        )}
                      </div>
                    </div>

                      <div className="grid gap-6 md:grid-cols-2">
                      <div>
                          <Label htmlFor="passport_photo" className="text-sm font-medium text-gray-700">
                            Passport Photo (National ID Front) <span className="text-red-500">*</span>
                          </Label>
                          <div className="mt-2">
                            {data.passport_photo ? (
                              <div className="relative">
                                <img
                                  src={URL.createObjectURL(data.passport_photo)}
                                  alt="Passport preview"
                                  className="w-full h-48 object-cover rounded-lg border-2 border-gray-200"
                                />
                                <button
                                  type="button"
                                  onClick={() => setData('passport_photo', null)}
                                  className="absolute top-2 right-2 bg-red-500 text-white rounded-full p-1 hover:bg-red-600"
                                >
                                  <X className="w-4 h-4" />
                                </button>
                              </div>
                            ) : (
                              <div className="flex justify-center px-6 pt-5 pb-6 border-2 border-gray-300 border-dashed rounded-lg hover:border-gray-400 transition-colors">
                          <div className="space-y-1 text-center">
                            <Camera className="mx-auto h-12 w-12 text-gray-400" />
                            <div className="flex text-sm text-gray-600">
                              <label
                                htmlFor="passport_photo"
                                className="relative cursor-pointer bg-white rounded-md font-medium text-blue-600 hover:text-blue-500 focus-within:outline-none focus-within:ring-2 focus-within:ring-offset-2 focus-within:ring-blue-500"
                              >
                                <span>Upload passport photo</span>
                                <input
                                  id="passport_photo"
                                  name="passport_photo"
                                  type="file"
                                  accept="image/*"
                                  className="sr-only"
                                  onChange={(e) => {
                                    const file = e.target.files?.[0];
                                    if (file) {
                                      setData('passport_photo', file);
                                    }
                                  }}
                                />
                              </label>
                            </div>
                            <p className="text-xs text-gray-500">PNG, JPG, GIF up to 10MB</p>
                          </div>
                              </div>
                            )}
                        </div>
                        {validationErrors.passport_photo && (
                          <p className="text-red-500 text-sm mt-1">{validationErrors.passport_photo[0]}</p>
                        )}
                      </div>

                      <div>
                          <Label htmlFor="birth_cert_photo" className="text-sm font-medium text-gray-700">
                            Birth Certificate / ID Photo (National ID Back) <span className="text-red-500">*</span>
                          </Label>
                          <div className="mt-2">
                            {data.birth_cert_photo ? (
                              <div className="relative">
                                {data.birth_cert_photo.type === 'application/pdf' ? (
                                  <div className="w-full h-48 bg-gray-100 rounded-lg border-2 border-gray-200 flex items-center justify-center">
                                    <div className="text-center">
                                      <FileText className="mx-auto h-12 w-12 text-gray-400 mb-2" />
                                      <p className="text-sm text-gray-600">{data.birth_cert_photo.name}</p>
                                      <p className="text-xs text-gray-500">PDF Document</p>
                                    </div>
                                  </div>
                                ) : (
                                  <img
                                    src={URL.createObjectURL(data.birth_cert_photo)}
                                    alt="Birth certificate preview"
                                    className="w-full h-48 object-cover rounded-lg border-2 border-gray-200"
                                  />
                                )}
                                <button
                                  type="button"
                                  onClick={() => setData('birth_cert_photo', null)}
                                  className="absolute top-2 right-2 bg-red-500 text-white rounded-full p-1 hover:bg-red-600"
                                >
                                  <X className="w-4 h-4" />
                                </button>
                              </div>
                            ) : (
                              <div className="flex justify-center px-6 pt-5 pb-6 border-2 border-gray-300 border-dashed rounded-lg hover:border-gray-400 transition-colors">
                          <div className="space-y-1 text-center">
                            <FileText className="mx-auto h-12 w-12 text-gray-400" />
                            <div className="flex text-sm text-gray-600">
                              <label
                                htmlFor="birth_cert_photo"
                                className="relative cursor-pointer bg-white rounded-md font-medium text-blue-600 hover:text-blue-500 focus-within:outline-none focus-within:ring-2 focus-within:ring-offset-2 focus-within:ring-blue-500"
                              >
                                <span>Upload birth cert/ID photo</span>
                                <input
                                  id="birth_cert_photo"
                                  name="birth_cert_photo"
                                  type="file"
                                        accept="image/*,.pdf"
                                  className="sr-only"
                                  onChange={(e) => {
                                    const file = e.target.files?.[0];
                                    if (file) {
                                      setData('birth_cert_photo', file);
                                    }
                                  }}
                                />
                              </label>
                            </div>
                                  <p className="text-xs text-gray-500">PNG, JPG, GIF, PDF up to 10MB</p>
                          </div>
                              </div>
                            )}
                        </div>
                        {validationErrors.birth_cert_photo && (
                          <p className="text-red-500 text-sm mt-1">{validationErrors.birth_cert_photo[0]}</p>
                        )}
                        </div>
                      </div>
                    </div>
                    
                  </form>
                ) : (
                  <div className="space-y-4">
                    {/* Approval Status */}
                    {player.approval_status && (
                      <div className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                        <div className="flex items-center gap-3">
                          {player.approval_status === 'approved' && <CheckCircle2 className="w-5 h-5 text-green-600" />}
                          {player.approval_status === 'rejected' && <XCircle className="w-5 h-5 text-red-600" />}
                          {player.approval_status === 'pending' && <Clock className="w-5 h-5 text-yellow-600" />}
                          <div>
                            <p className="font-medium">Approval Status</p>
                            <p className="text-sm text-gray-600">Document verification status</p>
                          </div>
                        </div>
                        <Badge className={
                          player.approval_status === 'approved' ? 'bg-green-100 text-green-800' :
                          player.approval_status === 'rejected' ? 'bg-red-100 text-red-800' :
                          'bg-yellow-100 text-yellow-800'
                        }>
                          {player.approval_status}
                        </Badge>
                      </div>
                    )}

                    <div className="grid gap-4 md:grid-cols-2">
                      <div className="space-y-2">
                        <Label className="text-sm font-medium text-gray-500">User Name</Label>
                        <p className="text-sm">{player.user.name}</p>
                      </div>
                      <div className="space-y-2">
                        <Label className="text-sm font-medium text-gray-500">
                          Display Name <span className="text-red-500">*</span>
                        </Label>
                        <p className={`text-sm ${!player.display_name ? 'text-red-600 font-semibold bg-red-50 p-2 rounded' : ''}`}>
                          {player.display_name || '⚠️ Not set - Required for approval'}
                        </p>
                      </div>
                      <div className="space-y-2">
                        <Label className="text-sm font-medium text-gray-500">
                          Phone <span className="text-red-500">*</span>
                        </Label>
                        <p className={`text-sm flex items-center gap-1 ${!player.phone ? 'text-red-600 font-semibold bg-red-50 p-2 rounded' : ''}`}>
                          <Phone className="w-4 h-4" />
                          {player.phone || '⚠️ Not provided - Required for approval'}
                        </p>
                      </div>
                      <div className="space-y-2">
                        <Label className="text-sm font-medium text-gray-500">
                          Date of Birth <span className="text-red-500">*</span>
                        </Label>
                        <p className={`text-sm flex items-center gap-1 ${!player.date_of_birth ? 'text-red-600 font-semibold bg-red-50 p-2 rounded' : ''}`}>
                          <Calendar className="w-4 h-4" />
                          {player.date_of_birth ? new Date(player.date_of_birth).toLocaleDateString() : '⚠️ Not provided - Required for approval'}
                        </p>
                      </div>
                      <div className="space-y-2">
                        <Label className="text-sm font-medium text-gray-500">
                          Gender <span className="text-red-500">*</span>
                        </Label>
                        <p className={`text-sm ${!player.gender ? 'text-red-600 font-semibold bg-red-50 p-2 rounded' : ''}`}>
                          {player.gender || '⚠️ Not specified - Required for approval'}
                        </p>
                      </div>
                      <div className="space-y-2">
                        <Label className="text-sm font-medium text-gray-500">Location</Label>
                        <p className="text-sm flex items-center gap-1">
                          <MapPin className="w-4 h-4" />
                          {[player.city, player.country].filter(Boolean).join(', ') || 'Not specified'}
                        </p>
                      </div>
                      <div className="space-y-2">
                        <Label className="text-sm font-medium text-gray-500">Bio</Label>
                        <p className="text-sm">{player.bio || 'No bio provided'}</p>
                      </div>
                    </div>

                    <div className="grid gap-4 md:grid-cols-2">
                      <div className="space-y-2">
                        <Label className="text-sm font-medium text-gray-500">
                          ID Number <span className="text-red-500">*</span>
                        </Label>
                        <p className={`text-sm font-mono p-2 rounded ${!player.id_number ? 'bg-red-50 text-red-600 font-semibold' : 'bg-gray-100'}`}>
                          {player.id_number || '⚠️ Not provided - Required for approval'}
                        </p>
                      </div>
                      <div className="space-y-2">
                        <Label className="text-sm font-medium text-gray-500">Birth Certificate Number</Label>
                        <p className="text-sm font-mono bg-gray-100 p-2 rounded">
                          {player.birth_certificate_number || 'Not provided'}
                        </p>
                      </div>
                    </div>

                    <div className="grid gap-6 md:grid-cols-2">
                      <div className="space-y-2">
                        <Label className="text-sm font-medium text-gray-500">
                          Passport Photo (National ID Front) <span className="text-red-500">*</span>
                        </Label>
                          {player.passport_photo ? (
                          <div className="space-y-2">
                            <div className="flex items-center gap-2 text-green-600">
                              <Camera className="w-4 h-4" />
                              <span className="text-sm">Uploaded</span>
                            </div>
                            <div className="w-full h-48 rounded-lg overflow-hidden border-2 border-green-200">
                              <img
                                src={player.passport_photo}
                                alt="Passport photo"
                                className="w-full h-full object-cover"
                              />
                            </div>
                            </div>
                          ) : (
                            <div className="flex flex-col gap-2 p-4 bg-red-50 border-2 border-red-300 rounded-lg">
                              <div className="flex items-center gap-2 text-red-600">
                                <Camera className="w-5 h-5" />
                                <span className="text-sm font-semibold">⚠️ Not uploaded - Required for approval</span>
                              </div>
                            </div>
                          )}
                      </div>
                      <div className="space-y-2">
                        <Label className="text-sm font-medium text-gray-500">
                          Birth Cert/ID Photo (National ID Back) <span className="text-red-500">*</span>
                        </Label>
                          {player.birth_cert_photo ? (
                          <div className="space-y-2">
                            <div className="flex items-center gap-2 text-green-600">
                              <FileText className="w-4 h-4" />
                              <span className="text-sm">Uploaded</span>
                            </div>
                            {player.birth_cert_photo.toLowerCase().endsWith('.pdf') ? (
                              <div className="w-full h-48 bg-gray-100 rounded-lg border-2 border-gray-200 flex items-center justify-center">
                                <div className="text-center">
                                  <FileText className="mx-auto h-12 w-12 text-gray-400 mb-2" />
                                  <p className="text-sm text-gray-600">PDF Document</p>
                                  <a 
                                    href={player.birth_cert_photo} 
                                    target="_blank" 
                                    rel="noopener noreferrer"
                                    className="text-blue-600 hover:text-blue-800 text-sm underline"
                                  >
                                    View PDF
                                  </a>
                                </div>
                              </div>
                            ) : (
                              <div className="w-full h-48 rounded-lg overflow-hidden border-2 border-gray-200">
                                <img
                                  src={player.birth_cert_photo}
                                  alt="Birth certificate photo"
                                  className="w-full h-full object-cover"
                                />
                              </div>
                            )}
                            </div>
                          ) : (
                            <div className="flex flex-col gap-2 p-4 bg-red-50 border-2 border-red-300 rounded-lg">
                              <div className="flex items-center gap-2 text-red-600">
                                <FileText className="w-5 h-5" />
                                <span className="text-sm font-semibold">⚠️ Not uploaded - Required for approval</span>
                              </div>
                            </div>
                          )}
                      </div>
                    </div>
                  </div>
                )}
              
              <CardHeader>
                <CardTitle>
                  <Gamepad2 className="w-5 h-5" />
                  Gaming Profile
                </CardTitle>
                <CardDescription>
                  Gaming preferences and experience level
                </CardDescription>
              </CardHeader>
                {isEditing ? (
                  <div className="space-y-4">
                    <div className="grid gap-4 md:grid-cols-2">
                      <div>
                        <Label htmlFor="player_type">Player Type</Label>
                        <Select
                          value={data.player_type}
                          onValueChange={(value) => setData('player_type', value)}
                        >
                          <SelectTrigger>
                            <SelectValue placeholder="Select player type" />
                          </SelectTrigger>
                          <SelectContent>
                            {PLAYER_TYPES.map((type) => (
                              <SelectItem key={type} value={type}>
                                {type}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                        {errors.player_type && (
                          <p className="text-red-500 text-sm mt-1">{errors.player_type}</p>
                        )}
                      </div>

                      <div>
                        <Label htmlFor="experience_level">Experience Level</Label>
                        <Select
                          value={data.experience_level}
                          onValueChange={(value) => setData('experience_level', value)}
                        >
                          <SelectTrigger>
                            <SelectValue placeholder="Select experience level" />
                          </SelectTrigger>
                          <SelectContent>
                            {EXPERIENCE_LEVELS.map((level) => (
                              <SelectItem key={level} value={level}>
                                {level}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                        {errors.experience_level && (
                          <p className="text-red-500 text-sm mt-1">{errors.experience_level}</p>
                        )}
                      </div>

                      <div>
                        <Label htmlFor="primary_game">Primary Game</Label>
                        <Select
                          value={data.primary_game}
                          onValueChange={(value) => setData('primary_game', value)}
                        >
                          <SelectTrigger>
                            <SelectValue placeholder="Select primary game" />
                          </SelectTrigger>
                          <SelectContent>
                            {GAMES.map((game) => (
                              <SelectItem key={game} value={game}>
                                {game}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                        {errors.primary_game && (
                          <p className="text-red-500 text-sm mt-1">{errors.primary_game}</p>
                        )}
                      </div>

                      <div>
                        <Label htmlFor="football_position">
                          Football Position <span className="text-red-500">*</span>
                        </Label>
                        <Select
                          value={data.football_position}
                          onValueChange={(value) => setData('football_position', value)}
                          required
                        >
                          <SelectTrigger>
                            <SelectValue placeholder="Select position" />
                          </SelectTrigger>
                          <SelectContent>
                            {POSITIONS.map((position) => (
                              <SelectItem key={position} value={position}>
                                {position}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                        {errors.football_position && (
                          <p className="text-red-500 text-sm mt-1">{errors.football_position}</p>
                        )}
                      </div>

                      <div>
                        <Label htmlFor="jersey_number">
                          Jersey Number
                        </Label>
                        <Input
                          id="jersey_number"
                          type="number"
                          min="1"
                          max="99"
                          value={data.jersey_number}
                          onChange={(e) => setData('jersey_number', e.target.value)}
                          placeholder="e.g., 10"
                          className={validationErrors.jersey_number ? 'border-red-500 focus:ring-red-500' : ''}
                        />
                        {validationErrors.jersey_number && (
                          <p className="text-red-500 text-sm mt-1">{validationErrors.jersey_number[0]}</p>
                        )}
                      </div>
                    </div>

                    <div className="flex items-center space-x-2">
                      <input
                        type="checkbox"
                        id="is_team_captain"
                        checked={data.is_team_captain}
                        onChange={(e) => setData('is_team_captain', e.target.checked)}
                        className="rounded border-gray-300"
                      />
                      <Label htmlFor="is_team_captain">Team Captain</Label>
                    </div>

                    <div className="flex justify-end gap-2">
                      <Button type="button" variant="outline" onClick={handleCancel}>
                        Cancel
                      </Button>
                      <Button type="button" onClick={handleSave} disabled={processing}>
                        {processing ? 'Saving...' : 'Save Changes'}
                      </Button>
                    </div>
                  </div>
                ) : (
                  <div className="grid gap-4 md:grid-cols-2">
                    <div className="space-y-2">
                      <Label className="text-sm font-medium text-gray-500">Player Type</Label>
                      <div className="text-sm">
                        {player.player_type ? (
                          <Badge className={getPlayerTypeColor(player.player_type)}>
                            {player.player_type}
                          </Badge>
                        ) : <span>Not set</span>}
                      </div>
                    </div>
                    <div className="space-y-2">
                      <Label className="text-sm font-medium text-gray-500">Experience Level</Label>
                      <p className="text-sm">{player.experience_level || 'Not specified'}</p>
                    </div>
                    <div className="space-y-2">
                      <Label className="text-sm font-medium text-gray-500">Primary Game</Label>
                      <p className="text-sm">{player.primary_game || 'Not specified'}</p>
                    </div>
                    <div className="space-y-2">
                      <Label className="text-sm font-medium text-gray-500">
                        Position <span className="text-red-500">*</span>
                      </Label>
                      <p className={`text-sm flex items-center gap-1 ${!player.football_position ? 'text-red-600 font-semibold bg-red-50 p-2 rounded' : ''}`}>
                        <Shirt className="w-4 h-4" />
                        {player.football_position || '⚠️ Not assigned - Required for approval'}
                      </p>
                    </div>
                    <div className="space-y-2">
                      <Label className="text-sm font-medium text-gray-500">Jersey Number</Label>
                      <p className="text-sm">
                        {player.jersey_number ? `#${player.jersey_number}` : 'Not assigned'}
                      </p>
                    </div>
                    <div className="space-y-2">
                      <Label className="text-sm font-medium text-gray-500">Team Captain</Label>
                      <div className="text-sm">
                        {player.is_team_captain ? (
                          <Badge className="bg-purple-100 text-purple-800">Yes</Badge>
                        ) : (
                          <Badge className="bg-gray-100 text-gray-800">No</Badge>
                        )}
                      </div>
                    </div>
                  </div>
                )}
              </CardContent>
            </Card>
                      </div>
      </div>
    </AppLayout>
  );
}
