import React, { useState, useEffect } from 'react';
import { Head, Link, usePage } from '@inertiajs/react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Checkbox } from '@/components/ui/checkbox';
import { Download, Printer, ArrowLeft, Eye, Users } from 'lucide-react';
import { toast } from 'sonner';
import AppLayout from '@/layouts/app-layout';

interface Player {
    id: number;
    display_name?: string;
    user: {
        name: string;
        email: string;
        phone?: string;
    };
    football_position?: string;
    jersey_number?: number;
    verification_status?: string;
    digital_id?: string;
    avatar?: string;
    player_type?: string;
    phone?: string;
}

interface Team {
    id: number;
    name: string;
}

interface Props {
    players: Player[];
    team: Team;
    teams: Team[];
}

export default function PlayerIdCards({ players, team, teams }: Props) {
    const [isLoading, setIsLoading] = useState(false);
    const [selectedPlayers, setSelectedPlayers] = useState<number[]>([]);
    const [idCardsUrl, setIdCardsUrl] = useState<string | null>(null);

    // Select all players by default
    useEffect(() => {
        setSelectedPlayers(players.map(player => player.id));
    }, [players]);

    const handleSelectPlayer = (playerId: number, checked: boolean) => {
        if (checked) {
            setSelectedPlayers(prev => [...prev, playerId]);
        } else {
            setSelectedPlayers(prev => prev.filter(id => id !== playerId));
        }
    };

    const handleSelectAll = (checked: boolean) => {
        if (checked) {
            setSelectedPlayers(players.map(player => player.id));
        } else {
            setSelectedPlayers([]);
        }
    };

    const handleViewIdCards = async () => {
        if (selectedPlayers.length === 0) {
            toast.error('Please select at least one player');
            return;
        }

        setIsLoading(true);
        try {
            const response = await fetch(`/team/players/id-cards`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
                },
                body: JSON.stringify({
                    player_ids: selectedPlayers,
                    team_id: team.id
                })
            });
            
            if (response.ok) {
                // Get the HTML content directly
                const htmlContent = await response.text();
                const blob = new Blob([htmlContent], { type: 'text/html' });
                const url = window.URL.createObjectURL(blob);
                setIdCardsUrl(url);
                // Open ID cards in new tab
                window.open(url, '_blank');
            } else {
                toast.error('Failed to load ID cards');
            }
        } catch (error) {
            toast.error('Error loading ID cards');
        } finally {
            setIsLoading(false);
        }
    };

    const handleDownloadIdCards = async () => {
        if (selectedPlayers.length === 0) {
            toast.error('Please select at least one player');
            return;
        }

        setIsLoading(true);
        try {
            const response = await fetch(`/team/players/id-cards/download`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
                },
                body: JSON.stringify({
                    player_ids: selectedPlayers,
                    team_id: team.id
                })
            });
            
            if (response.ok) {
                // Get the HTML content directly
                const htmlContent = await response.text();
                const blob = new Blob([htmlContent], { type: 'text/html' });
                const url = window.URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = `id-cards-${team.name.replace(/\s+/g, '-').toLowerCase()}.html`;
                document.body.appendChild(a);
                a.click();
                window.URL.revokeObjectURL(url);
                document.body.removeChild(a);
                toast.success('ID cards downloaded successfully');
            } else {
                toast.error('Failed to download ID cards');
            }
        } catch (error) {
            toast.error('Error downloading ID cards');
        } finally {
            setIsLoading(false);
        }
    };

    const handlePrintIdCards = () => {
        if (idCardsUrl) {
            const printWindow = window.open(idCardsUrl, '_blank');
            if (printWindow) {
                printWindow.onload = () => {
                    printWindow.print();
                };
            }
        } else {
            toast.error('Please view ID cards first');
        }
    };

    const selectedPlayersData = players.filter(player => selectedPlayers.includes(player.id));

    return (
        <AppLayout breadcrumbs={[{ title: 'Player ID Cards', href: '/team/players/id-cards' }]}>
            <Head title={`ID Cards - ${team.name}`} />
            
            <div className="">
                <div className="">
                    {/* Header */}
                    <div className="mb-8">
                        <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-4">
                                <Link
                                    href="/team/players"
                                    className="inline-flex items-center px-4 py-2 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50"
                                >
                                    <ArrowLeft className="h-4 w-4 mr-2" />
                                    Back to Players
                                </Link>
                                <div>
                                    <h1 className="text-3xl font-bold text-gray-900">
                                        Player ID Cards
                                    </h1>
                                    <p className="text-gray-600">
                                        Generate and manage ID cards for {team.name} players
                                    </p>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Team Info Card */}
                    <Card className="mb-8">
                        <CardHeader>
                            <CardTitle className="flex items-center">
                                <Users className="h-5 w-5 mr-2" />
                                Team Information
                            </CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                                <div>
                                    <span className="text-sm font-medium text-gray-500">Team Name:</span>
                                    <p className="text-gray-900 text-lg font-medium">{team.name}</p>
                                </div>
                                <div>
                                    <span className="text-sm font-medium text-gray-500">Total Players:</span>
                                    <p className="text-gray-900 text-lg font-medium">{players.length}</p>
                                </div>
                                <div>
                                    <span className="text-sm font-medium text-gray-500">Selected Players:</span>
                                    <p className="text-gray-900 text-lg font-medium">{selectedPlayers.length}</p>
                                </div>
                            </div>
                        </CardContent>
                    </Card>

                    {/* Player Selection */}
                    <Card className="mb-8">
                        <CardHeader>
                            <CardTitle className="flex items-center justify-between">
                                <span>Select Players for ID Cards</span>
                                <div className="flex items-center space-x-2">
                                    <Checkbox
                                        id="select-all"
                                        checked={selectedPlayers.length === players.length}
                                        onCheckedChange={handleSelectAll}
                                    />
                                    <label htmlFor="select-all" className="text-sm font-medium">
                                        Select All
                                    </label>
                                </div>
                            </CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                                {players.map((player) => {
                                    const playerName = player.user.name;
                                    const isSelected = selectedPlayers.includes(player.id);
                                    
                                    return (
                                        <div
                                            key={player.id}
                                            className={`border rounded-lg p-4 cursor-pointer transition-colors ${
                                                isSelected 
                                                    ? 'border-blue-500 bg-blue-50' 
                                                    : 'border-gray-200 hover:border-gray-300'
                                            }`}
                                            onClick={() => handleSelectPlayer(player.id, !isSelected)}
                                        >
                                            <div className="flex items-center space-x-3">
                                                <Checkbox
                                                    checked={isSelected}
                                                    onChange={() => {}}
                                                />
                                                <div className="flex-shrink-0">
                                                    {player.avatar ? (
                                                        <img
                                                            className="h-12 w-12 rounded-full object-cover"
                                                            src={player.avatar}
                                                            alt={player.user.name}
                                                        />
                                                    ) : (
                                                        <div className="h-12 w-12 rounded-full bg-gray-300 flex items-center justify-center">
                                                            <span className="text-gray-600 font-medium">
                                                                {player.user.name.charAt(0).toUpperCase()}
                                                            </span>
                                                        </div>
                                                    )}
                                                </div>
                                                <div className="flex-1 min-w-0">
                                                    <h3 className="text-sm font-medium text-gray-900 truncate">
                                                        {player.user.name}
                                                    </h3>
                                                    <p className="text-sm text-gray-600 truncate">
                                                        {player.football_position || 'No position'}
                                                    </p>
                                                    <div className="flex items-center space-x-2 mt-1">
                                                        <Badge 
                                                            variant={player.verification_status === 'verified' ? 'default' : 'secondary'}
                                                            className="text-xs"
                                                        >
                                                            {player.verification_status || 'Pending'}
                                                        </Badge>
                                                        {player.jersey_number && (
                                                            <Badge variant="outline" className="text-xs">
                                                                #{player.jersey_number}
                                                            </Badge>
                                                        )}
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    );
                                })}
                            </div>
                        </CardContent>
                    </Card>

                    {/* ID Card Actions */}
                    <Card className="mb-8">
                        <CardHeader>
                            <CardTitle>ID Card Actions</CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                                <Button
                                    onClick={handleViewIdCards}
                                    disabled={isLoading || selectedPlayers.length === 0}
                                    className="w-full"
                                    variant="outline"
                                >
                                    <Eye className="h-4 w-4 mr-2" />
                                    {isLoading ? 'Loading...' : 'View ID Cards'}
                                </Button>
                                
                                <Button
                                    onClick={handleDownloadIdCards}
                                    disabled={isLoading || selectedPlayers.length === 0}
                                    className="w-full"
                                    variant="outline"
                                >
                                    <Download className="h-4 w-4 mr-2" />
                                    {isLoading ? 'Downloading...' : 'Download ID Cards'}
                                </Button>
                                
                                <Button
                                    onClick={handlePrintIdCards}
                                    disabled={!idCardsUrl || isLoading}
                                    className="w-full"
                                    variant="outline"
                                >
                                    <Printer className="h-4 w-4 mr-2" />
                                    Print ID Cards
                                </Button>
                            </div>
                            
                            <div className="mt-6 p-4 bg-green-50 rounded-lg">
                                <h4 className="text-sm font-medium text-green-900 mb-2">ID Card Information</h4>
                                <ul className="text-sm text-green-800 space-y-1">
                                    <li>• ID cards are formatted for A4 portrait printing</li>
                                    <li>• Up to 3 ID cards per page (front and back)</li>
                                    <li>• Each card includes player photo, details, and QR code</li>
                                    <li>• Cards can be printed or downloaded as HTML files</li>
                                    <li>• Digital verification is available through QR codes</li>
                                </ul>
                            </div>
                        </CardContent>
                    </Card>

                    {/* Selected Players Summary */}
                    {selectedPlayersData.length > 0 && (
                        <Card>
                            <CardHeader>
                                <CardTitle>Selected Players Summary</CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                                    {selectedPlayersData.map((player) => {
                                        const playerName = player.user.name;
                                        return (
                                            <div key={player.id} className="flex items-center space-x-3 p-3 border rounded-lg">
                                                <div className="flex-shrink-0">
                                                    {player.avatar ? (
                                                        <img
                                                            className="h-10 w-10 rounded-full object-cover"
                                                            src={player.avatar}
                                                            alt={player.user.name}
                                                        />
                                                    ) : (
                                                        <div className="h-10 w-10 rounded-full bg-gray-300 flex items-center justify-center">
                                                            <span className="text-gray-600 font-medium text-sm">
                                                                {player.user.name.charAt(0).toUpperCase()}
                                                            </span>
                                                        </div>
                                                    )}
                                                </div>
                                                <div className="flex-1 min-w-0">
                                                    <h4 className="text-sm font-medium text-gray-900 truncate">
                                                        {player.user.name}
                                                    </h4>
                                                    <p className="text-xs text-gray-600">
                                                        {player.football_position} • {player.user.email}
                                                    </p>
                                                </div>
                                            </div>
                                        );
                                    })}
                                </div>
                            </CardContent>
                        </Card>
                    )}
                </div>
            </div>
        </AppLayout>
    );
}
