import React, { useState, useEffect } from 'react';
import { Head, Link, usePage } from '@inertiajs/react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Download, Printer, ArrowLeft, Eye } from 'lucide-react';
import { toast } from 'sonner';
import AppLayout from '@/layouts/app-layout';

interface Player {
    id: number;
    display_name?: string;
    user: {
        name: string;
        email: string;
    };
    football_position?: string;
    jersey_number?: number;
    verification_status?: string;
    digital_id?: string;
    avatar?: string;
    tournament_matches_played?: number;
    tournament_matches_won?: number;
    tournament_win_rate?: number;
    global_rating?: number;
    tournament_rating?: number;
    total_earnings?: number;
    tournaments_won?: number;
    tournament_current_streak?: number;
    tournament_longest_streak?: number;
}

interface Team {
    id: number;
    name: string;
}

interface Props {
    player: Player;
    team: Team;
    teams: Team[];
}

export default function PlayerCertificate({ player, team, teams }: Props) {
    const [isLoading, setIsLoading] = useState(false);
    const [certificateUrl, setCertificateUrl] = useState<string | null>(null);

    const playerName = player.user.name;

    const handleViewCertificate = async () => {
        setIsLoading(true);
        try {
            const response = await fetch(`/team/players/${player.id}/certificate/pdf`);
            if (response.ok) {
                // Get the PDF blob
                const blob = await response.blob();
                const url = window.URL.createObjectURL(blob);
                setCertificateUrl(url);
                // Open PDF in new tab
                window.open(url, '_blank');
            } else {
                toast.error('Failed to load certificate');
            }
        } catch (error) {
            toast.error('Error loading certificate');
        } finally {
            setIsLoading(false);
        }
    };

    const handleDownloadCertificate = async () => {
        setIsLoading(true);
        try {
            const response = await fetch(`/team/players/${player.id}/certificate/pdf`);
            if (response.ok) {
                // Get the PDF blob
                const blob = await response.blob();
                const url = window.URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = `player-certificate-${player.user.name.replace(/\s+/g, '-').toLowerCase()}.pdf`;
                document.body.appendChild(a);
                a.click();
                window.URL.revokeObjectURL(url);
                document.body.removeChild(a);
                toast.success('Certificate downloaded successfully');
            } else {
                toast.error('Failed to download certificate');
            }
        } catch (error) {
            toast.error('Error downloading certificate');
        } finally {
            setIsLoading(false);
        }
    };

    const handlePrintCertificate = () => {
        if (certificateUrl) {
            const printWindow = window.open(certificateUrl, '_blank');
            if (printWindow) {
                printWindow.onload = () => {
                    printWindow.print();
                };
            }
        } else {
            toast.error('Please view certificate first');
        }
    };

    return (
        <AppLayout breadcrumbs={[{ title: 'Player Certificate', href: '/team/players/certificate' }]}>
            <Head title={`Certificate - ${player.user.name}`} />
            
            <div className="">
                <div className="">
                    {/* Header */}
                    <div className="mb-8">
                        <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-4">
                                <Link
                                    href="/team/players"
                                    className="inline-flex items-center px-4 py-2 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50"
                                >
                                    <ArrowLeft className="h-4 w-4 mr-2" />
                                    Back to Players
                                </Link>
                                <div>
                                    <h1 className="text-3xl font-bold text-gray-900">
                                        Player Certificate
                                    </h1>
                                    <p className="text-gray-600">
                                        Generate and manage certificates for {player.user.name}
                                    </p>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Player Info Card */}
                    <Card className="mb-8">
                        <CardHeader>
                            <CardTitle className="flex items-center justify-between">
                                <span>Player Information</span>
                                <Badge variant={player.verification_status === 'verified' ? 'default' : 'secondary'}>
                                    {player.verification_status || 'Pending'}
                                </Badge>
                            </CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                                <div className="flex items-center space-x-4">
                                    <div className="flex-shrink-0">
                                        {player.avatar ? (
                                            <img
                                                className="h-16 w-16 rounded-full object-cover"
                                                src={player.avatar}
                                                alt={player.user.name}
                                            />
                                        ) : (
                                            <div className="h-16 w-16 rounded-full bg-gray-300 flex items-center justify-center">
                                                <span className="text-gray-600 font-medium text-lg">
                                                    {player.user.name.charAt(0).toUpperCase()}
                                                </span>
                                            </div>
                                        )}
                                    </div>
                                    <div>
                                        <h3 className="text-lg font-medium text-gray-900">{player.user.name}</h3>
                                        <p className="text-gray-600">{player.user.email}</p>
                                    </div>
                                </div>
                                
                                <div className="space-y-2">
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Team:</span>
                                        <p className="text-gray-900">{team.name}</p>
                                    </div>
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Position:</span>
                                        <p className="text-gray-900">{player.football_position || 'Not specified'}</p>
                                    </div>
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Jersey Number:</span>
                                        <p className="text-gray-900">{player.jersey_number || 'Not assigned'}</p>
                                    </div>
                                </div>
                                
                                <div className="space-y-2">
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Digital ID:</span>
                                        <p className="text-gray-900 font-mono text-sm">{player.digital_id || 'Not generated'}</p>
                                    </div>
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Global Rating:</span>
                                        <p className="text-gray-900">{player.global_rating || 0}</p>
                                    </div>
                                    <div>
                                        <span className="text-sm font-medium text-gray-500">Tournament Rating:</span>
                                        <p className="text-gray-900">{player.tournament_rating || 0}</p>
                                    </div>
                                </div>
                            </div>
                        </CardContent>
                    </Card>

                    {/* Player Statistics */}
                    <Card className="mb-8">
                        <CardHeader>
                            <CardTitle>Player Statistics</CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                                <div className="text-center p-4 bg-blue-50 rounded-lg">
                                    <div className="text-2xl font-bold text-blue-600">{player.tournament_matches_played || 0}</div>
                                    <div className="text-sm text-blue-800">Matches Played</div>
                                </div>
                                <div className="text-center p-4 bg-green-50 rounded-lg">
                                    <div className="text-2xl font-bold text-green-600">{player.tournament_matches_won || 0}</div>
                                    <div className="text-sm text-green-800">Matches Won</div>
                                </div>
                                <div className="text-center p-4 bg-purple-50 rounded-lg">
                                    <div className="text-2xl font-bold text-purple-600">{(Number(player.tournament_win_rate) || 0).toFixed(1)}%</div>
                                    <div className="text-sm text-purple-800">Win Rate</div>
                                </div>
                                <div className="text-center p-4 bg-orange-50 rounded-lg">
                                    <div className="text-2xl font-bold text-orange-600">{player.tournament_current_streak || 0}</div>
                                    <div className="text-sm text-orange-800">Current Streak</div>
                                </div>
                            </div>
                            
                            <div className="grid grid-cols-2 md:grid-cols-3 gap-4 mt-4">
                                <div className="text-center p-3 bg-gray-50 rounded-lg">
                                    <div className="text-lg font-bold text-gray-600">{player.tournament_longest_streak || 0}</div>
                                    <div className="text-sm text-gray-800">Longest Streak</div>
                                </div>
                                <div className="text-center p-3 bg-gray-50 rounded-lg">
                                    <div className="text-lg font-bold text-gray-600">{player.tournaments_won || 0}</div>
                                    <div className="text-sm text-gray-800">Tournaments Won</div>
                                </div>
                                <div className="text-center p-3 bg-gray-50 rounded-lg">
                                    <div className="text-lg font-bold text-gray-600">${(Number(player.total_earnings) || 0).toFixed(2)}</div>
                                    <div className="text-sm text-gray-800">Total Earnings</div>
                                </div>
                            </div>
                        </CardContent>
                    </Card>

                    {/* Certificate Actions */}
                    <Card>
                        <CardHeader>
                            <CardTitle>Certificate Actions</CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                                <Button
                                    onClick={handleViewCertificate}
                                    disabled={isLoading}
                                    className="w-full"
                                    variant="outline"
                                >
                                    <Eye className="h-4 w-4 mr-2" />
                                    {isLoading ? 'Loading...' : 'View Certificate'}
                                </Button>
                                
                                <Button
                                    onClick={handleDownloadCertificate}
                                    disabled={isLoading}
                                    className="w-full"
                                    variant="outline"
                                >
                                    <Download className="h-4 w-4 mr-2" />
                                    {isLoading ? 'Downloading...' : 'Download Certificate'}
                                </Button>
                                
                                <Button
                                    onClick={handlePrintCertificate}
                                    disabled={!certificateUrl || isLoading}
                                    className="w-full"
                                    variant="outline"
                                >
                                    <Printer className="h-4 w-4 mr-2" />
                                    Print Certificate
                                </Button>
                            </div>
                        </CardContent>
                    </Card>

                    {/* Certificate Preview */}
                    {certificateUrl && (
                        <Card className="mt-8">
                            <CardHeader>
                                <CardTitle>Certificate Preview</CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="border rounded-lg p-4 bg-white">
                                    <iframe
                                        src={certificateUrl}
                                        className="w-full h-96 border-0"
                                        title="Certificate Preview"
                                    />
                                </div>
                            </CardContent>
                        </Card>
                    )}
                </div>
            </div>
        </AppLayout>
    );
}
