import React, { useState, useEffect } from 'react';
import { Head, useForm, router } from '@inertiajs/react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { CreditCard, DollarSign, Calendar, CheckCircle, XCircle, Clock, ExternalLink, Download, Eye, Smartphone, Loader2, AlertCircle } from 'lucide-react';
import AppLayout from '@/layouts/app-layout';
import TeamSelector from '@/components/TeamManager/TeamSelector';
import { toast } from 'sonner';

interface Payment {
  id: number;
  tournament: {
    id: number;
    name: string;
  };
  amount: number;
  currency: string;
  status: string;
  payment_method: string;
  transaction_id?: string;
  paid_at?: string;
  created_at: string;
  due_date?: string;
  description?: string;
}

interface Team {
  id: number;
  name: string;
}

interface PaymentsProps {
  team: Team;
  teams: Team[];
  payments: {
    data: Payment[];
    current_page: number;
    last_page: number;
    per_page: number;
    total: number;
  };
}

interface PaymentState {
  status: 'idle' | 'initiating' | 'processing' | 'waiting' | 'completed' | 'failed';
  message: string;
  checkoutRequestId?: string;
  error?: string;
}

export default function TeamPayments({ team, teams, payments }: PaymentsProps) {
  const [isPaymentOpen, setIsPaymentOpen] = useState(false);
  const [selectedPayment, setSelectedPayment] = useState<Payment | null>(null);
  const [paymentState, setPaymentState] = useState<PaymentState>({
    status: 'idle',
    message: ''
  });
  const [pollingInterval, setPollingInterval] = useState<NodeJS.Timeout | null>(null);
  const [countdownInterval, setCountdownInterval] = useState<NodeJS.Timeout | null>(null);
  const [timeRemaining, setTimeRemaining] = useState(0);
  const [isPolling, setIsPolling] = useState(false);

  const { data, setData, post, processing, errors } = useForm({
    phone_number: '',
    amount: '',
    description: '',
  });

  // Cleanup intervals on unmount
  useEffect(() => {
    return () => {
      if (pollingInterval) {
        clearInterval(pollingInterval);
      }
      if (countdownInterval) {
        clearInterval(countdownInterval);
      }
    };
  }, [pollingInterval, countdownInterval]);

  const validatePhoneNumber = (phone: string): boolean => {
    const cleanPhone = phone.replace(/[\s\-\+]/g, '');
    const phoneRegex = /^07[0-9]{9}$/;
    return phoneRegex.test(cleanPhone);
  };

  const formatPhoneNumber = (phone: string): string => {
    let cleanPhone = phone.replace(/[\s\-\+]/g, '');
    
    if (cleanPhone.startsWith('0')) {
      cleanPhone = '07' + cleanPhone.substring(1);
    }
    
    if (cleanPhone.startsWith('7') || cleanPhone.startsWith('1')) {
      cleanPhone = '07' + cleanPhone;
    }
    
    return cleanPhone;
  };

  const startCountdown = (duration: number) => {
    setTimeRemaining(duration);
    const interval = setInterval(() => {
      setTimeRemaining((prev) => {
        if (prev <= 1) {
          clearInterval(interval);
          setCountdownInterval(null);
          return 0;
        }
        return prev - 1;
      });
    }, 1000);
    setCountdownInterval(interval);
  };

  const startPolling = (checkoutRequestId: string) => {
    setIsPolling(true);
    
    const poll = async () => {
      try {
        const response = await fetch(`/api/payment/status/${checkoutRequestId}`);
        const result = await response.json();
        
        if (result.success) {
          if (result.status === 'completed') {
            setPaymentState({
              status: 'completed',
              message: 'Payment completed successfully!',
            });
            setIsPolling(false);
            if (pollingInterval) {
              clearInterval(pollingInterval);
              setPollingInterval(null);
            }
            toast.success('Payment completed successfully!');
            
            // Refresh the page after 3 seconds
            setTimeout(() => {
              router.reload();
            }, 3000);
            
          } else if (result.status === 'failed') {
            setPaymentState({
              status: 'failed',
              message: result.failure_reason || 'Payment failed',
              error: result.failure_reason || 'Payment failed',
            });
            setIsPolling(false);
            if (pollingInterval) {
              clearInterval(pollingInterval);
              setPollingInterval(null);
            }
            toast.error('Payment failed: ' + (result.failure_reason || 'Unknown error'));
            
          } else {
            // Still processing, continue polling
          }
        }
      } catch (error) {
        console.error('Polling error:', error);
      }
    };

    // Start polling every 3 seconds
    poll();
    const interval = setInterval(poll, 3000);
    setPollingInterval(interval);
  };

  const handleMakePayment = (payment: Payment) => {
    setSelectedPayment(payment);
    setPaymentState({ status: 'idle', message: '' });
    setData({
      phone_number: '',
      amount: payment.amount.toString(),
      description: `Payment for ${payment?.tournament?.name}`,
    });
    setIsPaymentOpen(true);
  };

  const handleSubmitPayment = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!selectedPayment) return;

    const formattedPhone = formatPhoneNumber(data.phone_number);
    
    if (!validatePhoneNumber(formattedPhone)) {
      toast.error('Please enter a valid phone number in format 07XXXXXXXX');
      return;
    }

    setPaymentState({
      status: 'initiating',
      message: 'Initiating M-Pesa payment...'
    });

    try {
      const response = await fetch(route('team.payments.process', selectedPayment.id), {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
        },
        body: JSON.stringify({
          phone_number: formattedPhone,
        }),
      });

      const result = await response.json();

      if (result.success) {
        setPaymentState({
          status: 'waiting',
          message: result.message || 'Please check your phone and enter your M-Pesa PIN',
          checkoutRequestId: result.payment?.checkout_request_id,
        });
        
        startCountdown(120); // 2 minutes timeout
        startPolling(result.payment?.checkout_request_id);
        
        toast.success('M-Pesa payment initiated! Please check your phone.');
        
      } else {
        setPaymentState({
          status: 'failed',
          message: result.error || 'Payment initiation failed',
          error: result.error,
        });
        toast.error(result.error || 'Payment initiation failed');
      }

    } catch (error) {
      console.error('Payment error:', error);
      setPaymentState({
        status: 'failed',
        message: 'Network error occurred',
        error: 'Network error occurred',
      });
      toast.error('Network error occurred. Please try again.');
    }
  };

  const getStatusColor = (status: string) => {
    switch (status?.toLowerCase()) {
      case 'paid':
        return 'bg-green-100 text-green-800';
      case 'pending':
        return 'bg-yellow-100 text-yellow-800';
      case 'failed':
        return 'bg-red-100 text-red-800';
      case 'cancelled':
        return 'bg-gray-100 text-gray-800';
      case 'refunded':
        return 'bg-blue-100 text-blue-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status?.toLowerCase()) {
      case 'paid':
        return <CheckCircle className="w-4 h-4" />;
      case 'pending':
        return <Clock className="w-4 h-4" />;
      case 'failed':
        return <XCircle className="w-4 h-4" />;
      default:
        return <Clock className="w-4 h-4" />;
    }
  };

  const formatCurrency = (amount: number, currency: string) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: currency || 'KES',
    }).format(amount);
  };

  const formatDate = (date: string) => {
    return new Date(date).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
    });
  };

  const formatDateTime = (date: string) => {
    return new Date(date).toLocaleString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  const isOverdue = (payment: Payment) => {
    if (payment.status === 'paid' || !payment.due_date) return false;
    return new Date(payment.due_date) < new Date();
  };

  const totalPaid = payments.data
    .filter(p => p.status === 'paid')
    .reduce((sum, p) => sum + parseInt(p.amount.toString()), 0);

  const totalPending = payments.data
    .filter(p => p.status === 'pending')
    .reduce((sum, p) => sum + parseInt(p.amount.toString()), 0);

  const overduePayments = payments.data.filter(isOverdue);

  return (
    <AppLayout>
      <Head title="Team Payments" />
      
      <div className="space-y-6 p-6">
        {/* Team Selector */}
        <TeamSelector teams={teams} selectedTeamId={team.id} currentPath="/team/payments" />

        {/* Header */}
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-gray-900">Team Payments</h1>
            <p className="text-gray-600">Manage your team's tournament payments and invoices</p>
          </div>
        </div>

        {/* Payment Statistics */}
        <div className="grid gap-4 md:grid-cols-4">
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Total Paid</p>
                  <p className="text-2xl font-bold text-green-600">
                    {formatCurrency(totalPaid, 'KES')}
                  </p>
                </div>
                <CheckCircle className="w-8 h-8 text-green-600" />
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Pending</p>
                  <p className="text-2xl font-bold text-yellow-600">
                    {formatCurrency(totalPending, 'KES')}
                  </p>
                </div>
                <Clock className="w-8 h-8 text-yellow-600" />
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Overdue</p>
                  <p className="text-2xl font-bold text-red-600">
                    {overduePayments.length}
                  </p>
                </div>
                <XCircle className="w-8 h-8 text-red-600" />
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Total Payments</p>
                  <p className="text-2xl font-bold">{payments.total}</p>
                </div>
                <CreditCard className="w-8 h-8 text-blue-600" />
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Overdue Payments Alert */}
        {overduePayments.length > 0 && (
          <Card className="border-red-200 bg-red-50">
            <CardContent className="p-4">
              <div className="flex items-center gap-2 mb-2">
                <XCircle className="w-5 h-5 text-red-600" />
                <h3 className="font-semibold text-red-800">Overdue Payments</h3>
              </div>
              <p className="text-red-700 text-sm mb-3">
                You have {overduePayments.length} overdue payment(s) that need immediate attention.
              </p>
              <div className="flex gap-2">
                {overduePayments.map((payment) => (
                  <Button
                    key={payment.id}
                    size="sm"
                    variant="outline"
                    className="border-red-300 text-red-700 hover:bg-red-100"
                    onClick={() => handleMakePayment(payment)}
                  >
                    Pay {formatCurrency(payment.amount, payment.currency)}
                  </Button>
                ))}
              </div>
            </CardContent>
          </Card>
        )}

        {/* Payments List */}
        <Card>
          <CardHeader>
            <CardTitle>Payment History</CardTitle>
            <CardDescription>
              All payment transactions for {team.name}
            </CardDescription>
          </CardHeader>
          <CardContent>
            {payments.data.length > 0 ? (
              <div className="space-y-4">
                {payments.data.map((payment) => (
                  <div
                    key={payment.id}
                    className="flex items-center justify-between p-4 border rounded-lg hover:bg-gray-50"
                  >
                    <div className="flex items-center gap-4">
                      <div className="w-10 h-10 bg-gray-100 rounded-full flex items-center justify-center">
                        <CreditCard className="w-5 h-5 text-gray-600" />
                      </div>
                      <div>
                        <h3 className="font-semibold">{payment?.tournament?.name}</h3>
                        <p className="text-sm text-gray-500">
                          {payment.description || `Tournament entry fee`}
                        </p>
                        <p className="text-xs text-gray-400">
                          Created: {formatDateTime(payment.created_at)}
                        </p>
                      </div>
                    </div>

                    <div className="flex items-center gap-4">
                      <div className="text-right">
                        <p className="font-semibold">
                          {formatCurrency(payment.amount, payment.currency)}
                        </p>
                        <p className="text-sm text-gray-500 capitalize">
                          {payment.payment_method.replace('_', ' ')}
                        </p>
                        {payment.transaction_id && (
                          <p className="text-xs text-gray-400">
                            ID: {payment.transaction_id}
                          </p>
                        )}
                      </div>

                      <div className="flex items-center gap-2">
                        <Badge className={getStatusColor(payment.status)}>
                          <div className="flex items-center gap-1">
                            {getStatusIcon(payment.status)}
                            {payment.status}
                          </div>
                        </Badge>
                      </div>

                      <div className="flex gap-1">
                        {payment.status === 'pending' && (
                          <Button
                            size="sm"
                            onClick={() => handleMakePayment(payment)}
                          >
                            Pay Now
                          </Button>
                        )}
                        <Button size="sm" variant="outline">
                          <Eye className="w-4 h-4" />
                        </Button>
                        <Button size="sm" variant="outline">
                          <Download className="w-4 h-4" />
                        </Button>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            ) : (
              <div className="text-center py-12">
                <CreditCard className="w-12 h-12 text-gray-300 mx-auto mb-4" />
                <h3 className="text-lg font-semibold text-gray-900 mb-2">No Payments Yet</h3>
                <p className="text-gray-500">Your team hasn't made any payments yet</p>
              </div>
            )}
          </CardContent>
        </Card>

        {/* M-Pesa Payment Dialog */}
        <Dialog open={isPaymentOpen} onOpenChange={setIsPaymentOpen}>
          <DialogContent className="max-w-md">
            <DialogHeader>
              <DialogTitle className="flex items-center gap-2">
                <Smartphone className="w-5 h-5 text-green-600" />
                M-Pesa Payment
              </DialogTitle>
              <DialogDescription>
                Complete payment for {selectedPayment?.tournament.name}
              </DialogDescription>
            </DialogHeader>
            
            {selectedPayment && (
              <div className="space-y-6">
                {/* Payment Details */}
                <div className="bg-gray-50 p-4 rounded-lg">
                  <h3 className="font-semibold mb-3">Payment Details</h3>
                  <div className="space-y-2 text-sm">
                    <div className="flex justify-between">
                      <span>Tournament:</span>
                      <span className="font-medium">{selectedPayment?.tournament?.name}</span>
                    </div>
                    <div className="flex justify-between">
                      <span>Amount:</span>
                      <span className="font-medium text-green-600">
                        {formatCurrency(selectedPayment.amount, selectedPayment.currency)}
                      </span>
                    </div>
                    {selectedPayment.due_date && (
                      <div className="flex justify-between">
                        <span>Due Date:</span>
                        <span className="font-medium">{formatDate(selectedPayment.due_date)}</span>
                      </div>
                    )}
                  </div>
                </div>

                {/* Payment Status Display */}
                {paymentState.status !== 'idle' && (
                  <div className={`p-4 rounded-lg ${
                    paymentState.status === 'completed' ? 'bg-green-50 border border-green-200' :
                    paymentState.status === 'failed' ? 'bg-red-50 border border-red-200' :
                    paymentState.status === 'waiting' ? 'bg-blue-50 border border-blue-200' :
                    'bg-yellow-50 border border-yellow-200'
                  }`}>
                    <div className="flex items-center gap-2 mb-2">
                      {paymentState.status === 'completed' && <CheckCircle className="w-5 h-5 text-green-600" />}
                      {paymentState.status === 'failed' && <XCircle className="w-5 h-5 text-red-600" />}
                      {paymentState.status === 'waiting' && <Clock className="w-5 h-5 text-blue-600" />}
                      {paymentState.status === 'initiating' && <Loader2 className="w-5 h-5 text-yellow-600 animate-spin" />}
                      <span className={`font-medium ${
                        paymentState.status === 'completed' ? 'text-green-800' :
                        paymentState.status === 'failed' ? 'text-red-800' :
                        paymentState.status === 'waiting' ? 'text-blue-800' :
                        'text-yellow-800'
                      }`}>
                        {paymentState.status === 'completed' ? 'Payment Completed' :
                         paymentState.status === 'failed' ? 'Payment Failed' :
                         paymentState.status === 'waiting' ? 'Waiting for Payment' :
                         'Initiating Payment'}
                      </span>
                    </div>
                    <p className={`text-sm ${
                      paymentState.status === 'completed' ? 'text-green-700' :
                      paymentState.status === 'failed' ? 'text-red-700' :
                      paymentState.status === 'waiting' ? 'text-blue-700' :
                      'text-yellow-700'
                    }`}>
                      {paymentState.message}
                    </p>
                    
                    {/* Countdown Timer */}
                    {paymentState.status === 'waiting' && timeRemaining > 0 && (
                      <div className="mt-3 flex items-center gap-2">
                        <Clock className="w-4 h-4 text-blue-600" />
                        <span className="text-sm text-blue-700">
                          Time remaining: {Math.floor(timeRemaining / 60)}:{(timeRemaining % 60).toString().padStart(2, '0')}
                        </span>
                      </div>
                    )}
                  </div>
                )}

                {/* Phone Number Input (only show when not processing) */}
                {paymentState.status === 'idle' && (
                  <form onSubmit={handleSubmitPayment} className="space-y-4">
                    <div>
                      <Label htmlFor="phone_number">M-Pesa Phone Number</Label>
                      <Input
                        id="phone_number"
                        type="tel"
                        value={data.phone_number}
                        onChange={(e) => setData('phone_number', e.target.value)}
                        placeholder="07XXXXXXXX"
                        required
                      />
                      <p className="text-xs text-gray-500 mt-1">
                        Enter your M-Pesa registered phone number (e.g., 0712345678)
                      </p>
                      {errors.phone_number && (
                        <p className="text-red-500 text-sm mt-1">{errors.phone_number}</p>
                      )}
                    </div>

                    <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                      <div className="flex items-center gap-2 mb-2">
                        <Smartphone className="w-4 h-4 text-blue-600" />
                        <span className="font-medium text-blue-800">M-Pesa Payment</span>
                      </div>
                      <p className="text-sm text-blue-700">
                        You will receive an M-Pesa prompt on your phone. Enter your M-Pesa PIN to complete the payment.
                      </p>
                    </div>

                    <DialogFooter>
                      <Button type="button" variant="outline" onClick={() => setIsPaymentOpen(false)}>
                        Cancel
                      </Button>
                      <Button type="submit" disabled={processing || !data.phone_number}>
                        {processing ? 'Processing...' : 'Pay with M-Pesa'}
                      </Button>
                    </DialogFooter>
                  </form>
                )}

                {/* Close button for completed/failed states */}
                {(paymentState.status === 'completed' || paymentState.status === 'failed') && (
                  <DialogFooter>
                    <Button 
                      type="button" 
                      variant="outline" 
                      onClick={() => {
                        setIsPaymentOpen(false);
                        setPaymentState({ status: 'idle', message: '' });
                      }}
                      className="w-full"
                    >
                      Close
                    </Button>
                  </DialogFooter>
                )}
              </div>
            )}
          </DialogContent>
        </Dialog>
      </div>
    </AppLayout>
  );
}
