import React, { useState } from 'react';
import { Head, router } from '@inertiajs/react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Gamepad2, Calendar, Clock, Trophy, Users, MapPin, User, ExternalLink } from 'lucide-react';
import AppLayout from '@/layouts/app-layout';
import TeamSelector from '@/components/TeamManager/TeamSelector';

interface Match {
  id: number;
  name: string;
  tournament: {
    id: number;
    name: string;
  };
  team1: {
    id: number;
    name: string;
  };
  team2: {
    id: number;
    name: string;
  };
  referee?: {
    id: number;
    name: string;
  };
  scheduled_time: string;
  status: string;
  score_team1?: number;
  score_team2?: number;
  location?: string;
  match_type: string;
  round?: string;
  created_at: string;
}

interface Team {
  id: number;
  name: string;
}

interface MatchesProps {
  team: Team;
  teams: Team[];
  matches: {
    data: Match[];
    current_page: number;
    last_page: number;
    per_page: number;
    total: number;
  };
}

export default function TeamMatches({ team, teams, matches }: MatchesProps) {
  const [activeTab, setActiveTab] = useState('upcoming');

  const getStatusColor = (status: string) => {
    switch (status?.toLowerCase()) {
      case 'scheduled':
        return 'bg-blue-100 text-blue-800';
      case 'in_progress':
        return 'bg-yellow-100 text-yellow-800';
      case 'completed':
        return 'bg-green-100 text-green-800';
      case 'cancelled':
        return 'bg-red-100 text-red-800';
      case 'postponed':
        return 'bg-gray-100 text-gray-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const getMatchTypeColor = (matchType: string) => {
    switch (matchType?.toLowerCase()) {
      case 'group_stage':
        return 'bg-blue-100 text-blue-800';
      case 'knockout':
        return 'bg-red-100 text-red-800';
      case 'final':
        return 'bg-purple-100 text-purple-800';
      case 'semi_final':
        return 'bg-orange-100 text-orange-800';
      case 'quarter_final':
        return 'bg-yellow-100 text-yellow-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const formatDateTime = (dateTime: string) => {
    return new Date(dateTime).toLocaleString('en-US', {
      weekday: 'short',
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  const isUpcoming = (match: Match) => {
    return new Date(match.scheduled_time) > new Date() && match.status === 'scheduled';
  };

  const isCompleted = (match: Match) => {
    return match.status === 'completed';
  };

  const isInProgress = (match: Match) => {
    return match.status === 'in_progress';
  };

  const upcomingMatches = matches.data.filter(isUpcoming);
  const completedMatches = matches.data.filter(isCompleted);
  const inProgressMatches = matches.data.filter(isInProgress);

  const getMatchResult = (match: Match) => {
    if (match.status !== 'completed' || !match.score_team1 || !match.score_team2) {
      return null;
    }

    const isTeam1 = match.team1.id === team.id;
    const teamScore = isTeam1 ? match.score_team1 : match.score_team2;
    const opponentScore = isTeam1 ? match.score_team2 : match.score_team1;

    if (teamScore > opponentScore) {
      return { result: 'win', score: `${teamScore}-${opponentScore}` };
    } else if (teamScore < opponentScore) {
      return { result: 'loss', score: `${teamScore}-${opponentScore}` };
    } else {
      return { result: 'draw', score: `${teamScore}-${opponentScore}` };
    }
  };

  const getResultColor = (result: string) => {
    switch (result) {
      case 'win':
        return 'text-green-600';
      case 'loss':
        return 'text-red-600';
      case 'draw':
        return 'text-gray-600';
      default:
        return 'text-gray-600';
    }
  };

  const renderMatchCard = (match: Match) => {
    const result = getMatchResult(match);
    const isTeam1 = match.team1.id === team.id;
    const opponent = isTeam1 ? match.team2 : match.team1;

    return (
      <Card key={match.id} className="hover:shadow-md transition-shadow">
        <CardHeader>
          <div className="flex items-center justify-between">
            <CardTitle className="flex items-center gap-2">
              <Gamepad2 className="w-5 h-5" />
              {match.name}
            </CardTitle>
            <div className="flex gap-2">
              <Badge className={getStatusColor(match.status)}>
                {match.status.replace('_', ' ')}
              </Badge>
              <Badge className={getMatchTypeColor(match.match_type)}>
                {match.match_type.replace('_', ' ')}
              </Badge>
            </div>
          </div>
          <CardDescription>
            {match.tournament.name}
            {match.round && ` • ${match.round}`}
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            {/* Teams */}
            <div className="flex items-center justify-between">
              <div className="flex items-center gap-3">
                <div className="w-8 h-8 bg-gray-100 rounded-full flex items-center justify-center">
                  <Users className="w-4 h-4 text-gray-600" />
                </div>
                <div>
                  <p className="font-medium">{team.name}</p>
                  <p className="text-sm text-gray-500">Your Team</p>
                </div>
              </div>
              
              <div className="text-center">
                {match.status === 'completed' && result ? (
                  <div className="space-y-1">
                    <p className={`text-2xl font-bold ${getResultColor(result.result)}`}>
                      {result.score}
                    </p>
                    <p className={`text-sm font-medium ${getResultColor(result.result)}`}>
                      {result.result.toUpperCase()}
                    </p>
                  </div>
                ) : (
                  <div className="text-gray-400">VS</div>
                )}
              </div>

              <div className="flex items-center gap-3">
                <div>
                  <p className="font-medium text-right">{opponent.name}</p>
                  <p className="text-sm text-gray-500 text-right">Opponent</p>
                </div>
                <div className="w-8 h-8 bg-gray-100 rounded-full flex items-center justify-center">
                  <Users className="w-4 h-4 text-gray-600" />
                </div>
              </div>
            </div>

            {/* Match Details */}
            <div className="grid gap-2 text-sm">
              <div className="flex items-center justify-between">
                <span className="text-gray-600 flex items-center gap-1">
                  <Calendar className="w-4 h-4" />
                  Date & Time
                </span>
                <span className="font-medium">{formatDateTime(match.scheduled_time)}</span>
              </div>
              
              {match.location && (
                <div className="flex items-center justify-between">
                  <span className="text-gray-600 flex items-center gap-1">
                    <MapPin className="w-4 h-4" />
                    Location
                  </span>
                  <span className="font-medium">{match.location}</span>
                </div>
              )}

              {match.referee && (
                <div className="flex items-center justify-between">
                  <span className="text-gray-600 flex items-center gap-1">
                    <User className="w-4 h-4" />
                    Referee
                  </span>
                  <span className="font-medium">{match.referee.name}</span>
                </div>
              )}
            </div>

            {/* Actions */}
            <div className="flex gap-2 pt-2">
              {match.status === 'scheduled' && (
                <Button 
                  size="sm" 
                  variant="outline" 
                  className="flex-1"
                  onClick={() => router.visit(route('team.matches.show', match.id))}
                >
                  <Clock className="w-4 h-4 mr-2" />
                  View Details
                </Button>
              )}
              {match.status === 'in_progress' && (
                <Button 
                  size="sm" 
                  className="flex-1"
                  onClick={() => router.visit(route('team.matches.show', match.id))}
                >
                  <Gamepad2 className="w-4 h-4 mr-2" />
                  Live Match
                </Button>
              )}
              {match.status === 'completed' && (
                <Button 
                  size="sm" 
                  variant="outline" 
                  className="flex-1"
                  onClick={() => router.visit(route('team.matches.show', match.id))}
                >
                  <Trophy className="w-4 h-4 mr-2" />
                  View Results
                </Button>
              )}
            </div>
          </div>
        </CardContent>
      </Card>
    );
  };

  return (
    <AppLayout>
      <Head title="Team Matches" />
      
      <div className="space-y-6 p-6">
        {/* Team Selector */}
        <TeamSelector 
          teams={teams} 
          selectedTeamId={team.id} 
          currentPath="/team/matches" 
        />

        {/* Header */}
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-gray-900">Team Matches</h1>
            <p className="text-gray-600">Track your team's match schedule and results</p>
          </div>
        </div>

        {/* Match Statistics */}
        <div className="grid gap-4 md:grid-cols-4">
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Total Matches</p>
                  <p className="text-2xl font-bold">{matches.total}</p>
                </div>
                <Gamepad2 className="w-8 h-8 text-blue-600" />
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Upcoming</p>
                  <p className="text-2xl font-bold">{upcomingMatches.length}</p>
                </div>
                <Clock className="w-8 h-8 text-yellow-600" />
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">In Progress</p>
                  <p className="text-2xl font-bold">{inProgressMatches.length}</p>
                </div>
                <Gamepad2 className="w-8 h-8 text-orange-600" />
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Completed</p>
                  <p className="text-2xl font-bold">{completedMatches.length}</p>
                </div>
                <Trophy className="w-8 h-8 text-green-600" />
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Matches Tabs */}
        <Tabs value={activeTab} onValueChange={setActiveTab}>
          <TabsList className="grid w-full grid-cols-3">
            <TabsTrigger value="upcoming">Upcoming ({upcomingMatches.length})</TabsTrigger>
            <TabsTrigger value="in_progress">In Progress ({inProgressMatches.length})</TabsTrigger>
            <TabsTrigger value="completed">Completed ({completedMatches.length})</TabsTrigger>
          </TabsList>

          <TabsContent value="upcoming" className="space-y-4">
            {upcomingMatches.length > 0 ? (
              <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
                {upcomingMatches.map(renderMatchCard)}
              </div>
            ) : (
              <Card>
                <CardContent className="text-center py-12">
                  <Clock className="w-12 h-12 text-gray-300 mx-auto mb-4" />
                  <h3 className="text-lg font-semibold text-gray-900 mb-2">No Upcoming Matches</h3>
                  <p className="text-gray-500">Your team doesn't have any scheduled matches at the moment</p>
                </CardContent>
              </Card>
            )}
          </TabsContent>

          <TabsContent value="in_progress" className="space-y-4">
            {inProgressMatches.length > 0 ? (
              <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
                {inProgressMatches.map(renderMatchCard)}
              </div>
            ) : (
              <Card>
                <CardContent className="text-center py-12">
                  <Gamepad2 className="w-12 h-12 text-gray-300 mx-auto mb-4" />
                  <h3 className="text-lg font-semibold text-gray-900 mb-2">No Matches in Progress</h3>
                  <p className="text-gray-500">Your team doesn't have any matches currently in progress</p>
                </CardContent>
              </Card>
            )}
          </TabsContent>

          <TabsContent value="completed" className="space-y-4">
            {completedMatches.length > 0 ? (
              <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
                {completedMatches.map(renderMatchCard)}
              </div>
            ) : (
              <Card>
                <CardContent className="text-center py-12">
                  <Trophy className="w-12 h-12 text-gray-300 mx-auto mb-4" />
                  <h3 className="text-lg font-semibold text-gray-900 mb-2">No Completed Matches</h3>
                  <p className="text-gray-500">Your team hasn't completed any matches yet</p>
                </CardContent>
              </Card>
            )}
          </TabsContent>
        </Tabs>
      </div>
    </AppLayout>
  );
}
