import React from 'react';
import { Head, router } from '@inertiajs/react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { 
  Gamepad2, 
  Calendar, 
  Clock, 
  Trophy, 
  Users, 
  MapPin, 
  User, 
  ArrowLeft,
  Shield,
  Target,
  AlertCircle
} from 'lucide-react';
import AppLayout from '@/layouts/app-layout';
import TeamSelector from '@/components/TeamManager/TeamSelector';

interface Match {
  id: number;
  name: string;
  tournament: {
    id: number;
    name: string;
  };
  team1: {
    id: number;
    name: string;
  };
  team2: {
    id: number;
    name: string;
  };
  referee?: {
    id: number;
    name: string;
  };
  bracket?: {
    id: number;
    name: string;
  };
  scheduled_time: string;
  status: string;
  score_team1?: number;
  score_team2?: number;
  location?: string;
  match_type: string;
  round?: string;
  notes?: string;
  created_at: string;
  updated_at: string;
}

interface Team {
  id: number;
  name: string;
}

interface MatchDetailsProps {
  team: Team;
  teams: Team[];
  match: Match;
}

export default function MatchDetails({ team, teams, match }: MatchDetailsProps) {
  const getStatusColor = (status: string) => {
    switch (status?.toLowerCase()) {
      case 'scheduled':
        return 'bg-blue-100 text-blue-800';
      case 'in_progress':
        return 'bg-yellow-100 text-yellow-800';
      case 'completed':
        return 'bg-green-100 text-green-800';
      case 'cancelled':
        return 'bg-red-100 text-red-800';
      case 'postponed':
        return 'bg-gray-100 text-gray-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const getMatchTypeColor = (matchType: string) => {
    switch (matchType?.toLowerCase()) {
      case 'group_stage':
        return 'bg-blue-100 text-blue-800';
      case 'knockout':
        return 'bg-red-100 text-red-800';
      case 'final':
        return 'bg-purple-100 text-purple-800';
      case 'semi_final':
        return 'bg-orange-100 text-orange-800';
      case 'quarter_final':
        return 'bg-yellow-100 text-yellow-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const formatDateTime = (dateTime: string) => {
    return new Date(dateTime).toLocaleString('en-US', {
      weekday: 'long',
      year: 'numeric',
      month: 'long',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  const isTeam1 = match.team1.id === team.id;
  const opponent = isTeam1 ? match.team2 : match.team1;

  const getMatchResult = () => {
    if (match.status !== 'completed' || match.score_team1 === undefined || match.score_team2 === undefined) {
      return null;
    }

    const teamScore = isTeam1 ? match.score_team1 : match.score_team2;
    const opponentScore = isTeam1 ? match.score_team2 : match.score_team1;

    if (teamScore > opponentScore) {
      return { result: 'win', color: 'text-green-600', bgColor: 'bg-green-50' };
    } else if (teamScore < opponentScore) {
      return { result: 'loss', color: 'text-red-600', bgColor: 'bg-red-50' };
    } else {
      return { result: 'draw', color: 'text-gray-600', bgColor: 'bg-gray-50' };
    }
  };

  const result = getMatchResult();

  return (
    <AppLayout>
      <Head title={`Match Details - ${match.name}`} />
      
      <div className="space-y-6 p-6">
        {/* Team Selector */}
        <TeamSelector 
          teams={teams} 
          selectedTeamId={team.id} 
          currentPath="/team/matches" 
        />

        {/* Back Button */}
        <Button
          variant="outline"
          onClick={() => router.visit(route('team.matches'))}
        >
          <ArrowLeft className="w-4 h-4 mr-2" />
          Back to Matches
        </Button>

        {/* Match Header */}
        <div>
          <div className="flex items-center gap-2 mb-2">
            <Badge className={getStatusColor(match.status)}>
              {match.status.replace('_', ' ').toUpperCase()}
            </Badge>
            <Badge className={getMatchTypeColor(match.match_type)}>
              {match.match_type.replace('_', ' ')}
            </Badge>
          </div>
          <h1 className="text-3xl font-bold text-gray-900">{match.name}</h1>
          <p className="text-gray-600 mt-1">
            {match.tournament.name}
            {match.round && ` • ${match.round}`}
          </p>
        </div>

        {/* Match Result Card (if completed) */}
        {result && (
          <Card className={result.bgColor}>
            <CardContent className="pt-6">
              <div className="text-center">
                <p className={`text-4xl font-bold ${result.color} mb-2`}>
                  {result.result.toUpperCase()}
                </p>
                <p className="text-2xl font-semibold">
                  {isTeam1 ? match.score_team1 : match.score_team2} - {isTeam1 ? match.score_team2 : match.score_team1}
                </p>
              </div>
            </CardContent>
          </Card>
        )}

        <div className="grid gap-6 lg:grid-cols-2">
          {/* Teams Card */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Users className="w-5 h-5" />
                Teams
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-6">
              {/* Your Team */}
              <div className={`p-4 rounded-lg ${isTeam1 ? 'bg-blue-50' : 'bg-gray-50'}`}>
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm text-gray-600 mb-1">Your Team</p>
                    <p className="text-lg font-bold">{team.name}</p>
                  </div>
                  {match.status === 'completed' && match.score_team1 !== undefined && (
                    <div className="text-3xl font-bold">
                      {isTeam1 ? match.score_team1 : match.score_team2}
                    </div>
                  )}
                </div>
              </div>

              {/* VS Divider */}
              <div className="text-center text-gray-400 font-bold">VS</div>

              {/* Opponent Team */}
              <div className={`p-4 rounded-lg ${!isTeam1 ? 'bg-blue-50' : 'bg-gray-50'}`}>
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm text-gray-600 mb-1">Opponent</p>
                    <p className="text-lg font-bold">{opponent.name}</p>
                  </div>
                  {match.status === 'completed' && match.score_team2 !== undefined && (
                    <div className="text-3xl font-bold">
                      {isTeam1 ? match.score_team2 : match.score_team1}
                    </div>
                  )}
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Match Information */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <AlertCircle className="w-5 h-5" />
                Match Information
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex items-center justify-between py-2 border-b">
                <span className="text-gray-600 flex items-center gap-2">
                  <Calendar className="w-4 h-4" />
                  Date & Time
                </span>
                <span className="font-medium text-right">{formatDateTime(match.scheduled_time)}</span>
              </div>

              {match.location && (
                <div className="flex items-center justify-between py-2 border-b">
                  <span className="text-gray-600 flex items-center gap-2">
                    <MapPin className="w-4 h-4" />
                    Location
                  </span>
                  <span className="font-medium">{match.location}</span>
                </div>
              )}

              <div className="flex items-center justify-between py-2 border-b">
                <span className="text-gray-600 flex items-center gap-2">
                  <Trophy className="w-4 h-4" />
                  Tournament
                </span>
                <span className="font-medium">{match.tournament.name}</span>
              </div>

              {match.bracket && (
                <div className="flex items-center justify-between py-2 border-b">
                  <span className="text-gray-600 flex items-center gap-2">
                    <Target className="w-4 h-4" />
                    Bracket
                  </span>
                  <span className="font-medium">{match.bracket.name}</span>
                </div>
              )}

              {match.referee && (
                <div className="flex items-center justify-between py-2 border-b">
                  <span className="text-gray-600 flex items-center gap-2">
                    <User className="w-4 h-4" />
                    Referee
                  </span>
                  <span className="font-medium">{match.referee.name}</span>
                </div>
              )}

              <div className="flex items-center justify-between py-2">
                <span className="text-gray-600 flex items-center gap-2">
                  <Gamepad2 className="w-4 h-4" />
                  Match Type
                </span>
                <Badge className={getMatchTypeColor(match.match_type)}>
                  {match.match_type.replace('_', ' ')}
                </Badge>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Notes */}
        {match.notes && (
          <Card>
            <CardHeader>
              <CardTitle>Match Notes</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-gray-700">{match.notes}</p>
            </CardContent>
          </Card>
        )}

        {/* Action Buttons */}
        <div className="flex gap-2">
          {match.status === 'in_progress' && (
            <Card className="bg-yellow-50 border-yellow-200 flex-1">
              <CardContent className="pt-6">
                <div className="text-center">
                  <Gamepad2 className="w-12 h-12 text-yellow-600 mx-auto mb-2" />
                  <p className="font-semibold text-yellow-800">Match is currently in progress</p>
                  <p className="text-sm text-yellow-700 mt-1">Check back later for final results</p>
                </div>
              </CardContent>
            </Card>
          )}

          {match.status === 'scheduled' && (
            <Card className="bg-blue-50 border-blue-200 flex-1">
              <CardContent className="pt-6">
                <div className="text-center">
                  <Clock className="w-12 h-12 text-blue-600 mx-auto mb-2" />
                  <p className="font-semibold text-blue-800">Match is scheduled</p>
                  <p className="text-sm text-blue-700 mt-1">Make sure your team is ready for the match time</p>
                </div>
              </CardContent>
            </Card>
          )}
        </div>
      </div>
    </AppLayout>
  );
}

