import React, { useState } from 'react';
import { Head, router } from '@inertiajs/react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { 
  ArrowLeft, 
  Edit, 
  Trash2, 
  Star, 
  StarOff, 
  Users, 
  Target, 
  MapPin,
  Calendar,
  Shirt,
  User,
  Download,
  Copy,
  Share2,
  Settings
} from 'lucide-react';
import AppLayout from '@/layouts/app-layout';
import TeamSelector from '@/components/TeamManager/TeamSelector';
import FormationField from '@/components/TeamManager/FormationField';

interface Player {
  id: number;
  user: {
    id: number;
    name: string;
    email: string;
  };
  display_name?: string;
  avatar?: string;
  football_position?: string;
  jersey_number?: number;
  status: string;
}

interface Formation {
  id: number;
  name: string;
  description?: string;
  player_positions: Record<string, any>;
  formation_data: any;
  is_default: boolean;
  is_active: boolean;
  created_at: string;
  formation: {
    id: number;
    name: string;
    display_name: string;
    description: string;
    category: string;
    country: string;
  };
}

interface Team {
  id: number;
  name: string;
}

interface ShowFormationProps {
  team: Team;
  teams: Team[];
  formation: Formation;
  players: Player[];
}

export default function ShowFormation({ team, teams, formation, players }: ShowFormationProps) {
  const [showPlayerDetails, setShowPlayerDetails] = useState(false);

  const handleDeleteFormation = () => {
    if (confirm('Are you sure you want to delete this formation?')) {
      router.delete(route('team.formations.destroy', formation.id), {
        data: { team_id: team.id }
      });
    }
  };

  const handleSetDefault = () => {
    router.post(route('team.formations.set-default', formation.id), {
      team_id: team.id
    });
  };

  const getCategoryColor = (category: string) => {
    switch (category?.toLowerCase()) {
      case 'attacking':
        return 'bg-red-100 text-red-800';
      case 'defensive':
        return 'bg-blue-100 text-blue-800';
      case 'standard':
        return 'bg-green-100 text-green-800';
      case 'modern':
        return 'bg-purple-100 text-purple-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const getPositionColor = (key: string) => {
    if (key === 'goalkeeper') return 'bg-blue-100 text-blue-800';
    if (key.startsWith('defender')) return 'bg-green-100 text-green-800';
    if (key.startsWith('midfielder')) return 'bg-yellow-100 text-yellow-800';
    if (key.startsWith('forward')) return 'bg-red-100 text-red-800';
    return 'bg-gray-100 text-gray-800';
  };

  const getFormationStats = () => {
    const positions = formation.formation_data;
    const playerCount = Object.keys(formation.player_positions || {}).length;
    
    return {
      totalPlayers: playerCount,
      hasGoalkeeper: positions.goalkeeper ? 1 : 0,
      defenders: positions.defenders?.length || 0,
      midfielders: positions.midfielders?.length || 0,
      forwards: positions.forwards?.length || 0,
      assignedPlayers: Object.values(formation.player_positions || {}).filter((pos: any) => pos.playerId).length,
    };
  };

  const getPlayerForPosition = (positionKey: string) => {
    const position = formation.player_positions[positionKey];
    if (position?.playerId) {
      return players.find(p => p.id === position.playerId);
    }
    return null;
  };

  const stats = getFormationStats();

  return (
    <AppLayout>
      <Head title={`${formation.name} - Formation`} />
      
      <div className="space-y-6 p-6">
        {/* Team Selector */}
        <TeamSelector 
          teams={teams} 
          selectedTeamId={team.id} 
          currentPath="/team/formations" 
        />

        {/* Header */}
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-4">
            <Button
              variant="outline"
              size="sm"
              onClick={() => router.visit(route('team.formations', { team_id: team.id }))}
            >
              <ArrowLeft className="w-4 h-4 mr-2" />
              Back
            </Button>
            <div>
              <h1 className="text-3xl font-bold text-gray-900">{formation.name}</h1>
              <p className="text-gray-600">{formation.formation.display_name} • {team.name}</p>
            </div>
          </div>
          <div className="flex gap-2">
            {!formation.is_default && (
              <Button
                variant="outline"
                onClick={handleSetDefault}
              >
                <Star className="w-4 h-4 mr-2" />
                Set as Default
              </Button>
            )}
            <Button
              variant="outline"
              onClick={() => router.visit(route('team.formations.edit', formation.id))}
            >
              <Edit className="w-4 h-4 mr-2" />
              Edit
            </Button>
            <Button
              variant="outline"
              onClick={handleDeleteFormation}
              className="text-red-600 hover:text-red-700"
            >
              <Trash2 className="w-4 h-4 mr-2" />
              Delete
            </Button>
          </div>
        </div>

        {/* Formation Info */}
        <div className="grid gap-6 lg:grid-cols-3">
          <Card className="lg:col-span-2">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Target className="w-5 h-5" />
                Formation Details
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex items-center gap-4">
                <Badge className={getCategoryColor(formation.formation.category)}>
                  {formation.formation.category}
                </Badge>
                {formation.formation.country && (
                  <Badge variant="outline" className="text-xs">
                    <MapPin className="w-3 h-3 mr-1" />
                    {formation.formation.country}
                  </Badge>
                )}
                {formation.is_default && (
                  <Badge className="bg-yellow-100 text-yellow-800">
                    <Star className="w-3 h-3 mr-1" />
                    Default Formation
                  </Badge>
                )}
              </div>

              {formation.description && (
                <div>
                  <h4 className="font-medium text-gray-900 mb-2">Description</h4>
                  <p className="text-gray-600">{formation.description}</p>
                </div>
              )}

              <div>
                <h4 className="font-medium text-gray-900 mb-2">Base Formation</h4>
                <p className="text-gray-600">{formation.formation.description}</p>
              </div>

              <div className="grid grid-cols-2 gap-4 pt-4 border-t">
                <div>
                  <p className="text-sm text-gray-600">Created</p>
                  <p className="font-medium">{new Date(formation.created_at).toLocaleDateString()}</p>
                </div>
                <div>
                  <p className="text-sm text-gray-600">Status</p>
                  <p className="font-medium">{formation.is_active ? 'Active' : 'Inactive'}</p>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Users className="w-5 h-5" />
                Formation Stats
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-3">
                <div className="flex justify-between">
                  <span className="text-sm text-gray-600">Goalkeeper</span>
                  <span className="font-medium">{stats.hasGoalkeeper}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-sm text-gray-600">Defenders</span>
                  <span className="font-medium">{stats.defenders}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-sm text-gray-600">Midfielders</span>
                  <span className="font-medium">{stats.midfielders}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-sm text-gray-600">Forwards</span>
                  <span className="font-medium">{stats.forwards}</span>
                </div>
                <div className="pt-2 border-t">
                  <div className="flex justify-between">
                    <span className="text-sm text-gray-600">Players Assigned</span>
                    <span className="font-medium">{stats.assignedPlayers}/11</span>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Formation Visualization */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Target className="w-5 h-5" />
              Formation Layout
            </CardTitle>
            <CardDescription>
              Visual representation of your team formation on a football field
            </CardDescription>
          </CardHeader>
          <CardContent>
            <FormationField
              formation={formation}
              players={players}
              onPlayerClick={(player, position) => {
                console.log('Player clicked:', player, 'Position:', position);
                setShowPlayerDetails(!showPlayerDetails);
              }}
              showPlayerNames={true}
              className="w-full"
            />
          </CardContent>
        </Card>

        {/* Player Assignments */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Users className="w-5 h-5" />
              Player Assignments
            </CardTitle>
            <CardDescription>
              Current player assignments for each position
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
              {Object.entries(formation.player_positions).map(([key, position]) => {
                const player = getPlayerForPosition(key);
                return (
                  <div key={key} className="border rounded-lg p-4">
                    <div className="flex items-center justify-between mb-3">
                      <div className="flex items-center gap-2">
                        <Badge className={getPositionColor(key)}>
                          {position.label}
                        </Badge>
                      </div>
                    </div>

                    {player ? (
                      <div className="bg-green-50 rounded-lg p-3">
                        <div className="flex items-center gap-3">
                          <div className="w-10 h-10 bg-green-100 rounded-full flex items-center justify-center">
                            {player.avatar ? (
                              <img
                                src={player.avatar.startsWith('http') ? player.avatar : `/storage/${player.avatar}`}
                                alt={player.display_name || player.user.name}
                                className="w-8 h-8 rounded-full object-cover"
                              />
                            ) : (
                              <User className="w-5 h-5 text-green-600" />
                            )}
                          </div>
                          <div className="flex-1">
                            <p className="font-medium text-sm text-gray-900">
                              {player.display_name || player.user.name}
                            </p>
                            <div className="flex items-center gap-2 text-xs text-gray-600">
                              <Shirt className="w-3 h-3" />
                              #{player.jersey_number || 'N/A'}
                              <span>•</span>
                              <span>{player.football_position || 'Player'}</span>
                            </div>
                          </div>
                        </div>
                      </div>
                    ) : (
                      <div className="bg-gray-50 rounded-lg p-3 text-center">
                        <div className="w-10 h-10 bg-gray-200 rounded-full flex items-center justify-center mx-auto mb-2">
                          <User className="w-5 h-5 text-gray-400" />
                        </div>
                        <p className="text-sm text-gray-500">No player assigned</p>
                      </div>
                    )}
                  </div>
                );
              })}
            </div>
          </CardContent>
        </Card>

        {/* Quick Actions */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Settings className="w-5 h-5" />
              Quick Actions
            </CardTitle>
            <CardDescription>
              Manage this formation
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="grid gap-4 md:grid-cols-4">
              <Button
                variant="outline"
                onClick={() => router.visit(route('team.formations.edit', formation.id))}
                className="h-auto p-4 flex flex-col items-center gap-2"
              >
                <Edit className="w-6 h-6" />
                <span>Edit Formation</span>
              </Button>
              
              {!formation.is_default && (
                <Button
                  variant="outline"
                  onClick={handleSetDefault}
                  className="h-auto p-4 flex flex-col items-center gap-2"
                >
                  <Star className="w-6 h-6" />
                  <span>Set as Default</span>
                </Button>
              )}
              
              <Button
                variant="outline"
                onClick={() => router.visit(route('team.players', { team_id: team.id }))}
                className="h-auto p-4 flex flex-col items-center gap-2"
              >
                <Users className="w-6 h-6" />
                <span>Manage Players</span>
              </Button>
              
              <Button
                variant="outline"
                onClick={() => router.visit(route('team.formations', { team_id: team.id }))}
                className="h-auto p-4 flex flex-col items-center gap-2"
              >
                <Target className="w-6 h-6" />
                <span>All Formations</span>
              </Button>
            </div>
          </CardContent>
        </Card>
      </div>
    </AppLayout>
  );
}
